/*
 * Decompiled with CFR 0.152.
 */
package NC.noChance.detection.movement;

import NC.noChance.core.ACConfig;
import NC.noChance.core.CheckResult;
import NC.noChance.core.EnhancementTracker;
import NC.noChance.core.LayerFiltering;
import NC.noChance.core.MovementPredictor;
import NC.noChance.core.PlayerData;
import NC.noChance.core.ViolationType;
import NC.noChance.core.WaterHelper;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Deque;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import org.bukkit.Location;
import org.bukkit.Material;
import org.bukkit.block.Block;
import org.bukkit.entity.Player;
import org.bukkit.potion.PotionEffectType;
import org.bukkit.util.Vector;

public class FlyCheck {
    private final ACConfig config;
    private final Map<UUID, PlayerData> playerDataMap;
    private final LayerFiltering filtering;
    private final Map<UUID, FlyData> flyDataMap;
    private final Map<UUID, Vector> lastVelocity;
    private static final double GRAVITY = 0.08;
    private static final double DRAG = 0.98;
    private static final double JUMP_VELOCITY = 0.42;
    private static final double JUMP_BOOST_MULTIPLIER = 0.1;

    public FlyCheck(ACConfig config, Map<UUID, PlayerData> playerDataMap, LayerFiltering filtering) {
        this.config = config;
        this.playerDataMap = playerDataMap;
        this.filtering = filtering;
        this.flyDataMap = new HashMap<UUID, FlyData>();
        this.lastVelocity = new HashMap<UUID, Vector>();
    }

    public CheckResult check(Player player, Location from, Location to) {
        double deviation;
        if (!this.config.isCheckEnabled("fly")) {
            return CheckResult.passed();
        }
        PlayerData data = this.playerDataMap.get(player.getUniqueId());
        if (data == null) {
            return CheckResult.passed();
        }
        if (data.isInGracePeriod(this.config.getGracePeriod())) {
            return CheckResult.passed();
        }
        if (!player.hasGravity()) {
            return CheckResult.passed();
        }
        if (WaterHelper.isInWater(player)) {
            return CheckResult.passed();
        }
        EnhancementTracker.MovementEnhancements enhancements = EnhancementTracker.calculateMovementEnhancements(player);
        if (enhancements.hasLegitFlight) {
            return CheckResult.passed();
        }
        FlyData flyData = this.flyDataMap.computeIfAbsent(player.getUniqueId(), k -> new FlyData());
        UUID playerId = player.getUniqueId();
        Vector currentVelocity = to.toVector().subtract(from.toVector());
        Vector lastVel = this.lastVelocity.getOrDefault(playerId, currentVelocity);
        MovementPredictor.PredictionResult prediction = MovementPredictor.predict(player, from, to, lastVel);
        this.lastVelocity.put(playerId, currentVelocity);
        if (prediction.valid && prediction.confidence > 0.98 && !player.isFlying()) {
            return CheckResult.passed();
        }
        double deltaY = to.getY() - from.getY();
        boolean onGround = player.isOnGround() || this.isNearGround(to);
        boolean wasOnGround = data.wasOnGround();
        if (onGround) {
            data.setLastGroundTime(System.currentTimeMillis());
            data.resetAirTicks();
            flyData.reset();
            data.setWasOnGround(true);
            return CheckResult.passed();
        }
        data.incrementAirTicks();
        int airTicks = data.getAirTicks();
        double predictedVelocityY = this.calculatePredictedVelocity(player, data, flyData, wasOnGround, deltaY);
        double tolerance = this.calculateTolerance(player, data, airTicks);
        data.updateVelocityHistory(to.getX() - from.getX(), deltaY, to.getZ() - from.getZ());
        flyData.recordVelocity(deltaY);
        if (wasOnGround && deltaY > 0.0) {
            flyData.setJumpStart(System.currentTimeMillis());
            flyData.setExpectedVelocity(this.getJumpVelocity(player));
            data.setWasOnGround(false);
            return CheckResult.passed();
        }
        long timeSinceVelocity = System.currentTimeMillis() - data.getLastVelocityTime();
        if (timeSinceVelocity < 2000L) {
            double knockbackDecay = Math.max(0.0, data.getLastKnockbackVelocity() * Math.pow(0.9, (double)timeSinceVelocity / 100.0));
            tolerance += knockbackDecay;
        }
        if (Math.abs(deviation = deltaY - predictedVelocityY) > tolerance) {
            flyData.incrementViolationCount();
            if (flyData.getViolationCount() < 2) {
                return CheckResult.passed();
            }
            if (airTicks < 6) {
                return CheckResult.passed();
            }
            if (!this.isConsistentFlyPattern(flyData)) {
                flyData.resetViolationCount();
                return CheckResult.passed();
            }
            double severity = Math.min(1.0, Math.abs(deviation) / 0.5);
            EnhancementTracker.MovementEnhancements enhancementsForLog = EnhancementTracker.calculateMovementEnhancements(player);
            CheckResult prelimResult = CheckResult.failed(ViolationType.FLY, severity, String.format("Y-vel: %.3f, Predicted: %.3f, Dev: %.3f, Tol: %.3f, AirTicks: %d | Enhancements: %s", deltaY, predictedVelocityY, deviation, tolerance, airTicks, enhancementsForLog.reason));
            if (!this.filtering.passesLayer2HeuristicFiltering(player, ViolationType.FLY, prelimResult)) {
                flyData.resetViolationCount();
                return CheckResult.passed();
            }
            return prelimResult;
        }
        flyData.decrementViolationCount();
        flyData.setLastVelocity(deltaY);
        data.setWasOnGround(false);
        return CheckResult.passed();
    }

    private double calculatePredictedVelocity(Player player, PlayerData data, FlyData flyData, boolean wasOnGround, double actualVelocity) {
        Block blockAt;
        double lastVelocity = flyData.getLastVelocity();
        if (wasOnGround) {
            return 0.0;
        }
        if (player.isGliding()) {
            double expectedGlideFall = -0.15;
            long timeSinceFirework = System.currentTimeMillis() - data.getLastVelocityTime();
            if (timeSinceFirework < 1500L) {
                return actualVelocity;
            }
            if (actualVelocity > 0.3 && timeSinceFirework > 2000L) {
                return expectedGlideFall;
            }
            if (Math.abs(actualVelocity) < 0.02 && timeSinceFirework > 1500L) {
                return expectedGlideFall;
            }
            return Math.max(expectedGlideFall, Math.min(actualVelocity, 0.5));
        }
        double predicted = (lastVelocity - 0.08) * 0.98;
        if (player.hasPotionEffect(PotionEffectType.SLOW_FALLING)) {
            predicted = Math.max(predicted, -0.1);
        }
        if (player.hasPotionEffect(PotionEffectType.LEVITATION)) {
            int amplifier = player.getPotionEffect(PotionEffectType.LEVITATION).getAmplifier();
            predicted += 0.05 * (double)(amplifier + 1);
        }
        if ((blockAt = player.getLocation().getBlock()).getType() == Material.BUBBLE_COLUMN) {
            predicted = blockAt.getBlockData().getAsString().contains("drag=false") ? 0.6 : -0.3;
        }
        if (blockAt.getType() == Material.WATER || blockAt.getType().name().contains("WATER")) {
            predicted *= 0.5;
        }
        if (blockAt.getType() == Material.LAVA || blockAt.getType().name().contains("LAVA")) {
            predicted *= 0.4;
        }
        if (blockAt.getType() == Material.COBWEB) {
            predicted *= 0.15;
        }
        return predicted;
    }

    private double calculateTolerance(Player player, PlayerData data, int airTicks) {
        Block at;
        Block below;
        int ping;
        double baseTolerance = 0.18;
        if (player.isGliding()) {
            long timeSinceVelocity = System.currentTimeMillis() - data.getLastVelocityTime();
            baseTolerance = timeSinceVelocity < 1500L ? 0.45 : 0.25;
        }
        if ((ping = this.filtering.getPing(player)) > 200) {
            baseTolerance += 0.18;
        } else if (ping > 150) {
            baseTolerance += 0.12;
        } else if (ping > 100) {
            baseTolerance += 0.07;
        }
        if (airTicks < 5) {
            baseTolerance += 0.24;
        } else if (airTicks < 10) {
            baseTolerance += 0.12;
        }
        if (player.isSwimming() || player.isClimbing()) {
            baseTolerance += 0.25;
        }
        if (player.getVehicle() != null) {
            baseTolerance += 0.5;
        }
        if ((below = player.getLocation().clone().subtract(0.0, 1.0, 0.0).getBlock()).getType() == Material.SLIME_BLOCK) {
            baseTolerance += 0.8;
        }
        if (below.getType() == Material.HONEY_BLOCK) {
            baseTolerance += 0.6;
        }
        if (player.isRiptiding()) {
            baseTolerance += 1.0;
        }
        if ((at = player.getLocation().getBlock()).getType() == Material.SCAFFOLDING) {
            baseTolerance += 0.3;
        }
        if (at.getType() == Material.POWDER_SNOW) {
            baseTolerance += 0.25;
        }
        return baseTolerance;
    }

    private double getJumpVelocity(Player player) {
        Block below;
        double jumpVel = 0.42;
        if (player.hasPotionEffect(PotionEffectType.JUMP_BOOST)) {
            int amplifier = player.getPotionEffect(PotionEffectType.JUMP_BOOST).getAmplifier();
            jumpVel += 0.1 * (double)(amplifier + 1);
        }
        if ((below = player.getLocation().clone().subtract(0.0, 1.0, 0.0).getBlock()).getType() == Material.SLIME_BLOCK) {
            jumpVel *= 2.0;
        }
        if (below.getType() == Material.HONEY_BLOCK) {
            jumpVel *= 0.5;
        }
        return jumpVel;
    }

    private boolean isNearGround(Location loc) {
        for (double y = 0.0; y <= 0.5; y += 0.1) {
            Block block = loc.clone().subtract(0.0, y, 0.0).getBlock();
            Material type = block.getType();
            if (!type.isSolid() && type != Material.BARRIER && type != Material.GLASS && type != Material.ICE && type != Material.PACKED_ICE && type != Material.BLUE_ICE && !type.name().contains("GLASS") && !type.name().contains("PANE")) continue;
            return true;
        }
        return false;
    }

    private boolean isConsistentFlyPattern(FlyData flyData) {
        long hovering;
        List<Double> recentVelocities = flyData.getRecentVelocities(10);
        if (recentVelocities.size() < 5) {
            return false;
        }
        double avgVelocity = recentVelocities.stream().mapToDouble(Double::doubleValue).average().orElse(0.0);
        if (avgVelocity > -0.05 && avgVelocity < 0.05 && (hovering = recentVelocities.stream().filter(v -> Math.abs(v) < 0.05).count()) >= 4L) {
            return true;
        }
        long positiveCount = recentVelocities.stream().filter(v -> v > 0.1).count();
        return positiveCount >= 4L;
    }

    public void cleanup(UUID playerId) {
        this.flyDataMap.remove(playerId);
        this.lastVelocity.remove(playerId);
    }

    public void cleanupStale() {
        long now = System.currentTimeMillis();
        this.flyDataMap.entrySet().removeIf(entry -> {
            PlayerData data = this.playerDataMap.get(entry.getKey());
            return data == null || now - data.getLastMoveTime() > 300000L;
        });
        this.lastVelocity.entrySet().removeIf(entry -> {
            PlayerData data = this.playerDataMap.get(entry.getKey());
            return data == null || now - data.getLastMoveTime() > 300000L;
        });
    }

    private static class FlyData {
        private double lastVelocity = 0.0;
        private double expectedVelocity = 0.0;
        private long jumpStartTime = 0L;
        private int violationCount = 0;
        private final Deque<Double> velocityHistory = new ArrayDeque<Double>(20);

        public void recordVelocity(double velocity) {
            if (this.velocityHistory.size() >= 20) {
                this.velocityHistory.pollFirst();
            }
            this.velocityHistory.addLast(velocity);
        }

        public List<Double> getRecentVelocities(int count) {
            ArrayList<Double> result = new ArrayList<Double>();
            int size = this.velocityHistory.size();
            int start = Math.max(0, size - count);
            int index = 0;
            for (Double vel : this.velocityHistory) {
                if (index >= start) {
                    result.add(vel);
                }
                ++index;
            }
            return result;
        }

        public double getLastVelocity() {
            return this.lastVelocity;
        }

        public void setLastVelocity(double velocity) {
            this.lastVelocity = velocity;
        }

        public double getExpectedVelocity() {
            return this.expectedVelocity;
        }

        public void setExpectedVelocity(double velocity) {
            this.expectedVelocity = velocity;
        }

        public long getJumpStartTime() {
            return this.jumpStartTime;
        }

        public void setJumpStart(long time) {
            this.jumpStartTime = time;
        }

        public int getViolationCount() {
            return this.violationCount;
        }

        public void incrementViolationCount() {
            ++this.violationCount;
        }

        public void decrementViolationCount() {
            if (this.violationCount > 0) {
                --this.violationCount;
            }
        }

        public void resetViolationCount() {
            this.violationCount = 0;
        }

        public void reset() {
            this.lastVelocity = 0.0;
            this.expectedVelocity = 0.0;
            this.jumpStartTime = 0L;
            this.violationCount = 0;
            this.velocityHistory.clear();
        }
    }
}

