/*
 * Decompiled with CFR 0.152.
 */
package NC.noChance.detection.player;

import NC.noChance.core.ACConfig;
import NC.noChance.core.CheckResult;
import NC.noChance.core.LayerFiltering;
import NC.noChance.core.PlayerData;
import NC.noChance.core.ViolationType;
import NC.noChance.packetevents.api.event.PacketReceiveEvent;
import NC.noChance.packetevents.api.protocol.packettype.PacketType;
import NC.noChance.packetevents.api.wrapper.play.client.WrapperPlayClientPlayerPosition;
import NC.noChance.packetevents.api.wrapper.play.client.WrapperPlayClientPlayerPositionAndRotation;
import NC.noChance.packetevents.api.wrapper.play.client.WrapperPlayClientPlayerRotation;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Deque;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import org.bukkit.entity.Player;

public class BadPacketsCheck {
    private final ACConfig config;
    private final Map<UUID, PlayerData> playerDataMap;
    private final LayerFiltering filtering;
    private final Map<UUID, BadPacketData> badPacketDataMap;
    private static final double PITCH_MIN = -90.0;
    private static final double PITCH_MAX = 90.0;
    private static final double YAW_MIN = -180.0;
    private static final double YAW_MAX = 180.0;
    private static final double ENTROPY_THRESHOLD = 2.5;
    private static final double AUTOCORR_THRESHOLD = 0.85;
    private static final double CHI_SQUARED_CRITICAL = 15.507;

    public BadPacketsCheck(ACConfig config, Map<UUID, PlayerData> playerDataMap, LayerFiltering filtering) {
        this.config = config;
        this.playerDataMap = playerDataMap;
        this.filtering = filtering;
        this.badPacketDataMap = new ConcurrentHashMap<UUID, BadPacketData>();
    }

    public CheckResult processPacket(Player player, PacketReceiveEvent event) {
        if (!this.config.isCheckEnabled("badpackets")) {
            return CheckResult.passed();
        }
        PlayerData data = this.playerDataMap.get(player.getUniqueId());
        if (data == null) {
            return CheckResult.passed();
        }
        if (data.isInGracePeriod(this.config.getGracePeriod())) {
            return CheckResult.passed();
        }
        BadPacketData badData = this.badPacketDataMap.computeIfAbsent(player.getUniqueId(), k -> new BadPacketData());
        long timestamp = System.currentTimeMillis();
        CheckResult result = CheckResult.passed();
        if (event.getPacketType() == PacketType.Play.Client.PLAYER_POSITION || event.getPacketType() == PacketType.Play.Client.PLAYER_POSITION_AND_ROTATION) {
            WrapperPlayClientPlayerPosition posPacket = new WrapperPlayClientPlayerPosition(event);
            double x = posPacket.getPosition().getX();
            double y = posPacket.getPosition().getY();
            double z = posPacket.getPosition().getZ();
            if (Double.isNaN(x) || Double.isNaN(y) || Double.isNaN(z) || Double.isInfinite(x) || Double.isInfinite(y) || Double.isInfinite(z)) {
                return this.createViolation(player, "Invalid Position Values", 0.95, badData);
            }
            if (Math.abs(x) > 3.0E7 || Math.abs(z) > 3.0E7 || y < -512.0 || y > 512.0) {
                return this.createViolation(player, "Position Out of Bounds", 0.9, badData);
            }
        }
        if (event.getPacketType() == PacketType.Play.Client.PLAYER_ROTATION || event.getPacketType() == PacketType.Play.Client.PLAYER_POSITION_AND_ROTATION) {
            float pitch = 0.0f;
            float yaw = 0.0f;
            if (event.getPacketType() == PacketType.Play.Client.PLAYER_ROTATION) {
                WrapperPlayClientPlayerRotation rotPacket = new WrapperPlayClientPlayerRotation(event);
                pitch = rotPacket.getPitch();
                yaw = rotPacket.getYaw();
            } else {
                WrapperPlayClientPlayerPositionAndRotation posRotPacket = new WrapperPlayClientPlayerPositionAndRotation(event);
                pitch = posRotPacket.getPitch();
                yaw = posRotPacket.getYaw();
            }
            if (Float.isNaN(pitch) || Float.isNaN(yaw) || Float.isInfinite(pitch) || Float.isInfinite(yaw)) {
                return this.createViolation(player, "Invalid Rotation Values", 0.98, badData);
            }
            if ((double)pitch < -90.0 || (double)pitch > 90.0) {
                return this.createViolation(player, String.format("Invalid Pitch: %.2f", Float.valueOf(pitch)), 0.92, badData);
            }
            if ((double)Math.abs(yaw) > 360.0) {
                badData.recordRotation(yaw, pitch);
            }
            badData.recordRotation(yaw, pitch);
        }
        badData.recordPacketTime(timestamp, event.getPacketType().getName());
        CheckResult timerCheck = this.checkTimerAnomaly(player, badData, timestamp);
        if (timerCheck.isFailed()) {
            return timerCheck;
        }
        CheckResult sequenceCheck = this.checkPacketSequence(player, badData);
        if (sequenceCheck.isFailed()) {
            return sequenceCheck;
        }
        CheckResult entropyCheck = this.checkPacketEntropy(player, badData);
        if (entropyCheck.isFailed()) {
            return entropyCheck;
        }
        if (player.isDead() || !player.isOnline()) {
            badData.markPlayerState(PlayerState.DEAD);
            if (event.getPacketType() == PacketType.Play.Client.INTERACT_ENTITY || event.getPacketType() == PacketType.Play.Client.PLAYER_DIGGING || event.getPacketType() == PacketType.Play.Client.PLAYER_BLOCK_PLACEMENT) {
                return this.createViolation(player, "Action While Dead", 0.88, badData);
            }
        } else {
            badData.markPlayerState(PlayerState.ALIVE);
        }
        return result;
    }

    private CheckResult checkTimerAnomaly(Player player, BadPacketData badData, long currentTime) {
        List<Long> recentPackets = badData.getRecentPacketTimes(50);
        if (recentPackets.size() < 20) {
            return CheckResult.passed();
        }
        ArrayList<Long> intervals = new ArrayList<Long>();
        for (int i = 1; i < recentPackets.size(); ++i) {
            intervals.add(recentPackets.get(i) - recentPackets.get(i - 1));
        }
        double meanInterval = intervals.stream().mapToLong(Long::longValue).average().orElse(0.0);
        double variance = intervals.stream().mapToDouble(v -> Math.pow((double)v.longValue() - meanInterval, 2.0)).average().orElse(0.0);
        double stdDev = Math.sqrt(variance);
        double autocorrelation = this.calculateAutocorrelation(intervals, 1);
        long packetsInLastSecond = recentPackets.stream().filter(t -> currentTime - t < 1000L).count();
        double expectedRate = 20.0;
        double rateDeviation = Math.abs((double)packetsInLastSecond - expectedRate) / expectedRate;
        if (autocorrelation > 0.85 && rateDeviation > 0.5) {
            double severity = Math.min(1.0, (autocorrelation - 0.85) * 2.0 + rateDeviation * 0.3);
            return this.createViolation(player, String.format("Timer Detected: Rate=%.1f pps, Autocorr=%.3f", packetsInLastSecond, autocorrelation), severity, badData);
        }
        double expectedPackets = 20.0;
        double[] observed = new double[5];
        double[] expected = new double[5];
        for (int i = 0; i < 5; ++i) {
            long rangeStart = currentTime - (long)(1000 * (i + 1));
            long rangeEnd = currentTime - (long)(1000 * i);
            observed[i] = recentPackets.stream().filter(t -> t >= rangeStart && t < rangeEnd).count();
            expected[i] = expectedPackets / 5.0;
        }
        double chiSquared = 0.0;
        for (int i = 0; i < 5; ++i) {
            if (!(expected[i] > 0.0)) continue;
            chiSquared += Math.pow(observed[i] - expected[i], 2.0) / expected[i];
        }
        if (chiSquared > 15.507) {
            double severity = Math.min(1.0, (chiSquared - 15.507) / 15.507);
            return this.createViolation(player, String.format("Packet Distribution Anomaly: \u03c7\u00b2=%.2f", chiSquared), severity, badData);
        }
        return CheckResult.passed();
    }

    private CheckResult checkPacketSequence(Player player, BadPacketData badData) {
        String[][] invalidSequences;
        List<String> recentSequence = badData.getRecentPacketSequence(10);
        if (recentSequence.size() < 5) {
            return CheckResult.passed();
        }
        HashMap<String, Integer> duplicateCount = new HashMap<String, Integer>();
        String lastPacket = null;
        int consecutiveDuplicates = 0;
        for (String string : recentSequence) {
            consecutiveDuplicates = string.equals(lastPacket) ? ++consecutiveDuplicates : 1;
            duplicateCount.put(string, duplicateCount.getOrDefault(string, 0) + 1);
            if (consecutiveDuplicates > 5) {
                return this.createViolation(player, String.format("Duplicate Packets: %s x%d", string, consecutiveDuplicates), 0.75, badData);
            }
            lastPacket = string;
        }
        for (CharSequence[] charSequenceArray : invalidSequences = new String[][]{{"WrapperPlayClientInteractEntity", "WrapperPlayClientClickWindow"}, {"WrapperPlayClientPlayerFlying", "WrapperPlayClientPlayerFlying", "WrapperPlayClientPlayerFlying"}}) {
            if (recentSequence.size() < charSequenceArray.length) continue;
            boolean matches = true;
            for (int i = 0; i < charSequenceArray.length; ++i) {
                int checkIndex = recentSequence.size() - charSequenceArray.length + i;
                if (recentSequence.get(checkIndex).contains(charSequenceArray[i])) continue;
                matches = false;
                break;
            }
            if (!matches) continue;
            return this.createViolation(player, "Invalid Packet Sequence: " + String.join((CharSequence)"->", charSequenceArray), 0.82, badData);
        }
        double d = this.calculateMarkovTransitionAnomaly(recentSequence);
        if (d > 0.85) {
            return this.createViolation(player, String.format("Markov Anomaly: %.3f", d), d * 0.9, badData);
        }
        return CheckResult.passed();
    }

    private CheckResult checkPacketEntropy(Player player, BadPacketData badData) {
        double normalizedEntropy;
        List<String> recentSequence = badData.getRecentPacketSequence(30);
        if (recentSequence.size() < 15) {
            return CheckResult.passed();
        }
        HashMap<String, Integer> frequency = new HashMap<String, Integer>();
        for (String packet : recentSequence) {
            frequency.put(packet, frequency.getOrDefault(packet, 0) + 1);
        }
        double entropy = 0.0;
        int total = recentSequence.size();
        Iterator iterator = frequency.values().iterator();
        while (iterator.hasNext()) {
            int count = (Integer)iterator.next();
            double probability = (double)count / (double)total;
            entropy -= probability * (Math.log(probability) / Math.log(2.0));
        }
        if (entropy < 2.5 && recentSequence.size() > 20 && (normalizedEntropy = entropy / Math.log(frequency.size()) / Math.log(2.0)) < 0.6) {
            return this.createViolation(player, String.format("Low Packet Diversity: H=%.3f, Types=%d", entropy, frequency.size()), 0.7, badData);
        }
        return CheckResult.passed();
    }

    private double calculateAutocorrelation(List<Long> data, int lag) {
        if (data.size() < lag + 1) {
            return 0.0;
        }
        double mean = data.stream().mapToDouble(Long::doubleValue).average().orElse(0.0);
        double numerator = 0.0;
        double denominator = 0.0;
        for (int i = 0; i < data.size() - lag; ++i) {
            numerator += ((double)data.get(i).longValue() - mean) * ((double)data.get(i + lag).longValue() - mean);
        }
        for (Long value : data) {
            denominator += Math.pow((double)value.longValue() - mean, 2.0);
        }
        if (denominator == 0.0) {
            return 0.0;
        }
        return numerator / denominator;
    }

    private double calculateMarkovTransitionAnomaly(List<String> sequence) {
        if (sequence.size() < 3) {
            return 0.0;
        }
        HashMap transitions = new HashMap();
        HashMap<String, Integer> stateCounts = new HashMap<String, Integer>();
        for (int i = 0; i < sequence.size() - 1; ++i) {
            String current = sequence.get(i);
            String next = sequence.get(i + 1);
            stateCounts.put(current, stateCounts.getOrDefault(current, 0) + 1);
            transitions.putIfAbsent(current, new HashMap());
            Map nextStates = (Map)transitions.get(current);
            nextStates.put(next, nextStates.getOrDefault(next, 0) + 1);
        }
        double maxProbability = 0.0;
        for (Map.Entry entry : transitions.entrySet()) {
            String state = (String)entry.getKey();
            Map nextStates = (Map)entry.getValue();
            int totalTransitions = (Integer)stateCounts.get(state);
            Iterator iterator = nextStates.values().iterator();
            while (iterator.hasNext()) {
                int count = (Integer)iterator.next();
                double probability = (double)count / (double)totalTransitions;
                maxProbability = Math.max(maxProbability, probability);
            }
        }
        return maxProbability;
    }

    private CheckResult createViolation(Player player, String reason, double severity, BadPacketData badData) {
        badData.incrementViolation();
        if (badData.getViolationCount() < 2) {
            return CheckResult.passed();
        }
        CheckResult prelimResult = CheckResult.failed(ViolationType.BADPACKETS, severity, reason);
        if (!this.filtering.passesLayer2HeuristicFiltering(player, ViolationType.BADPACKETS, prelimResult)) {
            badData.decrementViolation();
            return CheckResult.passed();
        }
        return prelimResult;
    }

    public void cleanup(UUID playerId) {
        this.badPacketDataMap.remove(playerId);
    }

    public void cleanupStale() {
        long now = System.currentTimeMillis();
        this.badPacketDataMap.entrySet().removeIf(entry -> {
            PlayerData data = this.playerDataMap.get(entry.getKey());
            return data == null || now - data.getLastMoveTime() > 300000L;
        });
    }

    private static class BadPacketData {
        private final Deque<Long> packetTimes = new ArrayDeque<Long>(100);
        private final Deque<String> packetSequence = new ArrayDeque<String>(50);
        private final Deque<Double> yawHistory = new ArrayDeque<Double>(30);
        private final Deque<Double> pitchHistory = new ArrayDeque<Double>(30);
        private int violationCount = 0;
        private PlayerState lastState = PlayerState.ALIVE;

        public void recordPacketTime(long time, String packetType) {
            if (this.packetTimes.size() >= 100) {
                this.packetTimes.pollFirst();
            }
            this.packetTimes.addLast(time);
            if (this.packetSequence.size() >= 50) {
                this.packetSequence.pollFirst();
            }
            this.packetSequence.addLast(packetType);
        }

        public void recordRotation(double yaw, double pitch) {
            if (this.yawHistory.size() >= 30) {
                this.yawHistory.pollFirst();
                this.pitchHistory.pollFirst();
            }
            this.yawHistory.addLast(yaw);
            this.pitchHistory.addLast(pitch);
        }

        public List<Long> getRecentPacketTimes(int count) {
            ArrayList<Long> result = new ArrayList<Long>();
            int size = this.packetTimes.size();
            int start = Math.max(0, size - count);
            int index = 0;
            for (Long time : this.packetTimes) {
                if (index >= start) {
                    result.add(time);
                }
                ++index;
            }
            return result;
        }

        public List<String> getRecentPacketSequence(int count) {
            ArrayList<String> result = new ArrayList<String>();
            int size = this.packetSequence.size();
            int start = Math.max(0, size - count);
            int index = 0;
            for (String packet : this.packetSequence) {
                if (index >= start) {
                    result.add(packet);
                }
                ++index;
            }
            return result;
        }

        public void markPlayerState(PlayerState state) {
            this.lastState = state;
        }

        public int getViolationCount() {
            return this.violationCount;
        }

        public void incrementViolation() {
            ++this.violationCount;
        }

        public void decrementViolation() {
            if (this.violationCount > 0) {
                --this.violationCount;
            }
        }

        public void reset() {
            this.packetTimes.clear();
            this.packetSequence.clear();
            this.yawHistory.clear();
            this.pitchHistory.clear();
            this.violationCount = 0;
        }
    }

    private static enum PlayerState {
        ALIVE,
        DEAD,
        SPECTATOR;

    }
}

