/*
 * Decompiled with CFR 0.152.
 */
package NC.noChance.detection.player;

import NC.noChance.core.ACConfig;
import NC.noChance.core.BoundedDeque;
import NC.noChance.core.CheckResult;
import NC.noChance.core.FalsePositiveFilter;
import NC.noChance.core.LayerFiltering;
import NC.noChance.core.PlayerData;
import NC.noChance.core.ViolationType;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import org.bukkit.Bukkit;
import org.bukkit.Location;
import org.bukkit.entity.Player;

public class BlinkCheck {
    private final ACConfig config;
    private final Map<UUID, PlayerData> playerDataMap;
    private final LayerFiltering filtering;
    private final Map<UUID, BlinkTracker> blinkTrackers;
    private final FalsePositiveFilter falsePositiveFilter;
    private static final long MAX_PACKET_DELAY = 1200L;
    private static final int MAX_HELD_PACKETS = 75;
    private static final long PACKET_WINDOW = 1000L;
    private static final double PACKET_BURST_THRESHOLD = 40.0;
    private static final long CHUNK_LOAD_DELAY_THRESHOLD = 2000L;
    private static final int MIN_PACKETS_FOR_ANALYSIS = 40;

    public BlinkCheck(ACConfig config, Map<UUID, PlayerData> playerDataMap, LayerFiltering filtering) {
        this.config = config;
        this.playerDataMap = playerDataMap;
        this.filtering = filtering;
        this.blinkTrackers = new ConcurrentHashMap<UUID, BlinkTracker>();
        this.falsePositiveFilter = new FalsePositiveFilter();
    }

    public void onPacketReceived(Player player, String packetType, long timestamp) {
        if (!this.config.isCheckEnabled("blink")) {
            return;
        }
        UUID playerId = player.getUniqueId();
        PlayerData data = this.playerDataMap.get(playerId);
        if (data == null) {
            return;
        }
        if (data.isInGracePeriod(this.config.getGracePeriod())) {
            return;
        }
        BlinkTracker tracker = this.blinkTrackers.computeIfAbsent(playerId, k -> new BlinkTracker());
        tracker.addPacket(timestamp, packetType);
    }

    public CheckResult checkForBlink(Player player) {
        if (!this.config.isCheckEnabled("blink")) {
            return CheckResult.passed();
        }
        UUID playerId = player.getUniqueId();
        PlayerData data = this.playerDataMap.get(playerId);
        if (data == null) {
            return CheckResult.passed();
        }
        if (data.isInGracePeriod(this.config.getGracePeriod())) {
            return CheckResult.passed();
        }
        if (player.isFlying() || player.getAllowFlight()) {
            return CheckResult.passed();
        }
        BlinkTracker tracker = this.blinkTrackers.get(playerId);
        if (tracker == null) {
            return CheckResult.passed();
        }
        long now = System.currentTimeMillis();
        tracker.cleanOldPackets(now - 5000L);
        if (tracker.getPacketHistory().size() < 40) {
            return CheckResult.passed();
        }
        double tps = this.getCurrentTPS();
        if (tps < 18.0) {
            return CheckResult.passed();
        }
        if (this.isPlayerLoadingChunks(player, tracker)) {
            tracker.resetViolations();
            return CheckResult.passed();
        }
        PacketBurstAnalysis analysis = this.analyzePacketBursts(tracker, now);
        if (analysis.hasSuspiciousBurst) {
            double trustScore = this.falsePositiveFilter.getPlayerTrustScore(playerId);
            int ping = this.filtering.getPing(player);
            double maxDelay = 1200.0;
            if (ping > 100) {
                maxDelay += (double)ping * 1.2;
            }
            if (ping > 200) {
                maxDelay += (double)ping * 0.8;
            }
            if (trustScore > 0.75) {
                maxDelay *= 1.35;
            } else if (trustScore < 0.4) {
                maxDelay *= 0.85;
            }
            if (tps < 19.5) {
                maxDelay *= 1.15;
            }
            if ((double)analysis.maxDelay > maxDelay) {
                double severity = Math.min(0.85, 0.45 + ((double)analysis.maxDelay - maxDelay) / (maxDelay * 2.0));
                CheckResult prelimResult = CheckResult.failed(ViolationType.BLINK, severity, String.format("Packet delay: %dms (max: %.0fms), burst: %.1f packets/150ms, ping: %dms", analysis.maxDelay, maxDelay, analysis.burstRate, ping));
                if (!this.filtering.passesLayer2HeuristicFiltering(player, ViolationType.BLINK, prelimResult)) {
                    tracker.resetViolations();
                    this.falsePositiveFilter.recordLegitAction(player, ViolationType.BLINK);
                    return CheckResult.passed();
                }
                tracker.recordViolation();
                if (tracker.getViolations() < 4) {
                    return CheckResult.passed();
                }
                return prelimResult;
            }
        }
        if (analysis.hasPacketHolding) {
            long holdingDuration = analysis.holdingEndTime - analysis.holdingStartTime;
            if (player.getVelocity().length() > 0.5 || player.isFlying() || player.isGliding()) {
                return CheckResult.passed();
            }
            if (holdingDuration > 2000L && analysis.heldPacketCount > 30) {
                double severity = Math.min(0.8, 0.5 + (double)holdingDuration / 8000.0);
                CheckResult prelimResult = CheckResult.failed(ViolationType.BLINK, severity, String.format("Packet holding detected: %dms gap, %d packets released", holdingDuration, analysis.heldPacketCount));
                if (!this.filtering.passesLayer2HeuristicFiltering(player, ViolationType.BLINK, prelimResult)) {
                    tracker.resetViolations();
                    return CheckResult.passed();
                }
                tracker.recordViolation();
                if (tracker.getViolations() < 6) {
                    return CheckResult.passed();
                }
                return prelimResult;
            }
        }
        if (tracker.getConsistentDelayCount() > 50) {
            double avgDelay = tracker.getAverageDelay();
            double variance = tracker.getDelayVariance();
            if (variance < 1.0 && (avgDelay < 1.0 || avgDelay > 60.0)) {
                double severity = 0.7;
                CheckResult prelimResult = CheckResult.failed(ViolationType.BLINK, severity, String.format("Artificial delay pattern: %.1fms avg, %.2fms variance", avgDelay, variance));
                if (!this.filtering.passesLayer2HeuristicFiltering(player, ViolationType.BLINK, prelimResult)) {
                    tracker.resetViolations();
                    return CheckResult.passed();
                }
                tracker.recordViolation();
                if (tracker.getViolations() < 8) {
                    return CheckResult.passed();
                }
                return prelimResult;
            }
        }
        this.falsePositiveFilter.recordLegitAction(player, ViolationType.BLINK);
        return CheckResult.passed();
    }

    private PacketBurstAnalysis analyzePacketBursts(BlinkTracker tracker, long now) {
        PacketBurstAnalysis analysis = new PacketBurstAnalysis();
        BoundedDeque<PacketRecord> history = tracker.getPacketHistory();
        if (history.size() < 2) {
            return analysis;
        }
        List<PacketRecord> packetList = history.toList();
        if (packetList.size() < 2) {
            return analysis;
        }
        PacketRecord prev = null;
        long maxGap = 0L;
        long gapStart = 0L;
        for (PacketRecord packet : packetList) {
            if (prev != null) {
                long gap = packet.timestamp - prev.timestamp;
                if (gap > maxGap) {
                    maxGap = gap;
                    gapStart = prev.timestamp;
                }
                analysis.maxDelay = Math.max(analysis.maxDelay, gap);
            }
            prev = packet;
        }
        for (int i = 0; i < packetList.size(); ++i) {
            PacketRecord current = packetList.get(i);
            int packetsInWindow = 1;
            long windowStart = current.timestamp;
            for (int j = i + 1; j < packetList.size(); ++j) {
                PacketRecord next = packetList.get(j);
                if (next.timestamp - windowStart > 1000L) break;
                ++packetsInWindow;
            }
            if (packetsInWindow <= analysis.burstPacketCount) continue;
            analysis.burstPacketCount = packetsInWindow;
            analysis.burstStartTime = windowStart;
            analysis.burstRate = packetsInWindow;
        }
        if (analysis.burstRate > 40.0) {
            analysis.hasSuspiciousBurst = true;
        }
        if (maxGap > 1200L) {
            int packetsAfterGap = 0;
            for (PacketRecord packet : packetList) {
                if (packet.timestamp <= gapStart + maxGap || packet.timestamp >= gapStart + maxGap + 500L) continue;
                ++packetsAfterGap;
            }
            if (packetsAfterGap > 35) {
                analysis.hasPacketHolding = true;
                analysis.holdingStartTime = gapStart;
                analysis.holdingEndTime = gapStart + maxGap;
                analysis.heldPacketCount = packetsAfterGap;
            }
        }
        return analysis;
    }

    private double getCurrentTPS() {
        try {
            Object server = Bukkit.getServer().getClass().getMethod("getServer", new Class[0]).invoke((Object)Bukkit.getServer(), new Object[0]);
            double[] recentTps = (double[])server.getClass().getField("recentTps").get(server);
            return Math.min(20.0, recentTps[0]);
        }
        catch (Exception e) {
            return 20.0;
        }
    }

    private boolean isPlayerLoadingChunks(Player player, BlinkTracker tracker) {
        Location loc = player.getLocation();
        if (loc.getWorld() == null) {
            return false;
        }
        int chunkX = loc.getBlockX() >> 4;
        int chunkZ = loc.getBlockZ() >> 4;
        boolean hasUnloadedChunks = false;
        for (int x = -2; x <= 2; ++x) {
            for (int z = -2; z <= 2; ++z) {
                if (loc.getWorld().isChunkLoaded(chunkX + x, chunkZ + z)) continue;
                hasUnloadedChunks = true;
                break;
            }
            if (hasUnloadedChunks) break;
        }
        if (hasUnloadedChunks) {
            tracker.setChunkLoadingTime(System.currentTimeMillis());
            return true;
        }
        return System.currentTimeMillis() - tracker.getChunkLoadingTime() < 3000L;
    }

    public void cleanup(UUID playerId) {
        this.blinkTrackers.remove(playerId);
        this.falsePositiveFilter.cleanup(playerId);
    }

    public void cleanupStale() {
        long now = System.currentTimeMillis();
        this.blinkTrackers.entrySet().removeIf(entry -> {
            BlinkTracker tracker = (BlinkTracker)entry.getValue();
            return tracker.getLastPacketTime() < now - 60000L;
        });
    }

    private static class BlinkTracker {
        private final BoundedDeque<PacketRecord> packetHistory = new BoundedDeque(200);
        private int violations = 0;
        private long lastViolation = 0L;
        private double averageDelay = 0.0;
        private double delayVariance = 0.0;
        private int consistentDelayCount = 0;
        private long lastPacketTime = 0L;
        private long chunkLoadingTime = 0L;

        public void addPacket(long timestamp, String type) {
            this.packetHistory.add(new PacketRecord(timestamp, type));
            this.updateAverageDelay();
            this.lastPacketTime = timestamp;
        }

        private void updateAverageDelay() {
            if (this.packetHistory.size() < 10) {
                return;
            }
            List<PacketRecord> packets = this.packetHistory.toList();
            if (packets.size() < 10) {
                return;
            }
            double sum = 0.0;
            int count = 0;
            ArrayList<Long> delays = new ArrayList<Long>();
            PacketRecord prev = null;
            for (PacketRecord packet : packets) {
                if (prev != null) {
                    long delay = packet.timestamp - prev.timestamp;
                    sum += (double)delay;
                    delays.add(delay);
                    ++count;
                }
                prev = packet;
            }
            if (count > 0) {
                double newAvg = sum / (double)count;
                double varianceSum = 0.0;
                Iterator iterator = delays.iterator();
                while (iterator.hasNext()) {
                    long delay = (Long)iterator.next();
                    varianceSum += Math.pow((double)delay - newAvg, 2.0);
                }
                this.delayVariance = Math.sqrt(varianceSum / (double)count);
                this.consistentDelayCount = Math.abs(newAvg - this.averageDelay) < 5.0 && this.delayVariance < 3.0 ? ++this.consistentDelayCount : 0;
                this.averageDelay = newAvg;
            }
        }

        public void cleanOldPackets(long cutoff) {
            while (!this.packetHistory.isEmpty() && this.packetHistory.peek().timestamp < cutoff) {
                this.packetHistory.poll();
            }
        }

        public void recordViolation() {
            long now = System.currentTimeMillis();
            this.violations = now - this.lastViolation < 5000L ? ++this.violations : 1;
            this.lastViolation = now;
        }

        public void resetViolations() {
            if (System.currentTimeMillis() - this.lastViolation > 10000L) {
                this.violations = Math.max(0, this.violations - 1);
            }
        }

        public BoundedDeque<PacketRecord> getPacketHistory() {
            return this.packetHistory;
        }

        public int getViolations() {
            return this.violations;
        }

        public double getAverageDelay() {
            return this.averageDelay;
        }

        public double getDelayVariance() {
            return this.delayVariance;
        }

        public int getConsistentDelayCount() {
            return this.consistentDelayCount;
        }

        public long getLastPacketTime() {
            return this.lastPacketTime;
        }

        public void setChunkLoadingTime(long time) {
            this.chunkLoadingTime = time;
        }

        public long getChunkLoadingTime() {
            return this.chunkLoadingTime;
        }
    }

    private static class PacketBurstAnalysis {
        boolean hasSuspiciousBurst = false;
        boolean hasPacketHolding = false;
        double burstRate = 0.0;
        int burstPacketCount = 0;
        long burstStartTime = 0L;
        long maxDelay = 0L;
        long holdingStartTime = 0L;
        long holdingEndTime = 0L;
        int heldPacketCount = 0;

        private PacketBurstAnalysis() {
        }
    }

    private static class PacketRecord {
        final long timestamp;
        final String type;

        PacketRecord(long timestamp, String type) {
            this.timestamp = timestamp;
            this.type = type;
        }
    }
}

