/*
 * Decompiled with CFR 0.152.
 */
package NC.noChance.replay;

import NC.noChance.core.ViolationType;
import NC.noChance.replay.ReplayBuffer;
import NC.noChance.replay.ReplayData;
import NC.noChance.replay.ReplayIO;
import NC.noChance.replay.ReplayTask;
import NC.noChance.replay.Snapshot;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import org.bukkit.Bukkit;
import org.bukkit.entity.Player;
import org.bukkit.plugin.Plugin;
import org.bukkit.scheduler.BukkitTask;

public class ReplayMgr {
    private final Plugin plugin;
    private final ReplayIO io;
    private final Map<UUID, ReplayBuffer> buffers;
    private final Map<UUID, ReplayTask> viewers;
    private final Map<UUID, PendingReplay> pending;
    private final int bufferSize;
    private final long retentionMs;
    private final boolean enabled;
    private BukkitTask captureTask;
    private BukkitTask cleanupTask;

    public ReplayMgr(Plugin plugin, int bufferSeconds, int retentionDays, boolean enabled) {
        this.plugin = plugin;
        this.io = new ReplayIO(plugin.getDataFolder().toPath());
        this.buffers = new ConcurrentHashMap<UUID, ReplayBuffer>();
        this.viewers = new ConcurrentHashMap<UUID, ReplayTask>();
        this.pending = new ConcurrentHashMap<UUID, PendingReplay>();
        this.bufferSize = bufferSeconds * 20;
        this.retentionMs = (long)retentionDays * 24L * 60L * 60L * 1000L;
        this.enabled = enabled;
        if (enabled) {
            this.startCapture();
            this.startCleanup();
        }
    }

    private void startCapture() {
        this.captureTask = Bukkit.getScheduler().runTaskTimer(this.plugin, () -> {
            for (Player player : Bukkit.getOnlinePlayers()) {
                ReplayBuffer buffer = this.buffers.computeIfAbsent(player.getUniqueId(), k -> new ReplayBuffer(this.bufferSize));
                buffer.capture(player);
                PendingReplay pr = this.pending.get(player.getUniqueId());
                if (pr == null || !pr.isRecording()) continue;
                pr.addSnapshot(buffer.copy().get(buffer.size() - 1));
            }
        }, 1L, 1L);
    }

    private void startCleanup() {
        this.cleanupTask = Bukkit.getScheduler().runTaskTimerAsynchronously(this.plugin, () -> this.io.cleanup(this.retentionMs), 72000L, 72000L);
    }

    public void onPlayerJoin(Player player) {
        if (!this.enabled) {
            return;
        }
        this.buffers.put(player.getUniqueId(), new ReplayBuffer(this.bufferSize));
    }

    public void onPlayerQuit(Player player) {
        ReplayTask task;
        UUID pid = player.getUniqueId();
        this.buffers.remove(pid);
        PendingReplay pr = this.pending.remove(pid);
        if (pr != null) {
            pr.stopRecording();
            List<Snapshot> all = pr.getSnapshots();
            if (all.size() >= 20) {
                Bukkit.getScheduler().runTaskAsynchronously(this.plugin, () -> {
                    ReplayData data = new ReplayData(pid, pr.playerName, pr.world, pr.type, pr.confidence, System.currentTimeMillis(), all);
                    try {
                        this.io.save(data);
                    }
                    catch (IOException iOException) {
                        // empty catch block
                    }
                });
            }
        }
        if ((task = this.viewers.remove(pid)) != null) {
            task.stop();
        }
    }

    public void recordAction(Player player, Snapshot.Action action) {
        if (!this.enabled) {
            return;
        }
        ReplayBuffer buffer = this.buffers.get(player.getUniqueId());
        if (buffer != null) {
            buffer.setAction(action);
        }
    }

    public void recordAction(Player player, Snapshot.Action action, int targetId) {
        if (!this.enabled) {
            return;
        }
        ReplayBuffer buffer = this.buffers.get(player.getUniqueId());
        if (buffer != null) {
            buffer.setAction(action, targetId);
        }
    }

    public void saveReplay(Player player, ViolationType type, String confidence) {
        if (!this.enabled) {
            return;
        }
        UUID pid = player.getUniqueId();
        if (this.pending.containsKey(pid)) {
            return;
        }
        ReplayBuffer buffer = this.buffers.get(pid);
        if (buffer == null || buffer.size() < 20) {
            return;
        }
        List<Snapshot> before = buffer.copy();
        PendingReplay pr = new PendingReplay(player.getName(), player.getWorld().getName(), type, confidence, before);
        this.pending.put(pid, pr);
        Bukkit.getScheduler().runTaskLater(this.plugin, () -> {
            PendingReplay finished = this.pending.remove(pid);
            if (finished == null) {
                return;
            }
            finished.stopRecording();
            List<Snapshot> all = finished.getSnapshots();
            Bukkit.getScheduler().runTaskAsynchronously(this.plugin, () -> {
                ReplayData data = new ReplayData(pid, finished.playerName, finished.world, finished.type, finished.confidence, System.currentTimeMillis(), all);
                try {
                    this.io.save(data);
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            });
        }, 200L);
    }

    public void playReplay(Player viewer, String filename) {
        ReplayTask existing = this.viewers.get(viewer.getUniqueId());
        if (existing != null && !existing.isFinished()) {
            viewer.sendMessage("\u00a7c\u00a7lNoChance \u00a78\u00bb \u00a77Already watching. Use \u00a7f/nc replay stop \u00a77first.");
            return;
        }
        Bukkit.getScheduler().runTaskAsynchronously(this.plugin, () -> {
            try {
                ReplayData data = this.io.load(filename);
                Bukkit.getScheduler().runTask(this.plugin, () -> {
                    ReplayTask task = new ReplayTask(this.plugin, viewer, data);
                    this.viewers.put(viewer.getUniqueId(), task);
                    task.start();
                });
            }
            catch (IOException e) {
                Bukkit.getScheduler().runTask(this.plugin, () -> viewer.sendMessage("\u00a7c\u00a7lNoChance \u00a78\u00bb \u00a77Failed to load: " + e.getMessage()));
            }
        });
    }

    public void stopReplay(Player viewer) {
        ReplayTask task = this.viewers.remove(viewer.getUniqueId());
        if (task != null) {
            task.stop();
        } else {
            viewer.sendMessage("\u00a7c\u00a7lNoChance \u00a78\u00bb \u00a77Not watching a replay.");
        }
    }

    public void pauseReplay(Player viewer) {
        ReplayTask task = this.viewers.get(viewer.getUniqueId());
        if (task != null && !task.isFinished()) {
            task.pause();
        } else {
            viewer.sendMessage("\u00a7c\u00a7lNoChance \u00a78\u00bb \u00a77Not watching a replay.");
        }
    }

    public void setReplaySpeed(Player viewer, float speed) {
        ReplayTask task = this.viewers.get(viewer.getUniqueId());
        if (task != null && !task.isFinished()) {
            task.setSpeed(speed);
        } else {
            viewer.sendMessage("\u00a7c\u00a7lNoChance \u00a78\u00bb \u00a77Not watching a replay.");
        }
    }

    public void toggleFollow(Player viewer) {
        ReplayTask task = this.viewers.get(viewer.getUniqueId());
        if (task != null && !task.isFinished()) {
            task.toggleFollow();
        } else {
            viewer.sendMessage("\u00a7c\u00a7lNoChance \u00a78\u00bb \u00a77Not watching a replay.");
        }
    }

    public List<String> listReplays(String playerName) {
        return this.io.listReplays(playerName);
    }

    public void shutdown() {
        if (this.captureTask != null) {
            this.captureTask.cancel();
        }
        if (this.cleanupTask != null) {
            this.cleanupTask.cancel();
        }
        for (ReplayTask task : this.viewers.values()) {
            task.stop();
        }
        this.buffers.clear();
        this.viewers.clear();
        this.pending.clear();
    }

    public boolean isEnabled() {
        return this.enabled;
    }

    private static class PendingReplay {
        final String playerName;
        final String world;
        final ViolationType type;
        final String confidence;
        final List<Snapshot> snapshots;
        private volatile boolean recording = true;

        PendingReplay(String playerName, String world, ViolationType type, String confidence, List<Snapshot> before) {
            this.playerName = playerName;
            this.world = world;
            this.type = type;
            this.confidence = confidence;
            this.snapshots = new ArrayList<Snapshot>(before);
        }

        synchronized void addSnapshot(Snapshot s) {
            if (this.recording && s != null) {
                this.snapshots.add(s);
            }
        }

        void stopRecording() {
            this.recording = false;
        }

        boolean isRecording() {
            return this.recording;
        }

        synchronized List<Snapshot> getSnapshots() {
            return new ArrayList<Snapshot>(this.snapshots);
        }
    }
}

