package _3650.builders_inventory.feature.extended_inventory;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.stream.Collectors;

import org.apache.commons.lang3.tuple.Pair;

import _3650.builders_inventory.BuildersInventory;
import _3650.builders_inventory.config.Config;
import _3650.builders_inventory.datafixer.ModDataFixer;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_124;
import net.minecraft.class_156;
import net.minecraft.class_1799;
import net.minecraft.class_2487;
import net.minecraft.class_2499;
import net.minecraft.class_2507;
import net.minecraft.class_2509;
import net.minecraft.class_2512;
import net.minecraft.class_2520;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_6903;

public class ExtendedInventoryPages {
	
	private static final String FILE_PREFIX = "inventory-";
	private static final String FILE_SUFFIX = "-page.nbt";
	
	private static final ArrayBlockingQueue<class_2561> PLAYER_MESSAGE_QUEUE = new ArrayBlockingQueue<>(50);
	private static final ArrayList<ExtendedInventoryPage> PAGES = new ArrayList<>();
	private static int deleted = 0;
	private static boolean loaded = false;
	private static boolean valid = false;
	private static int timeToSave = 0;
	private static boolean hasChanged = false;
	private static boolean forceUpdate = false;
	
	static void tick(class_310 mc) {
		if (hasChanged && --timeToSave <= 0) save();
		
		if (!PLAYER_MESSAGE_QUEUE.isEmpty() && mc.field_1724 != null) {
			ArrayList<class_2561> messages = new ArrayList<>();
			PLAYER_MESSAGE_QUEUE.drainTo(messages);
			for (var msg : messages) mc.field_1724.method_43496(msg);
		}
	}
	
	public static void setChanged() {
		if (!loaded) load();
		if (!valid) return;
		
		hasChanged = true;
		timeToSave = Config.instance().extended_inventory_save_delay * 20;
	}
	
	static void forceUpdate() {
		forceUpdate = true;
	}
	
	public static void load() {
		class_310 mc = class_310.method_1551();
		if (mc.field_1687 != null) load(mc.field_1687.method_30349().method_57093(class_2509.field_11560));
		else if (mc.method_1562() != null) load(mc.method_1562().method_29091().method_57093(class_2509.field_11560));
		else {
			BuildersInventory.LOGGER.error("Error loading extended inventory saved data: Not in-game!");
			PLAYER_MESSAGE_QUEUE.add(class_2561.method_43471("error.builders_inventory.extended_inventory.load_failed"));
		}
	}
	
	public static void load(class_6903<class_2520> registryOps) {
		BuildersInventory.LOGGER.info("Loading Extended Inventory...");
		BuildersInventory.LOGGER.info("Hey Log Readers: LOGS ARE ZERO-INDEXED");
		
		if (loaded && valid && hasChanged) {
			BuildersInventory.LOGGER.info("Must save extended inventory before reloading");
			if (save(registryOps)) {
				BuildersInventory.LOGGER.info("Saved");
			} else {
				BuildersInventory.LOGGER.error("Error loading extended inventory saved data: Could not save first!");
				PLAYER_MESSAGE_QUEUE.add(class_2561.method_43471("error.builders_inventory.extended_inventory.load_failed"));
				return;
			}
		}
		
		loaded = true;
		valid = false;
		
		PAGES.clear();
		ExtendedInventory.PAGE_CONTAINER.reset();
		
		Path root = FabricLoader.getInstance().getGameDir().resolve(BuildersInventory.MOD_ID).resolve("extended_inventory");
		try {
			if (!Files.isDirectory(root)) {
				Path oldRoot = FabricLoader.getInstance().getConfigDir().resolve(BuildersInventory.MOD_ID);
				if (Files.isDirectory(oldRoot)) {
					root = oldRoot;
					forceUpdate = true;
				}
			}
			
			// Get possible filenames
			String[] filenames = root.toFile().list((dir, name) -> name.startsWith(FILE_PREFIX) && name.endsWith(FILE_SUFFIX));
			if (filenames == null) filenames = new String[0];
			
			// order and number validity NOT guaranteed, construct array of files and save max id to fill in blanks
			int substrStart = FILE_PREFIX.length();
			int substrEnd = FILE_SUFFIX.length();
			int max = -1;
			var pageMap = new Int2ObjectOpenHashMap<ExtendedInventoryPage>(filenames.length);
			for (var name : filenames) {
				String idStr = name.substring(substrStart, name.length() - substrEnd);
				try {
					int id = Integer.parseInt(idStr) - 1;
					
					var optPage = loadPage(registryOps, root.resolve(name), id);
					if (optPage.isPresent()) {
						if (id > max) max = id;
						pageMap.put(id, optPage.get());
					}
					
				} catch (NumberFormatException e) {
					BuildersInventory.LOGGER.error("Error loading extended inventory page " + idStr + ": Invalid Integer!", e);
					continue;
				}
			}
			
			// Load items in, filling in the blanks
			for (int i = 0; i < max + 1; i++) {
				if (pageMap.containsKey(i)) PAGES.add(pageMap.get(i));
				else {
					BuildersInventory.LOGGER.warn("Could not find extended inventory page {}", i);
					PAGES.add(ExtendedInventoryPage.INVALID);
				}
			}
			BuildersInventory.LOGGER.info("Loaded {} extended inventory pages...", PAGES.size());
			
			// Set valid, it loaded at least and everything after this fails if invalid
			valid = true;
			
			// If no pages loaded, make a blank page
			if (PAGES.isEmpty()) create();
			
			// Load Saved Data
			if (Files.isRegularFile(root.resolve("inventory-data.nbt"))) {
				class_2487 tag = class_2507.method_10633(root.resolve("inventory-data.nbt"));
				if (tag == null) {
					BuildersInventory.LOGGER.error("Error loading extended inventory saved data: Invalid Data!");
				}
				
				if (tag.method_10573("page", class_2520.field_33253)) {
					int savedPage = tag.method_10550("page");
					if (savedPage >= 0 && savedPage < PAGES.size()) {
						ExtendedInventory.PAGE_CONTAINER.setPage(savedPage);
						BuildersInventory.LOGGER.info("Loaded selected page as {}...", savedPage);
					} else if (savedPage >= 0 && PAGES.size() > 0) {
						int newPage = Math.max(0, PAGES.size() - 1);
						ExtendedInventory.PAGE_CONTAINER.setPage(newPage);
						BuildersInventory.LOGGER.warn("Loaded page out of bounds {}, switched to page {}", savedPage, newPage);
					} else {
						BuildersInventory.LOGGER.error("Failed to load invalid selected page {}...", savedPage);
					}
				}
				
			}
			
			// Select page 0 if none selected
			if (ExtendedInventory.getPage() < 0) {
				BuildersInventory.LOGGER.info("No page selected! Selecting first page...");
				ExtendedInventory.PAGE_CONTAINER.setPage(0);
			}
			
			// Save if forced update (after datafixer run or folder switch)
			if (forceUpdate) {
				BuildersInventory.LOGGER.info("Pages have been migrated from an older version, saving...");
				hasChanged = true;
				for (var page : PAGES) page.discreteChange();
				deleted = 0;
				if (!save(registryOps)) {
					BuildersInventory.LOGGER.error("Could not save migrated pages!");
				}
				BuildersInventory.LOGGER.info("Saved migrated data!");
				forceUpdate = false;
			}
			
		} catch (Exception e) {
			BuildersInventory.LOGGER.error("Error loading extended inventory pages!", e);
			PLAYER_MESSAGE_QUEUE.add(class_2561.method_43471("error.builders_inventory.extended_inventory.load_failed").method_27692(class_124.field_1061));
		}
	}
	
	public static Optional<ExtendedInventoryPage> loadPage(class_6903<class_2520> registryOps, Path path, int id) throws Exception {
		
		class_2487 tag = class_2507.method_10633(path);
		
		var datafix = ModDataFixer.extendedInventoryPage(tag, 1);
		if (datafix.isPresent()) tag = datafix.get();
		
		if (tag == null || !tag.method_10573("items", class_2520.field_33259)) {
			BuildersInventory.LOGGER.error("Error loading extended inventory page {} tag {}: Invalid Data!", id, tag);
			return Optional.empty();
		}
		
		class_2499 itemTags = tag.method_10554("items", class_2520.field_33260);
		if (itemTags == null || itemTags.isEmpty()) {
			BuildersInventory.LOGGER.error("Error loading extended inventory page {} items {}: Invalid Data!", id, itemTags);
			return Optional.empty();
		}
		
		List<class_1799> items = itemTags.stream()
				.map(itemTag -> class_1799.field_49266
						.parse(registryOps, itemTag)
						.resultOrPartial(err -> BuildersInventory.LOGGER.error("Could not parse extended inventory item: '{}'", err))
						.orElse(class_1799.field_8037))
				.collect(Collectors.toList());
		
		boolean locked = false;
		if (tag.method_10573("locked", class_2520.field_33251)) {
			locked = tag.method_10577("locked");
		}
		
		String name = "";
		if (tag.method_10573("name", class_2520.field_33258)) {
			name = tag.method_10558("name");
		}
		
		class_1799 icon = class_1799.field_8037;
		if (tag.method_10573("icon", class_2520.field_33260)) {
			class_2487 iconTag = tag.method_10562("icon");
			icon = class_1799.field_49266
					.parse(registryOps, iconTag)
					.resultOrPartial(err -> BuildersInventory.LOGGER.error("Could not parse extended inventory icon {}: '{}'", iconTag, err))
					.orElse(class_1799.field_8037);
		}
		
		class_1799 originalIcon = class_1799.field_8037;
		if (tag.method_10573("original_icon", class_2520.field_33260)) {
			class_2487 iconTag = tag.method_10562("original_icon");
			icon = class_1799.field_49266
					.parse(registryOps, iconTag)
					.resultOrPartial(err -> BuildersInventory.LOGGER.error("Could not parse extended inventory original icon {}: '{}'", iconTag, err))
					.orElse(class_1799.field_8037);
		}
		
		boolean iconDataActive = false;
		if (tag.method_10573("icon_data", class_2520.field_33251)) {
			iconDataActive = tag.method_10577("icon_data");
		}
		
		int iconScaleDown = 0;
		if (tag.method_10573("icon_scale_down", class_2520.field_33253)) {
			iconScaleDown = tag.method_10550("icon_scale_down");
		}
		
		var page = ExtendedInventoryPage.of(items, locked, name, icon, originalIcon, iconDataActive, iconScaleDown);
		if (datafix.isPresent()) {
			forceUpdate = true;
			page.discreteChange();
		}
		return Optional.of(page);
	}
	
	public static boolean save() {
		class_310 mc = class_310.method_1551();
		if (mc.field_1687 != null) return save(mc.field_1687.method_30349().method_57093(class_2509.field_11560));
		else if (mc.method_1562() != null) return save(mc.method_1562().method_29091().method_57093(class_2509.field_11560));
		else {
			BuildersInventory.LOGGER.error("Error saving extended inventory saved data: Not in-game!");
			PLAYER_MESSAGE_QUEUE.add(class_2561.method_43471("error.builders_inventory.extended_inventory.save_failed"));
			return false;
		}
	}
	
	public static boolean save(class_6903<class_2520> registryOps) {
		BuildersInventory.LOGGER.info("Saving Extended Inventory...");
		if (!hasChanged) {
			BuildersInventory.LOGGER.info("Nothing to save.");
			return false;
		}
		BuildersInventory.LOGGER.info("Hey Log Readers: LOGS ARE ZERO-INDEXED");
		
		timeToSave = 0;
		hasChanged = false;
		
		if (!loaded) return false;
		if (!valid) {
			BuildersInventory.LOGGER.error("Refusing to save pages; pages failed to load!");
			return false;
		}
		
		Path root = FabricLoader.getInstance().getGameDir().resolve(BuildersInventory.MOD_ID).resolve("extended_inventory");
		try {
			
			final ArrayList<Pair<class_2487, Path>> storeFiles = new ArrayList<>(PAGES.size());
			final ArrayList<Path> deleteFiles = new ArrayList<>();
			
			// Prepare Page Data
			for (int i = 0; i < PAGES.size(); i++) {
				var page = PAGES.get(i);
				if (!page.valid) continue;
				if (!page.resetChanged()) continue; // resetChanged returns true if changed
				
				class_2487 pageTag = writeTag(registryOps, page);
				
				storeFiles.add(Pair.of(pageTag, root.resolve(FILE_PREFIX + (i + 1) + FILE_SUFFIX)));
			}
			
			// Prepare deletions
			if (deleted > 0) {
				int deleteMax = PAGES.size() + deleted;
				for (int i = PAGES.size(); i < deleteMax; i++) {
					deleteFiles.add(root.resolve(FILE_PREFIX + (i + 1) + FILE_SUFFIX));
				}
				deleted = 0;
			}
			
			// Create directory (doing this outside of thread to make sure it gets made before anything else)
			if (!Files.isDirectory(root)) {
				try {
					Files.createDirectories(root);
				} catch (IOException e) {
					BuildersInventory.LOGGER.error("Error saving extended inventory pages!", e);
					PLAYER_MESSAGE_QUEUE.add(class_2561.method_43471("error.builders_inventory.extended_inventory.save_failed").method_27692(class_124.field_1061));
					return false;
				}
			}
			
			if (storeFiles.size() > 0 || deleteFiles.size() > 0) {
				// Make another thread save the stuff instead of the main one :troll:
				class_156.method_27958().execute(() -> {
					int counter = 0;
					int failCounter = 0;
					
					for (int i = 0; i < storeFiles.size(); i++) {
						var pair = storeFiles.get(i);
						try {
							class_2507.method_10630(pair.getLeft(), pair.getRight());
							++counter;
						} catch (Exception e) {
							++failCounter;
							BuildersInventory.LOGGER.error("Error saving extended inventory page " + i + "!", e);
							PLAYER_MESSAGE_QUEUE.add(class_2561.method_43469("error.builders_inventory.extended_inventory.save_failed.page", i + 1).method_27692(class_124.field_1061));
						}
					}
					
					BuildersInventory.LOGGER.info("Saved {} modified pages!", counter);
					if (failCounter > 0) BuildersInventory.LOGGER.error("Also failed to save {} modified pages...", counter);
					
					counter = 0;
					failCounter = 0;
					
					for (int i = 0; i < deleteFiles.size(); i++) {
						var path = deleteFiles.get(i);
						try {
							if (Files.deleteIfExists(path)) ++counter;
						} catch (Exception e) {
							++failCounter;
							BuildersInventory.LOGGER.error("Error deleting extended inventory file " + path.toString() + "!", e);
							PLAYER_MESSAGE_QUEUE.add(class_2561.method_43469("error.builders_inventory.extended_inventory.delete_failed", i + 1).method_27692(class_124.field_1061));
						}
					}
					
					BuildersInventory.LOGGER.info("Deleted {} old pages!", counter);
					if (failCounter > 0) BuildersInventory.LOGGER.error("Also failed to delete {} old pages...", counter);
				});
			} else {
				BuildersInventory.LOGGER.info("No pages needed to be saved.");
			}
			
			// While that goes on, save the things
			if (ExtendedInventory.getPage() >= 0) {
				class_2487 tag = new class_2487();
				tag.method_10569("version", ModDataFixer.VERSION);
				tag.method_10569("page", ExtendedInventory.getPage());
				class_2507.method_10630(tag, root.resolve("inventory-data.nbt"));
				
				BuildersInventory.LOGGER.info("Saved extended inventory extra data!");
			}
		} catch (Exception e) {
			BuildersInventory.LOGGER.error("Error saving extended inventory pages!", e);
			PLAYER_MESSAGE_QUEUE.add(class_2561.method_43471("error.builders_inventory.extended_inventory.save_failed").method_27692(class_124.field_1061));
			return false;
		}
		return true;
	}
	
	public static class_2487 writeTag(class_6903<class_2520> registryOps, ExtendedInventoryPage page) {
		
		class_2487 tag = new class_2487();
		
		class_2512.method_48310(tag);
		tag.method_10569("version", ModDataFixer.VERSION);
		tag.method_10566("items", page.streamItems()
				.map(stack -> class_1799.field_49266.encodeStart(registryOps, stack)
						.resultOrPartial()
						.orElse(new class_2487()))
				.collect(Collectors.toCollection(class_2499::new)));
		tag.method_10556("locked", page.isLocked());
		if (!page.getName().isBlank()) tag.method_10582("name", page.getName());
		if (!page.icon.method_7960()) {
			var result = class_1799.field_49266.encodeStart(class_2509.field_11560, page.icon).resultOrPartial();
			if (result.isPresent()) {
				tag.method_10566("icon", result.get());
				tag.method_10556("icon_data", page.iconDataActive);
				tag.method_10569("icon_scale_down", page.iconScaleDown);
			}
		}
		if (!page.originalIcon.method_7960()) {
			var result = class_1799.field_49266.encodeStart(class_2509.field_11560, page.icon).resultOrPartial();
			if (result.isPresent()) tag.method_10566("original_icon", result.get());
		}
		
		return tag;
	}
	
	public static ExtendedInventoryPage get(int i) {
		if (!loaded) load();
		if (!valid) return ExtendedInventoryPage.INVALID;
		
		var page = PAGES.get(i);
		return page != null ? page : ExtendedInventoryPage.INVALID;
	}
	
	public static ExtendedInventoryPage create() {
		if (!loaded) load();
		if (!valid) return ExtendedInventoryPage.INVALID;
		
		var page = new ExtendedInventoryPage();
		page.setChanged();
		page.setLocked(false);
		PAGES.add(page);
		if (deleted > 0) --deleted;
		return page;
	}
	
	public static ExtendedInventoryPage reset(int n) {
		if (!loaded) {
			BuildersInventory.LOGGER.warn("Resetting page {} before pages were even loaded... I don't think that's meant to happen.", n);
			load();
		}
		if (!valid) return ExtendedInventoryPage.INVALID;
		if (n >= PAGES.size()) {
			BuildersInventory.LOGGER.error("Tried to reset page {}, but the index was out of bounds!", n);
			return ExtendedInventoryPage.INVALID;
		}
		
		var page = new ExtendedInventoryPage();
		page.setChanged();
		page.setLocked(false);
		PAGES.set(n, page);
		return page;
	}
	
	public static void delete(int n) {
		if (!loaded) {
			BuildersInventory.LOGGER.warn("Deleting page {} before pages were even loaded... I don't think that's meant to happen.", n);
			load();
		}
		if (!valid) return;
		if (n > PAGES.size()) {
			BuildersInventory.LOGGER.error("Tried to delete page {}, but the index was out of bounds!", n);
			return;
		}
		
		PAGES.remove(n);
		++deleted;
		if (PAGES.size() <= 0) create();
		for (int i = n; i < PAGES.size(); i++) PAGES.get(i).setChanged();
		setChanged();
	}
	
	public static boolean contains(int i) {
		if (!loaded) load();
		if (!valid) return false;
		return i < PAGES.size();
	}
	
	public static int size() {
		if (!loaded) load();
		if (!valid) return 1;
		return PAGES.size();
	}
	
	public static boolean isLoaded() {
		return loaded;
	}
	
	public static boolean isValid() {
		if (!loaded) load();
		return valid;
	}
	
	public static void rotatePages(int from, int to, int dist) {
		Collections.rotate(PAGES.subList(from, to), dist);
	}
	
}
