/*
 * Decompiled with CFR 0.152.
 */
package net.lunark.io.scoreboard;

import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import net.lunark.io.database.DatabaseManager;
import org.bukkit.plugin.Plugin;

public final class ScoreboardStorage {
    private final DatabaseManager dbManager;
    private final String poolKey = "scoreboard";
    private final Map<UUID, ScoreboardPlayerData> cache = new ConcurrentHashMap<UUID, ScoreboardPlayerData>();
    private final AtomicInteger enabledCount = new AtomicInteger(0);

    public ScoreboardStorage(Plugin plugin, DatabaseManager dbManager) {
        this.dbManager = dbManager;
        this.initTable();
        plugin.getLogger().info("Scoreboard storage initialized with async support");
    }

    private void initTable() {
        String sql = "CREATE TABLE IF NOT EXISTS scoreboard_players (player_uuid TEXT PRIMARY KEY, enabled BOOLEAN, layout TEXT, last_update BIGINT, join_time BIGINT)";
        this.dbManager.executeUpdate("scoreboard", sql, new Object[0]);
    }

    public CompletableFuture<ScoreboardPlayerData> loadPlayer(UUID playerId) {
        String sql = "SELECT enabled, layout, last_update, join_time FROM scoreboard_players WHERE player_uuid = ?";
        return this.dbManager.executeQuery("scoreboard", sql, rs -> {
            if (rs.next()) {
                return new ScoreboardPlayerData(rs.getBoolean("enabled"), rs.getString("layout"), rs.getLong("last_update"), rs.getLong("join_time"));
            }
            return null;
        }, playerId.toString()).thenApply(opt -> {
            ScoreboardPlayerData data = opt.orElse(new ScoreboardPlayerData(true, null, 0L, System.currentTimeMillis()));
            this.cache.put(playerId, data);
            if (data.enabled()) {
                this.enabledCount.incrementAndGet();
            }
            return data;
        });
    }

    public CompletableFuture<Void> savePlayer(UUID playerId, boolean enabled, String layout) {
        long now = System.currentTimeMillis();
        long joinTime = this.cache.containsKey(playerId) ? this.cache.get(playerId).joinTime() : now;
        this.cache.put(playerId, new ScoreboardPlayerData(enabled, layout, now, joinTime));
        String sql = "INSERT OR REPLACE INTO scoreboard_players VALUES (?, ?, ?, ?, ?)";
        return this.dbManager.executeUpdate("scoreboard", sql, playerId.toString(), enabled, layout, now, joinTime);
    }

    public CompletableFuture<Void> setEnabled(UUID playerId, boolean enabled) {
        ScoreboardPlayerData current = this.cache.getOrDefault(playerId, new ScoreboardPlayerData(true, null, 0L, System.currentTimeMillis()));
        if (current.enabled() != enabled) {
            if (enabled) {
                this.enabledCount.incrementAndGet();
            } else {
                this.enabledCount.decrementAndGet();
            }
        }
        return this.savePlayer(playerId, enabled, current.layout());
    }

    public CompletableFuture<Void> setLayout(UUID playerId, String layout) {
        ScoreboardPlayerData current = this.cache.getOrDefault(playerId, new ScoreboardPlayerData(true, null, 0L, System.currentTimeMillis()));
        return this.savePlayer(playerId, current.enabled(), layout);
    }

    public boolean isEnabled(UUID playerId) {
        return this.cache.getOrDefault(playerId, new ScoreboardPlayerData(true, null, 0L, 0L)).enabled();
    }

    public String getLayout(UUID playerId) {
        return this.cache.getOrDefault(playerId, new ScoreboardPlayerData(true, null, 0L, 0L)).layout();
    }

    public void removeFromCache(UUID playerId) {
        ScoreboardPlayerData data = this.cache.remove(playerId);
        if (data != null && data.enabled()) {
            this.enabledCount.decrementAndGet();
        }
    }

    public int getEnabledPlayerCount() {
        return this.enabledCount.get();
    }

    public Map<UUID, ScoreboardPlayerData> getCacheSnapshot() {
        return Map.copyOf(this.cache);
    }

    public record ScoreboardPlayerData(boolean enabled, String layout, long lastUpdate, long joinTime) {
    }
}

