/*
 * Decompiled with CFR 0.152.
 */
package net.litetex.authback.common.gameprofile;

import com.mojang.authlib.GameProfile;
import com.mojang.authlib.minecraft.client.ObjectMapper;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.time.Duration;
import java.time.Instant;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import net.litetex.authback.shared.collections.AdvancedCollectors;
import net.litetex.authback.shared.external.com.google.common.base.Suppliers;
import net.litetex.authback.shared.json.JSONSerializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GameProfileCacheManager {
    private static final Logger LOG = LoggerFactory.getLogger(GameProfileCacheManager.class);
    private static final Duration DELETE_AFTER_EXECUTION_INTERVAL = Duration.ofHours(12L);
    private static final float TARGET_PROFILE_COUNT_PERCENT = 0.9f;
    private final ObjectMapper objectMapper = ObjectMapper.create();
    private final Path file;
    private final Duration deleteAfter;
    private Instant nextDeletedAfterExecuteTime = Instant.now().plus(DELETE_AFTER_EXECUTION_INTERVAL);
    private final int maxTargetedProfileCount;
    private final int targetedProfileCount;
    private Map<String, UUID> usernameUuids = new HashMap<String, UUID>();
    private Map<UUID, String> uuidUsernames = new HashMap<UUID, String>();
    private Map<UUID, ProfileContainer> uuidProfileContainers = new HashMap<UUID, ProfileContainer>();

    public GameProfileCacheManager(Path file, Duration deleteAfter, int maxTargetedProfileCount) {
        this.file = file;
        this.deleteAfter = deleteAfter;
        if (maxTargetedProfileCount <= 0) {
            throw new IllegalArgumentException("maxTargetedProfileCount needs to be > 1");
        }
        this.maxTargetedProfileCount = maxTargetedProfileCount;
        this.targetedProfileCount = Math.max(Math.round((float)maxTargetedProfileCount * 0.9f), 1);
        this.readFile();
    }

    public void add(GameProfile profile) {
        LOG.debug("Add {}", (Object)profile.id());
        this.uuidProfileContainers.put(profile.id(), new ProfileContainer(this.objectMapper.writeValueAsString((Object)profile), () -> profile, Instant.now()));
        this.usernameUuids.put(profile.name(), profile.id());
        this.uuidUsernames.put(profile.id(), profile.name());
        this.saveAsync();
    }

    public GameProfile findByName(String username) {
        LOG.debug("FindByName {}", (Object)username);
        UUID uuid = this.usernameUuids.get(username);
        if (uuid == null) {
            return null;
        }
        return this.findByUUID(uuid);
    }

    public GameProfile findByUUID(UUID id) {
        LOG.debug("FindByUUID {}", (Object)id);
        long startMs = System.currentTimeMillis();
        ProfileContainer container = this.uuidProfileContainers.get(id);
        if (container == null) {
            return null;
        }
        this.cleanUpIfRequired();
        try {
            GameProfile gameProfile = container.gameProfileSupplier().get();
            LOG.debug("Took {}ms for findByUUID[id={}] to return result", (Object)(System.currentTimeMillis() - startMs), (Object)id);
            return gameProfile;
        }
        catch (Exception ex) {
            LOG.warn("Failed to deserialize game profile", (Throwable)ex);
            this.remove(id);
            return null;
        }
    }

    public Set<UUID> uuids() {
        return new HashSet<UUID>(this.uuidUsernames.keySet());
    }

    public Set<String> names() {
        return new HashSet<String>(this.usernameUuids.keySet());
    }

    private void cleanUpIfRequired() {
        Instant now = Instant.now();
        if (this.nextDeletedAfterExecuteTime.isBefore(now) || this.uuidProfileContainers.size() > this.maxTargetedProfileCount) {
            LOG.debug("Executing cleanup");
            this.nextDeletedAfterExecuteTime = now.plus(DELETE_AFTER_EXECUTION_INTERVAL);
            Instant deleteBefore = now.minus(this.deleteAfter);
            this.removeAll(this.uuidProfileContainers.entrySet().stream().filter(e -> ((ProfileContainer)e.getValue()).createdAt().isBefore(deleteBefore)));
            if (this.uuidProfileContainers.size() > this.targetedProfileCount) {
                Comparator<Map.Entry> comparator = Comparator.comparing(e -> ((ProfileContainer)e.getValue()).createdAt()).reversed();
                this.removeAll(this.uuidProfileContainers.entrySet().stream().sorted(comparator).skip(this.targetedProfileCount));
            }
        }
    }

    private void removeAll(Stream<Map.Entry<UUID, ProfileContainer>> stream) {
        stream.map(Map.Entry::getKey).toList().forEach(this::remove);
    }

    private void remove(UUID uuid) {
        this.uuidProfileContainers.remove(uuid);
        String usernameToRemove = this.uuidUsernames.remove(uuid);
        if (usernameToRemove != null) {
            this.usernameUuids.remove(usernameToRemove);
        }
    }

    private void readFile() {
        if (!Files.exists(this.file, new LinkOption[0])) {
            return;
        }
        long startMs = System.currentTimeMillis();
        try {
            Instant deleteBefore = Instant.now().minus(this.deleteAfter);
            PersistentState persistentState = (PersistentState)JSONSerializer.GSON.fromJson(Files.readString(this.file), PersistentState.class);
            HashMap stringToUUIDCache = new HashMap(persistentState.ensureIdProfiles().size());
            Function<String, UUID> stringToUUIDFunc = s -> stringToUUIDCache.computeIfAbsent(s, UUID::fromString);
            this.uuidProfileContainers = Collections.synchronizedMap(persistentState.ensureIdProfiles().entrySet().stream().filter(e -> ((PersistentState.PersistentProfileContainer)e.getValue()).createdAt().isAfter(deleteBefore)).collect(AdvancedCollectors.toLinkedHashMap(e -> (UUID)stringToUUIDFunc.apply((String)e.getKey()), e -> new ProfileContainer(((PersistentState.PersistentProfileContainer)e.getValue()).serializedGameProfile(), Suppliers.memoize(() -> (GameProfile)this.objectMapper.readValue(((PersistentState.PersistentProfileContainer)e.getValue()).serializedGameProfile(), GameProfile.class)), ((PersistentState.PersistentProfileContainer)e.getValue()).createdAt()))));
            this.usernameUuids = Collections.synchronizedMap(persistentState.ensureUsernameUUIDs().entrySet().stream().map(e -> Map.entry((String)e.getKey(), (UUID)stringToUUIDFunc.apply((String)e.getValue()))).filter(e -> this.uuidProfileContainers.containsKey(e.getValue())).collect(AdvancedCollectors.toLinkedHashMap(Map.Entry::getKey, Map.Entry::getValue)));
            this.uuidUsernames = Collections.synchronizedMap(this.usernameUuids.entrySet().stream().collect(Collectors.toMap(Map.Entry::getValue, Map.Entry::getKey)));
            LOG.debug("Took {}ms to read {}x profiles", (Object)(System.currentTimeMillis() - startMs), (Object)this.uuidProfileContainers.size());
        }
        catch (Exception ex) {
            LOG.warn("Failed to read file['{}']", (Object)this.file, (Object)ex);
        }
    }

    private void saveAsync() {
        CompletableFuture.runAsync(this::saveToFile);
    }

    private synchronized void saveToFile() {
        long startMs = System.currentTimeMillis();
        this.cleanUpIfRequired();
        try {
            PersistentState persistentState = new PersistentState((Map<String, String>)this.usernameUuids.entrySet().stream().collect(AdvancedCollectors.toLinkedHashMap(Map.Entry::getKey, e -> ((UUID)e.getValue()).toString())), (Map<String, PersistentState.PersistentProfileContainer>)this.uuidProfileContainers.entrySet().stream().collect(AdvancedCollectors.toLinkedHashMap(e -> ((UUID)e.getKey()).toString(), e -> ((ProfileContainer)e.getValue()).persist())));
            Files.writeString(this.file, (CharSequence)JSONSerializer.GSON.toJson((Object)persistentState), new OpenOption[0]);
            LOG.debug("Took {}ms to write {}x profiles", (Object)(System.currentTimeMillis() - startMs), (Object)this.uuidProfileContainers.size());
        }
        catch (Exception ex) {
            LOG.warn("Failed to write file['{}']", (Object)this.file, (Object)ex);
        }
    }

    record ProfileContainer(String serializedGameProfile, Supplier<GameProfile> gameProfileSupplier, Instant createdAt) {
        public PersistentState.PersistentProfileContainer persist() {
            return new PersistentState.PersistentProfileContainer(this.serializedGameProfile(), this.createdAt());
        }
    }

    record PersistentState(Map<String, String> usernameUUIDs, Map<String, PersistentProfileContainer> idProfiles) {
        Map<String, String> ensureUsernameUUIDs() {
            return this.usernameUUIDs != null ? this.usernameUUIDs : Map.of();
        }

        Map<String, PersistentProfileContainer> ensureIdProfiles() {
            return this.idProfiles != null ? this.idProfiles : Map.of();
        }

        record PersistentProfileContainer(String serializedGameProfile, Instant createdAt) {
        }
    }
}

