/*
 * Decompiled with CFR 0.152.
 */
package dev.djefrey.colorwheel_patcher;

import com.google.common.collect.Lists;
import dev.djefrey.colorwheel_patcher.FileUtils;
import dev.djefrey.colorwheel_patcher.Version;
import dev.djefrey.colorwheel_patcher.ZipUtils;
import dev.djefrey.colorwheel_patcher.config.ClrwlConfig;
import dev.djefrey.colorwheel_patcher.config.ProcessedConfig;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ClrwlPatcher {
    public static final String MOD_ID = "colorwheel_patcher";
    public static final Logger LOGGER = LoggerFactory.getLogger((String)"colorwheel_patcher");
    public static final String BRAND = "Clrwl";
    public static final String JAR_PATCHES_SUBPATH = "/patches/";
    public static final String USER_PATCHES_SUBPATH = "/patches/";
    public static final String CONFIG_FILENAME = "/config.json";
    public static final Pattern OLD_CLRWL_BRAND_VERSION_REGEX = Pattern.compile(" \\+ Colorwheel_([0-9]+\\.[0-9]+\\.[0-9]+)");
    public static final Pattern CLRWL_BRAND_VERSION_REGEX = Pattern.compile(" \\+ Clrwl_([0-9]+\\.[0-9]+\\.[0-9]+)");
    public static final String OUTDATED_PREFIX = "\u00a7cOutdated\u00a7r ";

    public static void init(Version modVersion, Path shadesrFolder, Path configFolder) {
        ProcessedConfig config = ClrwlPatcher.readPatchConfigs(configFolder);
        LOGGER.info("Available patches: {}", (Object)config.patches().size());
        for (ProcessedConfig.PatchConfig patch : config.patches()) {
            LOGGER.info(" - {} [{}] ({})", new Object[]{patch.shaderName(), String.join((CharSequence)", ", patch.versions()), patch.userPatch() ? "user" : "jar"});
        }
        List<String> shaders = ClrwlPatcher.listShaderpacks(shadesrFolder);
        Map<String, List<Version>> baseShaders = ClrwlPatcher.matchColorwheelShaders(shaders);
        LOGGER.info("Existing base shaders: {}", (Object)baseShaders.size());
        for (Map.Entry<String, List<Version>> entry : baseShaders.entrySet()) {
            List<String> versions = entry.getValue().stream().map(Version::toString).toList();
            LOGGER.info(" - {} [{}]", (Object)entry.getKey(), (Object)String.join((CharSequence)", ", versions));
        }
        ClrwlPatcher.patchShaderpacks(modVersion, config.patches(), baseShaders, shadesrFolder, configFolder);
        ClrwlPatcher.flagOutdatedShaderpacks(modVersion, baseShaders, shadesrFolder);
    }

    private static ProcessedConfig readPatchConfigs(Path configFolder) {
        ClrwlConfig userConfig;
        ClrwlConfig jarConfig;
        try (InputStream in = ClrwlPatcher.class.getResourceAsStream(CONFIG_FILENAME);){
            if (in != null) {
                jarConfig = ClrwlConfig.loadJson(new InputStreamReader(in));
            } else {
                LOGGER.error("Could not read config.json in .jar");
                jarConfig = ClrwlConfig.empty();
            }
        }
        catch (IOException e) {
            LOGGER.error("Could not read config.json in .jar");
            jarConfig = ClrwlConfig.empty();
        }
        Path configPath = configFolder.resolve(CONFIG_FILENAME);
        if (Files.exists(configPath, new LinkOption[0])) {
            try (InputStream in = Files.newInputStream(configPath, new OpenOption[0]);){
                userConfig = ClrwlConfig.loadJson(new InputStreamReader(in));
            }
            catch (IOException e) {
                LOGGER.error("Could not read config.json in config/");
                userConfig = ClrwlConfig.empty();
            }
        } else {
            LOGGER.info("No user config.json found");
            userConfig = ClrwlConfig.empty();
        }
        return ProcessedConfig.processConfigs(jarConfig, userConfig);
    }

    private static List<String> listShaderpacks(Path shadersFolder) {
        List<String> shaders;
        if (!Files.exists(shadersFolder, new LinkOption[0])) {
            LOGGER.info("No shaderpacks folder");
            return Collections.emptyList();
        }
        try (Stream<Path> stream = Files.list(shadersFolder);){
            shaders = stream.filter(p -> p.getFileName().toString().endsWith(".zip") || Files.isDirectory(p, new LinkOption[0])).map(p -> p.getFileName().toString()).toList();
        }
        catch (IOException e) {
            LOGGER.error("Could not list installed shaderpacks");
            return Collections.emptyList();
        }
        return shaders;
    }

    private static Map<String, List<Version>> matchColorwheelShaders(List<String> shaders) {
        ArrayList<String> nonClrwlShaders = new ArrayList<String>();
        HashMap<String, List<Version>> matchMap = new HashMap<String, List<Version>>();
        for (String shader : shaders) {
            Optional<String> optional = ClrwlPatcher.extractBaseShaderName(shader);
            if (optional.isPresent()) {
                Optional<Version> version = ClrwlPatcher.extractColorwheelVersion(shader);
                if (version.isEmpty()) {
                    LOGGER.error("Could not extract Colorwheel version from {}", (Object)shader);
                    continue;
                }
                String baseShader = optional.get();
                matchMap.computeIfAbsent(baseShader, $ -> new ArrayList()).add(version.get());
                continue;
            }
            nonClrwlShaders.add(shader);
            matchMap.computeIfAbsent(shader, $ -> new ArrayList());
        }
        List<String> keysToRm = matchMap.keySet().stream().filter(k -> !nonClrwlShaders.contains(k)).toList();
        for (String string : keysToRm) {
            LOGGER.warn("Base shader '{}' does not exists", (Object)string);
            matchMap.remove(string);
        }
        for (Map.Entry entry : matchMap.entrySet()) {
            ((List)entry.getValue()).sort(Version::compareTo);
        }
        return matchMap;
    }

    private static void patchShaderpacks(Version modVersion, List<ProcessedConfig.PatchConfig> patches, Map<String, List<Version>> baseShaders, Path shaderpacksFolder, Path configFolder) {
        for (Map.Entry<String, List<Version>> entry : baseShaders.entrySet()) {
            String shader = entry.getKey();
            List<Version> clrwlVersions = entry.getValue();
            if (clrwlVersions.contains(modVersion)) {
                LOGGER.info("'{}' already patched with version {}", (Object)shader, (Object)modVersion);
                continue;
            }
            Optional<ProcessedConfig.PatchConfig> maybePatch = patches.stream().filter(p -> p.doesMatchWith(shader)).findFirst();
            if (maybePatch.isEmpty()) {
                LOGGER.info("Could not find patch for {}", (Object)shader);
                continue;
            }
            ProcessedConfig.PatchConfig patch = maybePatch.get();
            boolean isZip = shader.toLowerCase().endsWith(".zip");
            LOGGER.info("Found {}patch '{}' for {}", new Object[]{patch.userPatch() ? "user " : "", patch.shaderName(), shader});
            if (patch.versions().stream().noneMatch(shader::contains)) {
                LOGGER.info("'{}' version is unknown or unsupported", (Object)shader);
                continue;
            }
            Path shaderpack = shaderpacksFolder.resolve(shader);
            String patchedName = ClrwlPatcher.getPatchShaderpackName(shader, modVersion, isZip);
            try {
                Path tmpFolder = ClrwlPatcher.createTmpDirectory();
                if (isZip) {
                    try (FileInputStream in = new FileInputStream(shaderpack.toFile());){
                        ZipUtils.extract(in, tmpFolder.toFile());
                    }
                } else {
                    FileUtils.copyAndMerge(shaderpack, tmpFolder);
                }
                try {
                    InputStream zipIn = patch.getPatchZip(configFolder);
                    Optional<File> shadersFolder = FileUtils.findFolderInChildren(tmpFolder.toFile(), "shaders");
                    if (shadersFolder.isEmpty()) {
                        LOGGER.error("Could not find shaders/ in {}", (Object)shader);
                        continue;
                    }
                    File parent = shadersFolder.get().getParentFile();
                    ZipUtils.extract(zipIn, parent);
                }
                catch (FileNotFoundException e) {
                    LOGGER.error("Could not find {}patch {}", (Object)(patch.userPatch() ? "user " : ""), (Object)patch.shaderName());
                    FileUtils.deleteRecursive(tmpFolder.toFile());
                    continue;
                }
                Path patchedPath = shaderpacksFolder.resolve(patchedName);
                if (isZip) {
                    ZipUtils.compress(tmpFolder.toFile(), patchedPath.toFile());
                    try {
                        FileUtils.deleteRecursive(tmpFolder.toFile());
                    }
                    catch (IOException e) {
                        LOGGER.warn("Could not delete tmp folder");
                    }
                } else {
                    FileUtils.moveRecursive(tmpFolder, patchedPath);
                }
            }
            catch (IOException e) {
                LOGGER.error("Could not patch shaderpack", (Throwable)e);
                continue;
            }
            try {
                Path path;
                Path ogConfigPath = null;
                for (Version version : Lists.reverse(clrwlVersions)) {
                    String filename = ClrwlPatcher.getPatchShaderpackName(shader, version, shader.endsWith(".zip")) + ".txt";
                    Path path2 = shaderpacksFolder.resolve(filename);
                    if (Files.exists(path2, new LinkOption[0])) {
                        ogConfigPath = path2;
                        break;
                    }
                    path2 = shaderpacksFolder.resolve(OUTDATED_PREFIX + filename);
                    if (!Files.exists(path2, new LinkOption[0])) continue;
                    ogConfigPath = path2;
                    break;
                }
                if (ogConfigPath == null && Files.exists(path = shaderpacksFolder.resolve(shader + ".txt"), new LinkOption[0])) {
                    ogConfigPath = path;
                }
                if (ogConfigPath != null) {
                    Path newConfigPath = shaderpacksFolder.resolve(patchedName + ".txt");
                    LOGGER.info("Found option file '{}'", (Object)ogConfigPath.getFileName().toString());
                    Files.copy(ogConfigPath, newConfigPath, new CopyOption[0]);
                    continue;
                }
                LOGGER.info("Could not find option file for {}", (Object)shader);
            }
            catch (IOException e) {
                LOGGER.error("Could not copy shaderpack config file", (Throwable)e);
            }
        }
    }

    private static void flagOutdatedShaderpacks(Version modVersion, Map<String, List<Version>> baseShaders, Path shaderpacksFolder) {
        for (Map.Entry<String, List<Version>> entry : baseShaders.entrySet()) {
            String shader = entry.getKey();
            List<Version> clrwlVersions = entry.getValue();
            for (Version version : clrwlVersions) {
                String patchedShader;
                Path path;
                if (version.compareTo(modVersion) >= 0 || !Files.exists(path = shaderpacksFolder.resolve(patchedShader = ClrwlPatcher.getPatchShaderpackName(shader, version, shader.endsWith(".zip"))), new LinkOption[0])) continue;
                LOGGER.info("Flag '{}' as outdated", (Object)patchedShader);
                Path outdatedPath = shaderpacksFolder.resolve(OUTDATED_PREFIX + patchedShader);
                try {
                    Files.move(path, outdatedPath, new CopyOption[0]);
                }
                catch (IOException e) {
                    LOGGER.error("Could not set shader to outdated", (Throwable)e);
                    continue;
                }
                Path configPath = shaderpacksFolder.resolve(patchedShader + ".txt");
                if (!Files.exists(configPath, new LinkOption[0])) continue;
                LOGGER.info("Flag config file as outdated");
                Path outdatedConfigPath = shaderpacksFolder.resolve(OUTDATED_PREFIX + patchedShader + ".txt");
                try {
                    Files.move(configPath, outdatedConfigPath, new CopyOption[0]);
                }
                catch (IOException e) {
                    LOGGER.error("Could not set config file to outdated", (Throwable)e);
                }
            }
        }
    }

    private static Path createTmpDirectory() throws IOException {
        return Files.createTempDirectory("clrwl-patcher", new FileAttribute[0]);
    }

    private static String getPatchShaderpackName(String shader, Version version, boolean isZip) {
        String brand = version.compareTo(new Version(1, 0, 0)) < 0 ? "Colorwheel" : BRAND;
        if (isZip) {
            return shader.substring(0, shader.length() - 4) + " + " + brand + "_" + String.valueOf(version) + ".zip";
        }
        return shader + " + " + brand + "_" + String.valueOf(version);
    }

    private static Optional<String> extractBaseShaderName(String shader) {
        Matcher matcher = CLRWL_BRAND_VERSION_REGEX.matcher(shader);
        if (matcher.find()) {
            String clean = matcher.replaceAll("");
            if (clean.startsWith(OUTDATED_PREFIX)) {
                clean = clean.substring(OUTDATED_PREFIX.length());
            }
            return Optional.of(clean);
        }
        Matcher oldMatcher = OLD_CLRWL_BRAND_VERSION_REGEX.matcher(shader);
        if (oldMatcher.find()) {
            String clean = oldMatcher.replaceAll("");
            if (clean.startsWith(OUTDATED_PREFIX)) {
                clean = clean.substring(OUTDATED_PREFIX.length());
            }
            return Optional.of(clean);
        }
        return Optional.empty();
    }

    private static Optional<Version> extractColorwheelVersion(String shader) {
        Matcher matcher = CLRWL_BRAND_VERSION_REGEX.matcher(shader);
        if (matcher.find()) {
            int[] version = Arrays.stream(matcher.group(1).split("\\.")).mapToInt(Integer::parseInt).toArray();
            return Optional.of(Version.fromArray(version));
        }
        Matcher oldMatcher = OLD_CLRWL_BRAND_VERSION_REGEX.matcher(shader);
        if (oldMatcher.find()) {
            int[] version = Arrays.stream(oldMatcher.group(1).split("\\.")).mapToInt(Integer::parseInt).toArray();
            return Optional.of(Version.fromArray(version));
        }
        return Optional.empty();
    }

    public static Optional<String> findPatchedShaderpackInFolder(String shaderpack, Path shaderpacksFolder) {
        File folder = shaderpacksFolder.toFile();
        if (!folder.isDirectory()) {
            return Optional.empty();
        }
        File[] children = folder.listFiles();
        if (children == null) {
            return Optional.empty();
        }
        Version maxVersion = null;
        String patchedShaderpack = null;
        for (File pack : children) {
            Optional<Version> maybeVersion;
            Optional<String> maybeBasePack;
            if (pack.getName().endsWith(".txt") || (maybeBasePack = ClrwlPatcher.extractBaseShaderName(pack.getName())).isEmpty() || !maybeBasePack.get().equals(shaderpack) || (maybeVersion = ClrwlPatcher.extractColorwheelVersion(pack.getName())).isEmpty() || maxVersion != null && maybeVersion.get().compareTo(maxVersion) <= 0) continue;
            maxVersion = maybeVersion.get();
            patchedShaderpack = pack.getName();
        }
        return Optional.ofNullable(patchedShaderpack);
    }
}

