package com.diamondgoobird.trialspawnertimer;

import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.HashMap;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_5321;
import net.minecraft.class_8967;

/**
 * Handles every cooldown timer for Trail Spawners
 * Uses nested HashMap of World then BlockPos to get the Long of when the timer ends
 */
public class TimerHandler {
    private static final HashMap<class_5321<class_1937>, HashMap<class_2338, Timer>> timers = new HashMap<>();

    /**
     * Returns whether a Trial Spawner's timer should get deleted when the block switches to this state
     *
     * @param state the state to test
     * @return true if changing to the state should delete its timer, false otherwise
     */
    public static boolean shouldReset(class_8967 state) {
        return !TrialSpawnerTimer.ACCEPTABLE_STATES.contains(state);
    }

    /**
     * Returns whether the state should trigger the creation of a timer if there isn't one already
     *
     * @param state the state to test
     * @return true if changing to the state should create a timer (assuming there isn't one), false otherwise
     */
    public static boolean shouldCreate(class_8967 state) {
        return !shouldReset(state);
    }

    /**
     * Returns true if there's a cooldown being tracked
     * at a given position in a given world
     *
     * @param world the world to check the block of
     * @param pos the position where the trial spawner is
     * @return true if there is an active cooldown timer, false otherwise
     */
    public static boolean hasTimer(class_1937 world, class_2338 pos) {
        return getTimer(world, pos) != null;
    }

    /**
     * Inserts the ending time (in milliseconds) of the trial spawner cooldown
     * at a specific position in a given world
     *
     * @param world the World in which the blockPos refers to (dimension)
     * @param pos   the position where the TrialSpawner is at
     * @param time  the time in milliseconds when the timer should end
     */
    public static void insertTime(class_1937 world, class_2338 pos, long time, long cooldown) {
        // Get the map or have a new one inserted
        HashMap<class_2338, Timer> t = timers.computeIfAbsent(world.method_27983(), k -> new HashMap<>());
        t.put(pos, new Timer(time, cooldown));
        TrialSpawnerTimer.LOGGER.info("Timer added at block {} in {} ending {} minutes from now", pos, world.method_27983().method_29177(), Duration.of(cooldown, ChronoUnit.SECONDS).toMinutes() / 20);
    }

    /**
     * Fetches the time in milliseconds when the timer should end
     * for a specific block position in a given world, or returns 0
     *
     * @param world the World in which the blockPos refers to (dimension)
     * @param pos   the position where the TrialSpawner is at
     * @return      the time in milliseconds when the timer should end or 0 if nonexistent
     */
    public static Timer getTimer(class_1937 world, class_2338 pos) {
        // Gets the timer map for the specific world
        HashMap<class_2338, Timer> t = timers.get(world.method_27983());
        // If it doesn't exist yet just return 0
        if (t == null) {
            return null;
        }
        // Return time or 0
        return t.get(pos);
    }

    /**
     * Deletes the timer tracking the trial spawner cooldown
     * at a specific position in a given world
     *
     * @param world the World in which the blockPos refers to (dimension)
     * @param pos   the position where the TrialSpawner is at
     */
    public static void deleteTime(class_1937 world, class_2338 pos) {
        // Gets the timer map for the specific world
        HashMap<class_2338, Timer> t = timers.get(world.method_27983());
        // Can't delete if it's already null, so we're done
        if (t == null) {
            return;
        }
        // It's not null so remove it
        Timer ti = t.remove(pos);
        if (ti != null) {
            long timeLeft = Duration.of(ti.getTimerEnd() - world.method_8510(), ChronoUnit.SECONDS).toMinutes() / 20;
            TrialSpawnerTimer.LOGGER.info("Timer removed at block {} in {} with {} minutes left", pos, world.method_27983().method_29177(), timeLeft);
        }
        // If it's empty then remove the hashmap since we're not using it
        if (t.isEmpty()) {
            timers.remove(world.method_27983());
        }
    }
}
