import gzip
import os
import re
from pathlib import Path

# Mapping of old names to new names
MAPPING = {
    "weaponsrack2": "bs_prop_pack_weaponsrack2",
    "minecart_filled": "bs_prop_pack_minecart_filled",
    "minecart": "bs_prop_pack_minecart",
    "ladder_tall": "bs_prop_pack_ladder_tall",
    "ladder_short": "bs_prop_pack_ladder_short",
    "weaponsrack": "bs_prop_pack_weaponsrack",
    "skulls": "bs_prop_pack_skulls",
    "skeleton": "bs_prop_pack_skeleton",
    "crypt": "bs_prop_pack_crypt",
    "fallenswordbbmodel": "bs_prop_pack_fallenswordbbmodel",
    "coffin": "bs_prop_pack_coffin",
    "crate2": "bs_prop_pack_crate2",
    "bench2": "bs_prop_pack_bench2",
    "bench": "bs_prop_pack_bench",
    "flag": "bs_prop_pack_flag",
    "flag_standing": "bs_prop_pack_flag_standing",
    "flag2": "bs_prop_pack_flag2",
    "chair2": "bs_prop_pack_chair2",
    "sling": "bs_prop_pack_sling",
    "scrolltable": "bs_prop_pack_scrolltable",
    "apple_crate": "bs_prop_pack_apple_crate",
    "potato_crate": "bs_prop_pack_potato_crate",
    "empty_crate": "bs_prop_pack_empty_crate",
    "carrot_crate": "bs_prop_pack_carrot_crate",
    "emptybasket": "bs_prop_pack_emptybasket",
    "carrotbasket": "bs_prop_pack_carrotbasket",
    "applebasket": "bs_prop_pack_applebasket",
    "stool": "bs_prop_pack_stool",
    "alchemy_table": "bs_prop_pack_alchemy_table",
    "shelf2": "bs_prop_pack_shelf2",
    "crate4": "bs_prop_pack_crate4",
    "crate": "bs_prop_pack_crate",
    "plant": "bs_prop_pack_plant",
    "parktable": "bs_prop_pack_parktable",
    "crate3": "bs_prop_pack_crate3",
    "target": "bs_prop_pack_target",
    "birdhouse": "bs_prop_pack_birdhouse",
    "vase-small": "bs_prop_pack_vase-small",
    "vase": "bs_prop_pack_vase",
    "light2": "bs_prop_pack_light2",
    "light": "bs_prop_pack_light",
    "stool2": "bs_prop_pack_stool2",
    "tallstool": "bs_prop_pack_tallstool",
    "tablestool": "bs_prop_pack_tablestool",
    "board": "bs_prop_pack_board",
    "firecauldron": "bs_prop_pack_firecauldron",
    "awning": "bs_prop_pack_awning",
    "chair": "bs_prop_pack_chair",
    "clothtable": "bs_prop_pack_clothtable",
    "table big": "bs_prop_pack_table big",
    "table": "bs_prop_pack_table",
    "bones": "bs_prop_pack_bones",
    "shelf": "bs_prop_pack_shelf",
    "throne": "bs_prop_pack_throne",
    "weaponsrack3": "bs_prop_pack_weaponsrack3",
}

def process_schematic(file_path):
    """Process a single schematic file"""
    try:
        # Read and decompress the file as binary
        with gzip.open(file_path, 'rb') as f:
            content = f.read()
        
        # Track if any replacements were made
        modified = False
        original_content = content
        
        # Replace each mapping (working with bytes)
        # NBT format: key\x00\xLENGTHvalue where LENGTH is the byte length of value
        for old_name, new_name in MAPPING.items():
            old_bytes = old_name.encode('utf-8')
            new_bytes = new_name.encode('utf-8')
            old_len = len(old_bytes)
            new_len = len(new_bytes)
            
            # Pattern for freeminecraftmodels:entity
            pattern1 = b'freeminecraftmodels:entity\x00' + bytes([old_len]) + old_bytes
            replacement1 = b'freeminecraftmodels:entity\x00' + bytes([new_len]) + new_bytes
            
            # Pattern for freeminecraftmodels:prop
            pattern2 = b'freeminecraftmodels:prop\x00' + bytes([old_len]) + old_bytes
            replacement2 = b'freeminecraftmodels:prop\x00' + bytes([new_len]) + new_bytes
            
            if pattern1 in content or pattern2 in content:
                content = content.replace(pattern1, replacement1)
                content = content.replace(pattern2, replacement2)
                modified = True
        
        # Save back if modified
        if modified:
            with gzip.open(file_path, 'wb') as f:
                f.write(content)
            print(f"[+] Modified: {file_path}")
            return True
        else:
            print(f"[-] Skipped: {file_path} (no matches)")
            return False
            
    except Exception as e:
        print(f"[!] Error processing {file_path}: {e}")
        return False

def process_directory(directory_path):
    """Process all schematic files in a directory"""
    directory = Path(directory_path)
    
    if not directory.exists():
        print(f"Error: Directory '{directory_path}' does not exist")
        return
    
    # Find all .schem files
    schematic_files = list(directory.rglob('*.schem'))
    
    if not schematic_files:
        print(f"No .schem files found in '{directory_path}'")
        return
    
    print(f"Found {len(schematic_files)} schematic file(s)")
    print("Processing...\n")
    
    modified_count = 0
    for file_path in schematic_files:
        if process_schematic(file_path):
            modified_count += 1
    
    print(f"\nComplete! Modified {modified_count} out of {len(schematic_files)} files.")

if __name__ == "__main__":
    import sys
    
    if len(sys.argv) != 2:
        print("Usage: python script.py <directory_path>")
        sys.exit(1)
    
    directory = sys.argv[1]
    process_directory(directory)