/*
 * Decompiled with CFR 0.152.
 */
package org.texboobcat.worldManager.service;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import org.bukkit.Axis;
import org.bukkit.Bukkit;
import org.bukkit.Location;
import org.bukkit.Material;
import org.bukkit.World;
import org.bukkit.block.Block;
import org.bukkit.block.data.BlockData;
import org.bukkit.block.data.Orientable;
import org.bukkit.configuration.ConfigurationSection;
import org.bukkit.configuration.file.FileConfiguration;
import org.bukkit.configuration.file.YamlConfiguration;
import org.bukkit.entity.Player;
import org.bukkit.event.Event;
import org.bukkit.plugin.Plugin;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.texboobcat.worldManager.WorldManager;
import org.texboobcat.worldManager.api.PortalService;
import org.texboobcat.worldManager.api.event.PortalActivateEvent;
import org.texboobcat.worldManager.api.event.PortalTeleportEvent;
import org.texboobcat.worldManager.model.Portal;
import org.texboobcat.worldManager.model.PortalDestination;
import org.texboobcat.worldManager.model.PortalRegion;

public class WorldManagerPortalService
implements PortalService {
    private final WorldManager plugin;
    private final Map<String, Portal> portals = new ConcurrentHashMap<String, Portal>();
    private final Map<UUID, Map<String, Long>> playerCooldowns = new ConcurrentHashMap<UUID, Map<String, Long>>();
    private final Map<UUID, String> pendingTeleports = new ConcurrentHashMap<UUID, String>();
    private volatile boolean saveScheduled = false;
    private int saveTaskId = -1;
    private int appearanceTickerTaskId = -1;
    private int appearanceTickCounter = 0;
    private File dataFile;
    private FileConfiguration data;

    public WorldManagerPortalService(WorldManager plugin) {
        this.plugin = plugin;
        this.load();
        this.startAppearanceTicker();
    }

    private void load() {
        try {
            if (!this.plugin.getDataFolder().exists()) {
                this.plugin.getDataFolder().mkdirs();
            }
            this.dataFile = new File(this.plugin.getDataFolder(), "portals.yml");
            if (!this.dataFile.exists()) {
                this.dataFile.createNewFile();
            }
            this.data = YamlConfiguration.loadConfiguration((File)this.dataFile);
            this.portals.clear();
            ConfigurationSection portalsSection = this.data.getConfigurationSection("portals");
            if (portalsSection != null) {
                for (String name : portalsSection.getKeys(false)) {
                    ConfigurationSection portalSection = portalsSection.getConfigurationSection(name);
                    if (portalSection == null) continue;
                    Portal portal = Portal.load(name, portalSection);
                    this.portals.put(name.toLowerCase(), portal);
                }
            }
            this.plugin.getLogger().info("Loaded " + this.portals.size() + " portals");
        }
        catch (IOException e) {
            this.plugin.getLogger().severe("Failed to load portals.yml: " + e.getMessage());
        }
    }

    @Override
    public void scheduleSave() {
        if (this.saveScheduled) {
            return;
        }
        this.saveScheduled = true;
        this.saveTaskId = Bukkit.getScheduler().runTaskLater((Plugin)this.plugin, () -> {
            try {
                this.savePortals();
            }
            finally {
                this.saveScheduled = false;
                this.saveTaskId = -1;
            }
        }, 100L).getTaskId();
    }

    private void startAppearanceTicker() {
        if (this.appearanceTickerTaskId != -1) {
            return;
        }
        this.appearanceTickerTaskId = Bukkit.getScheduler().runTaskTimer((Plugin)this.plugin, () -> {
            try {
                ++this.appearanceTickCounter;
                for (Portal portal : this.portals.values()) {
                    int period;
                    if (portal.getRegion() == null || !portal.getAppearanceType().equalsIgnoreCase("PARTICLES") || this.appearanceTickCounter % (period = Math.max(1, portal.getAppearanceParticlePeriod())) != 0) continue;
                    this.spawnConfiguredParticles(portal);
                }
            }
            catch (Throwable throwable) {
                // empty catch block
            }
        }, 40L, 40L).getTaskId();
    }

    public void applyAppearance(@NotNull Portal portal) {
        PortalRegion region = portal.getRegion();
        if (region == null) {
            return;
        }
        World world = region.getWorld();
        if (world == null) {
            return;
        }
        String type = portal.getAppearanceType();
        this.clearPortalBlocks(region);
        if (type.equalsIgnoreCase("NETHER")) {
            this.fillPlaneWithNetherPortal(portal);
        } else if (type.equalsIgnoreCase("END")) {
            this.fillPlaneWithEndPortal(portal);
        }
    }

    private void fillPlaneWithNetherPortal(Portal portal) {
        PortalRegion region = portal.getRegion();
        if (region == null) {
            return;
        }
        World world = region.getWorld();
        if (world == null) {
            return;
        }
        String plane = portal.getAppearancePlane();
        int minX = region.getMinX();
        int maxX = region.getMaxX();
        int minY = region.getMinY();
        int maxY = region.getMaxY();
        int minZ = region.getMinZ();
        int maxZ = region.getMaxZ();
        int midX = (minX + maxX) / 2;
        int midZ = (minZ + maxZ) / 2;
        BlockData data = Bukkit.createBlockData((Material)Material.NETHER_PORTAL);
        if ("X".equalsIgnoreCase(plane)) {
            Axis axis = Axis.Z;
            if (data instanceof Orientable) {
                ((Orientable)data).setAxis(axis);
            }
            int x = midX;
            for (int y = minY; y <= maxY; ++y) {
                for (int z = minZ; z <= maxZ; ++z) {
                    Block b = world.getBlockAt(x, y, z);
                    if (b.getType() != Material.AIR) continue;
                    b.setBlockData(data, false);
                }
            }
            return;
        }
        if ("Z".equalsIgnoreCase(plane)) {
            Axis axis = Axis.X;
            if (data instanceof Orientable) {
                ((Orientable)data).setAxis(axis);
            }
            int z = midZ;
            for (int y = minY; y <= maxY; ++y) {
                for (int x = minX; x <= maxX; ++x) {
                    Block b = world.getBlockAt(x, y, z);
                    if (b.getType() != Material.AIR) continue;
                    b.setBlockData(data, false);
                }
            }
            return;
        }
        int thicknessX = maxX - minX + 1;
        int thicknessZ = maxZ - minZ + 1;
        if (thicknessX <= thicknessZ) {
            Axis axis = Axis.Z;
            if (data instanceof Orientable) {
                ((Orientable)data).setAxis(axis);
            }
            int x = midX;
            for (int y = minY; y <= maxY; ++y) {
                for (int z = minZ; z <= maxZ; ++z) {
                    Block b = world.getBlockAt(x, y, z);
                    if (b.getType() != Material.AIR) continue;
                    b.setBlockData(data, false);
                }
            }
        } else {
            Axis axis = Axis.X;
            if (data instanceof Orientable) {
                ((Orientable)data).setAxis(axis);
            }
            int z = midZ;
            for (int y = minY; y <= maxY; ++y) {
                for (int x = minX; x <= maxX; ++x) {
                    Block b = world.getBlockAt(x, y, z);
                    if (b.getType() != Material.AIR) continue;
                    b.setBlockData(data, false);
                }
            }
        }
    }

    private void fillPlaneWithEndPortal(Portal portal) {
        PortalRegion region = portal.getRegion();
        if (region == null) {
            return;
        }
        World world = region.getWorld();
        if (world == null) {
            return;
        }
        String plane = portal.getAppearancePlane();
        int minX = region.getMinX();
        int maxX = region.getMaxX();
        int minY = region.getMinY();
        int maxY = region.getMaxY();
        int minZ = region.getMinZ();
        int maxZ = region.getMaxZ();
        int midX = (minX + maxX) / 2;
        int midY = (minY + maxY) / 2;
        int midZ = (minZ + maxZ) / 2;
        if ("HORIZONTAL".equalsIgnoreCase(plane)) {
            int y = midY;
            for (int x = minX; x <= maxX; ++x) {
                for (int z = minZ; z <= maxZ; ++z) {
                    Block b = world.getBlockAt(x, y, z);
                    if (b.getType() != Material.AIR) continue;
                    b.setType(Material.END_PORTAL, false);
                }
            }
            return;
        }
        if ("X".equalsIgnoreCase(plane)) {
            int x = midX;
            for (int y = minY; y <= maxY; ++y) {
                for (int z = minZ; z <= maxZ; ++z) {
                    Block b = world.getBlockAt(x, y, z);
                    if (b.getType() != Material.AIR) continue;
                    b.setType(Material.END_PORTAL, false);
                }
            }
            return;
        }
        if ("Z".equalsIgnoreCase(plane)) {
            int z = midZ;
            for (int y = minY; y <= maxY; ++y) {
                for (int x = minX; x <= maxX; ++x) {
                    Block b = world.getBlockAt(x, y, z);
                    if (b.getType() != Material.AIR) continue;
                    b.setType(Material.END_PORTAL, false);
                }
            }
            return;
        }
        int thicknessX = maxX - minX + 1;
        int thicknessY = maxY - minY + 1;
        int thicknessZ = maxZ - minZ + 1;
        if (thicknessY <= thicknessX && thicknessY <= thicknessZ) {
            int y = midY;
            for (int x = minX; x <= maxX; ++x) {
                for (int z = minZ; z <= maxZ; ++z) {
                    Block b = world.getBlockAt(x, y, z);
                    if (b.getType() != Material.AIR) continue;
                    b.setType(Material.END_PORTAL, false);
                }
            }
        } else if (thicknessX <= thicknessZ) {
            int x = midX;
            for (int y = minY; y <= maxY; ++y) {
                for (int z = minZ; z <= maxZ; ++z) {
                    Block b = world.getBlockAt(x, y, z);
                    if (b.getType() != Material.AIR) continue;
                    b.setType(Material.END_PORTAL, false);
                }
            }
        } else {
            int z = midZ;
            for (int y = minY; y <= maxY; ++y) {
                for (int x = minX; x <= maxX; ++x) {
                    Block b = world.getBlockAt(x, y, z);
                    if (b.getType() != Material.AIR) continue;
                    b.setType(Material.END_PORTAL, false);
                }
            }
        }
    }

    public void clearPortalBlocks(PortalRegion region) {
        World world = region.getWorld();
        if (world == null) {
            return;
        }
        for (int x = region.getMinX(); x <= region.getMaxX(); ++x) {
            for (int y = region.getMinY(); y <= region.getMaxY(); ++y) {
                for (int z = region.getMinZ(); z <= region.getMaxZ(); ++z) {
                    Block block = world.getBlockAt(x, y, z);
                    Material t = block.getType();
                    if (t != Material.NETHER_PORTAL && t != Material.END_PORTAL) continue;
                    block.setType(Material.AIR, false);
                }
            }
        }
    }

    private void spawnConfiguredParticles(@NotNull Portal portal) {
        PortalRegion region = portal.getRegion();
        if (region == null) {
            return;
        }
        World world = region.getWorld();
        if (world == null) {
            return;
        }
        Location center = region.getCenter();
        if (center == null) {
            return;
        }
        world.spawnParticle(portal.getAppearanceParticle(), center, portal.getAppearanceParticleCount(), portal.getAppearanceParticleSpreadX(), portal.getAppearanceParticleSpreadY(), portal.getAppearanceParticleSpreadZ(), portal.getAppearanceParticleSpeed());
    }

    @Override
    public void savePortals() {
        try {
            if (this.data == null) {
                this.data = new YamlConfiguration();
            }
            this.data.set("portals", null);
            ConfigurationSection portalsSection = this.data.createSection("portals");
            for (Portal portal : this.portals.values()) {
                ConfigurationSection portalSection = portalsSection.createSection(portal.getName());
                portal.save(portalSection);
            }
            this.data.save(this.dataFile);
        }
        catch (IOException e) {
            this.plugin.getLogger().severe("Failed to save portals.yml: " + e.getMessage());
        }
    }

    @Override
    public void reloadPortals() {
        this.load();
    }

    @Override
    @Nullable
    public Portal createPortal(@NotNull String name) {
        String key = name.toLowerCase();
        if (this.portals.containsKey(key)) {
            return null;
        }
        Portal portal = new Portal(name);
        this.portals.put(key, portal);
        this.scheduleSave();
        return portal;
    }

    @Override
    public boolean deletePortal(@NotNull String name) {
        String key = name.toLowerCase();
        Portal removed = this.portals.remove(key);
        if (removed != null) {
            for (Map<String, Long> cooldowns : this.playerCooldowns.values()) {
                cooldowns.remove(key);
            }
            this.scheduleSave();
            return true;
        }
        return false;
    }

    @Override
    @Nullable
    public Portal getPortal(@NotNull String name) {
        return this.portals.get(name.toLowerCase());
    }

    @Override
    @NotNull
    public Collection<Portal> getAllPortals() {
        return new ArrayList<Portal>(this.portals.values());
    }

    @Override
    @NotNull
    public Collection<Portal> getPortalsInWorld(@NotNull String worldName) {
        return this.portals.values().stream().filter(p -> p.getRegion() != null && p.getRegion().getWorldName().equalsIgnoreCase(worldName)).collect(Collectors.toList());
    }

    @Override
    @Nullable
    public Portal getPortalAt(@NotNull Location location) {
        for (Portal portal : this.portals.values()) {
            if (!portal.isEnabled() || !portal.contains(location)) continue;
            return portal;
        }
        return null;
    }

    @Override
    public boolean isNameAvailable(@NotNull String name) {
        return !this.portals.containsKey(name.toLowerCase());
    }

    @Override
    @NotNull
    public CompletableFuture<Boolean> teleportThroughPortal(@NotNull Player player, @NotNull Portal portal) {
        CompletableFuture<Boolean> future = new CompletableFuture<Boolean>();
        if (!portal.isComplete()) {
            future.complete(false);
            return future;
        }
        PortalActivateEvent activateEvent = new PortalActivateEvent(player, portal);
        Bukkit.getPluginManager().callEvent((Event)activateEvent);
        if (activateEvent.isCancelled()) {
            if (activateEvent.getCancelReason() != null) {
                player.sendMessage("\u00a7c" + activateEvent.getCancelReason());
            }
            future.complete(false);
            return future;
        }
        if (!this.canUsePortal(player, portal)) {
            player.sendMessage("\u00a7cYou don't have permission to use this portal.");
            future.complete(false);
            return future;
        }
        long cooldown = this.getRemainingCooldown(player, portal);
        if (cooldown > 0L) {
            player.sendMessage("\u00a7cPlease wait " + cooldown + " seconds before using this portal again.");
            future.complete(false);
            return future;
        }
        Location destination = this.resolveDestination(portal);
        if (destination == null) {
            player.sendMessage("\u00a7cPortal destination is invalid or unavailable.");
            future.complete(false);
            return future;
        }
        Location from = player.getLocation().clone();
        int warmup = portal.getWarmupSeconds();
        if (warmup > 0) {
            player.sendMessage("\u00a77Teleporting in \u00a7b" + warmup + "\u00a77 seconds...");
            this.pendingTeleports.put(player.getUniqueId(), portal.getName());
            Bukkit.getScheduler().runTaskLater((Plugin)this.plugin, () -> {
                if (!player.isOnline()) {
                    future.complete(false);
                    return;
                }
                String pendingPortal = this.pendingTeleports.remove(player.getUniqueId());
                if (pendingPortal == null) {
                    player.sendMessage("\u00a7cTeleport cancelled - you moved!");
                    future.complete(false);
                    return;
                }
                if (!pendingPortal.equals(portal.getName())) {
                    player.sendMessage("\u00a7cTeleport cancelled!");
                    future.complete(false);
                    return;
                }
                this.performTeleport(player, portal, from, destination, future);
            }, (long)warmup * 20L);
        } else {
            this.performTeleport(player, portal, from, destination, future);
        }
        return future;
    }

    private void performTeleport(Player player, Portal portal, Location from, Location destination, CompletableFuture<Boolean> future) {
        player.teleportAsync(destination).thenAccept(success -> {
            if (success.booleanValue()) {
                portal.incrementUseCount();
                this.scheduleSave();
                if (portal.getCooldownSeconds() > 0) {
                    this.playerCooldowns.computeIfAbsent(player.getUniqueId(), k -> new ConcurrentHashMap()).put(portal.getName().toLowerCase(), System.currentTimeMillis());
                }
                if (portal.isPlaySound()) {
                    player.playSound(player.getLocation(), portal.getSound(), 1.0f, 1.0f);
                }
                PortalTeleportEvent teleportEvent = new PortalTeleportEvent(player, portal, from, destination);
                Bukkit.getPluginManager().callEvent((Event)teleportEvent);
                future.complete(true);
            } else {
                player.sendMessage("\u00a7cTeleport failed!");
                future.complete(false);
            }
        });
    }

    @Nullable
    private Location resolveDestination(Portal portal) {
        return this.resolveDestination(portal, new HashSet<String>(), 0);
    }

    private Location resolveDestination(Portal portal, Set<String> visited, int depth) {
        if (portal == null || depth > 32) {
            return null;
        }
        PortalDestination dest = portal.getDestination();
        if (dest == null) {
            return null;
        }
        if (dest.getType() == PortalDestination.Type.PORTAL) {
            String linkedName = dest.getLinkedPortalName();
            if (linkedName == null) {
                return null;
            }
            String currentName = portal.getName();
            if (!visited.add(currentName.toLowerCase())) {
                return null;
            }
            if (currentName.equalsIgnoreCase(linkedName)) {
                return null;
            }
            Portal linkedPortal = this.getPortal(linkedName);
            if (linkedPortal == null) {
                return null;
            }
            return this.resolveDestination(linkedPortal, visited, depth + 1);
        }
        return dest.getLocation();
    }

    @Override
    public boolean canUsePortal(@NotNull Player player, @NotNull Portal portal) {
        if (!portal.isEnabled()) {
            return false;
        }
        String permission = portal.getPermission();
        if (permission != null && !permission.isEmpty()) {
            return player.hasPermission(permission);
        }
        return true;
    }

    @Override
    public long getRemainingCooldown(@NotNull Player player, @NotNull Portal portal) {
        int cooldownSeconds = portal.getCooldownSeconds();
        if (cooldownSeconds <= 0) {
            return 0L;
        }
        Map<String, Long> cooldowns = this.playerCooldowns.get(player.getUniqueId());
        if (cooldowns == null) {
            return 0L;
        }
        Long lastUse = cooldowns.get(portal.getName().toLowerCase());
        if (lastUse == null) {
            return 0L;
        }
        long elapsed = System.currentTimeMillis() - lastUse;
        long remaining = (long)cooldownSeconds * 1000L - elapsed;
        return Math.max(0L, (remaining + 999L) / 1000L);
    }

    @Override
    public void spawnPortalEffects(@NotNull Portal portal) {
        if (!portal.isShowParticles()) {
            return;
        }
        PortalRegion region = portal.getRegion();
        if (region == null || region.getWorld() == null) {
            return;
        }
        Location center = region.getCenter();
        if (center == null) {
            return;
        }
        center.getWorld().spawnParticle(portal.getParticle(), center, 20, 1.0, 1.0, 1.0, 0.1);
    }

    public void cancelPendingTeleport(UUID playerUUID) {
        this.pendingTeleports.remove(playerUUID);
    }

    public boolean hasPendingTeleport(UUID playerUUID) {
        return this.pendingTeleports.containsKey(playerUUID);
    }
}

