package cc.thonly.reverie_dreams.server;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.mojang.serialization.Codec;
import com.mojang.serialization.DataResult;
import com.mojang.serialization.JsonOps;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.objects.ObjectOpenHashSet;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_3218;
import net.minecraft.class_5218;
import net.minecraft.class_5321;
import net.minecraft.class_7924;
import net.minecraft.server.MinecraftServer;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.*;
import java.util.stream.Collectors;

@Getter
@Slf4j
public class DreamPillowManager {
    public static final Codec<List<WorldEntry>> CODEC = Codec.list(WorldEntry.CODEC);
    private static final Gson GSON = new Gson();
    private static DreamPillowManager INSTANCE;
    private final MinecraftServer server;
    private final Path savedPath;
    private final Map<class_1937, WorldEntry> data;

    public DreamPillowManager(MinecraftServer server) {
        this.server = server;
        this.savedPath = server.method_27050(class_5218.field_24188).resolve("data/dream_pillow.json");
        this.data = new Object2ObjectOpenHashMap<>();
        INSTANCE = this;
    }

    public void init() {
        for (class_3218 world : this.server.method_3738()) {
            WorldEntry worldEntry = new WorldEntry(world.method_27983(), new ArrayList<>());
            worldEntry.setWorld(world);
            this.data.put(world, worldEntry);
        }
        this.load();
    }

    public WorldEntry get(class_3218 world) {
        return this.data.get(world);
    }

    public void load() {
        try {
            Files.createDirectories(this.savedPath.getParent());
        } catch (Exception err) {
            log.error("Can't create dir", err);
        }
        if (Files.exists(this.savedPath)) {
            try (BufferedReader reader = Files.newBufferedReader(this.savedPath, StandardCharsets.UTF_8)) {
                JsonElement json = GSON.fromJson(reader, JsonElement.class);
                var result = CODEC.parse(JsonOps.INSTANCE, json);
                Optional<List<WorldEntry>> optional = result.result();
                if (optional.isPresent()) {
                    for (WorldEntry worldEntry : optional.get()) {
                        for (Map.Entry<class_1937, WorldEntry> managerEntry : this.data.entrySet()) {
                            final WorldEntry value = managerEntry.getValue();
                            if (value.registryKey.equals(worldEntry.registryKey)) {
                                value.loadData(worldEntry.signLocations);
                                break;
                            }
                        }
                    }
                }
            } catch (IOException ioException) {
                log.error("Can't load dream pillow data", ioException);
            }
        }
    }

    public void save() {
        try {
            Files.createDirectories(this.savedPath.getParent());
            List<WorldEntry> values = this.data.values().stream().toList();
            DataResult<JsonElement> result = CODEC.encodeStart(JsonOps.INSTANCE, values);
            Optional<JsonElement> optional = result.result();
            if (optional.isPresent()) {
                JsonElement element = optional.get();
                String json = GSON.toJson(element);
                try (BufferedWriter writer = Files.newBufferedWriter(this.savedPath, StandardCharsets.UTF_8)) {
                    writer.write(json);
                }
            }
        } catch (IOException ioException) {
            log.error("Can't save dream pillow data", ioException);
        }
    }

    public static Optional<DreamPillowManager> getInstance() {
        return Optional.ofNullable(INSTANCE);
    }

    @Getter
    public static class WorldEntry {
        public static final Codec<WorldEntry> CODEC = RecordCodecBuilder.create(instance -> instance.group(
                class_5321.method_39154(class_7924.field_41223).fieldOf("world").forGetter(WorldEntry::getRegistryKey),
                Codec.list(class_2338.field_25064).fieldOf("sign_locations").forGetter(WorldEntry::getSignLocations)
        ).apply(instance, WorldEntry::new));

        private class_5321<class_1937> registryKey;
        private class_3218 world;
        private final List<class_2338> signLocations;

        public WorldEntry(class_5321<class_1937> registryKey, Collection<class_2338> signLocations) {
            this.registryKey = registryKey;
            this.signLocations = new ArrayList<>(new ObjectOpenHashSet<>(signLocations));
        }

        public void add(class_2338 pos) {
            this.signLocations.add(pos);
        }

        public boolean contains(class_2338 pos) {
            if (pos == null) {
                return false;
            }
            Set<Long> collect = this.signLocations.stream().filter(Objects::nonNull).map(class_2338::method_10063).collect(Collectors.toSet());
            return this.signLocations.contains(pos) || (!collect.isEmpty() && collect.contains(pos.method_10063()));
        }

        public void remove(class_2338 pos) {
            long target = pos.method_10063();
            this.signLocations.removeIf(p -> p.method_10063() == target);
        }

        public void loadData(List<class_2338> signLocations) {
            Set<class_2338> merged = new ObjectOpenHashSet<>(this.signLocations);
            merged.addAll(signLocations);
            this.signLocations.clear();
            this.signLocations.addAll(merged);
        }

        public void setWorld(class_3218 world) {
            if (this.world == null) {
                this.registryKey = world.method_27983();
                this.world = world;
            }
        }
    }
}
