package cc.thonly.reverie_dreams.data.danmaku;

import cc.thonly.reverie_dreams.component.DanmakuProperties;
import cc.thonly.reverie_dreams.data.danmaku.spellcard.SpellCardFrameConfig;
import cc.thonly.reverie_dreams.entity.misc.DanmakuEntity;
import cc.thonly.reverie_dreams.registry.content.component.RDDataComponents;
import com.mojang.serialization.Codec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import lombok.Getter;
import lombok.Setter;
import net.minecraft.class_1297;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_243;
import net.minecraft.class_3218;
import net.minecraft.class_9282;
import net.minecraft.class_9334;
import net.minecraft.server.MinecraftServer;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.concurrent.ThreadLocalRandom;

@Getter
public class SpellcardRenderer {
    public static final Codec<List<List<SpellCardFrameConfig>>> FRAMES_CODEC =
            Codec.list(Codec.list(SpellCardFrameConfig.CODEC));
    public static final Codec<SpellcardRenderer> CODEC = RecordCodecBuilder.create(instance -> instance.group(
            FRAMES_CODEC.fieldOf("frames").forGetter(SpellcardRenderer::getFrames)
    ).apply(instance, SpellcardRenderer::new));

    private static final Set<SpellcardRenderer> TICKER = new LinkedHashSet<>();
    private static final int MAX_SPAWN_PER_TICK = 64;

    @Nullable
    @Setter
    private class_1297 source;
    @Setter
    private class_243 position;
    @Setter
    @Nullable
    private PositionGetter positionGetter;
    @Setter
    private class_3218 world;
    private final List<List<SpellCardFrameConfig>> frames;
    private final Random random = new Random();

    private final Map<DanmakuType, Map<Integer, class_1799>> STACK_CACHED = new Object2ObjectOpenHashMap<>();

    private boolean canceled = false;
    private int tick = 0;
    private int maxTick = 100;

    public SpellcardRenderer(@Nullable class_1297 source, @NotNull class_3218 world, List<List<SpellCardFrameConfig>> frames, int tick, int maxTick) {
        this.source = source;
        this.position = source != null ? source.method_33571() : class_243.field_1353;
        this.world = world;
        this.frames = copyFramesConfig(frames);
        this.tick = tick;
        this.maxTick = maxTick;
    }

    public SpellcardRenderer(@NotNull class_243 position, @NotNull class_3218 world, List<List<SpellCardFrameConfig>> frames, int tick, int maxTick) {
        this.source = null;
        this.position = position;
        this.world = world;
        this.frames = this.copyFramesConfig(frames);
        this.tick = tick;
        this.maxTick = maxTick;
    }

    public SpellcardRenderer(List<List<SpellCardFrameConfig>> frames) {
        this.frames = this.copyFramesConfig(frames);
        this.maxTick = this.searchMaxTick(frames);
    }

    public static SpellcardRenderer addRenderer(SpellcardRenderer renderer) {
        TICKER.add(renderer);
        return renderer;
    }

    public static void tick(MinecraftServer server) {
        Iterator<SpellcardRenderer> iterator = TICKER.iterator();
        while (iterator.hasNext()) {
            SpellcardRenderer renderer = iterator.next();
            renderer.update();
            if (renderer.canceled || renderer.tick > renderer.maxTick) {
                iterator.remove();
                renderer.cleanup();
            }
        }
    }

    private int searchMaxTick(List<List<SpellCardFrameConfig>> frames) {
        int maxTick = 0;
        for (List<SpellCardFrameConfig> frameList : frames) {
            for (SpellCardFrameConfig config : frameList) {
                int endTick = config.getTickDelay() + config.getTickDuration();
                if (endTick > maxTick) {
                    maxTick = endTick;
                }
            }
        }
        return maxTick;
    }

    private List<List<SpellCardFrameConfig>> copyFramesConfig(List<List<SpellCardFrameConfig>> frames) {
        List<List<SpellCardFrameConfig>> copy = new ArrayList<>();
        for (List<SpellCardFrameConfig> frame : frames) {
            List<SpellCardFrameConfig> frameCopy = new ArrayList<>();
            for (SpellCardFrameConfig config : frame) {
                frameCopy.add(config.copy());
            }
            copy.add(frameCopy);
        }
        return copy;
    }

    protected void update() {
        if (this.canceled) {
            return;
        }

        this.tick++;
        if (this.tick > this.maxTick) {
            this.cancel();
            return;
        }
        if (this.source != null && this.source.method_31481()) {
            this.cancel();
            return;
        }

        int spawned = 0;

        for (List<SpellCardFrameConfig> frame : this.frames) {
            for (SpellCardFrameConfig config : frame) {
                int relativeTick = this.tick - config.getTickDelay();
                if (relativeTick < 0 || relativeTick > config.getTickDuration()) continue;
                if (relativeTick % config.getTickInterval() != 0) continue;

                float pitchStart = config.getPitchRange().getStart();
                float pitchEnd = config.getPitchRange().getEnd();
                float yawStart = config.getYawRange().getStart();
                float yawEnd = config.getYawRange().getEnd();

                int count = Math.max(1, config.getDensity());
                float pitchStep = (pitchEnd - pitchStart) / count;
                float yawStep = (yawEnd - yawStart) / count;

                for (int i = 0; i < count && spawned < MAX_SPAWN_PER_TICK; i++) {
                    float pitch = pitchStart + pitchStep * i;
                    float yaw = yawStart + yawStep * i;
                    if (config.isRandomColor()) {
                        this.spawnDanmaku(config.getType(), config.getSpeed(), pitch, yaw);
                    } else {
                        this.spawnDanmaku(config.getType(), config.getColor(), pitch, yaw);
                    }
                    spawned++;
                }
                if (spawned >= MAX_SPAWN_PER_TICK) break;
            }
            if (spawned >= MAX_SPAWN_PER_TICK) break;
        }
    }

    private void spawnDanmaku(DanmakuType type, float speed, float pitch, float yaw) {
        int r = 128 + ThreadLocalRandom.current().nextInt(128);
        int g = 128 + ThreadLocalRandom.current().nextInt(128);
        int b = 128 + ThreadLocalRandom.current().nextInt(128);
        int color = (r << 16) | (g << 8) | b;
        class_1799 danmakuStack = getDanmakuStack(type, color);
        spawnDanmaku(danmakuStack, speed, pitch, yaw);
    }

    private void spawnDanmaku(DanmakuType type, float speed, int color, float pitch, float yaw) {
        class_1799 danmakuStack = getDanmakuStack(type, color);
        spawnDanmaku(danmakuStack, speed, pitch, yaw);
    }

    private void spawnDanmaku(class_1799 itemStack, float speed, float pitch, float yaw) {
        DanmakuProperties properties = itemStack.method_58694(RDDataComponents.DANMAKU_PROPERTIES);
        if (properties == null) {
            return;
        }
        properties = properties.withSpeed(speed);
        if (this.source != null) {
            this.position = new class_243(this.source.method_23317(), this.source.method_23320(), this.source.method_23321());
        }
        if (this.position == null && this.positionGetter != null) {
            this.position = this.positionGetter.getPosition(this.tick);
        }
        if (this.position == null) {
            return;
        }

        DanmakuEntity danmakuEntity = new DanmakuEntity(
                this.source,
                this.world,
                this.position.field_1352, this.position.field_1351, this.position.field_1350,
                itemStack,
                properties,
                pitch, yaw,
                0f, 0f,
                false
        );
        this.world.method_8649(danmakuEntity);
    }

    private class_1799 getDanmakuStack(DanmakuType type, int color) {
        Map<Integer, class_1799> colorStackMap = STACK_CACHED.computeIfAbsent(type, t -> new Object2ObjectOpenHashMap<>());

        if (colorStackMap.size() > 64) {
            colorStackMap.clear(); // 防止无限增长
        }

        class_1799 itemStack = colorStackMap.get(color);
        if (itemStack != null) {
            return itemStack.method_7972();
        }

        class_1792 item = type.getItem();
        itemStack = item.method_7854();
        itemStack.method_57379(class_9334.field_49644, new class_9282(color));
        colorStackMap.put(color, itemStack);
        return itemStack;
    }

    public void cancel() {
        this.canceled = true;
        this.cleanup();
    }

    private void cleanup() {
        this.frames.clear();
        this.STACK_CACHED.clear();
        this.source = null;
    }

    public SpellcardRenderer copy() {
        return new SpellcardRenderer(this.frames);
    }
}
