package com.eightsidedsquare.zine.common.registry;

import com.eightsidedsquare.zine.common.recipe.RecipeTypeImpl;
import com.eightsidedsquare.zine.common.text.CustomStyleAttribute;
import com.eightsidedsquare.zine.common.text.TextUtil;
import com.eightsidedsquare.zine.common.text.TextUtilImpl;
import com.eightsidedsquare.zine.common.util.codec.RegistryCodecGroup;
import com.eightsidedsquare.zine.core.ZineRegistries;
import com.google.common.collect.ImmutableSet;
import com.mojang.brigadier.arguments.ArgumentType;
import com.mojang.serialization.Codec;
import com.mojang.serialization.MapCodec;
import net.fabricmc.fabric.api.command.v2.ArgumentTypeRegistry;
import net.fabricmc.fabric.api.itemgroup.v1.FabricItemGroup;
import net.fabricmc.fabric.api.object.builder.v1.block.entity.FabricBlockEntityTypeBuilder;
import net.fabricmc.fabric.api.object.builder.v1.block.type.BlockSetTypeBuilder;
import net.fabricmc.fabric.api.object.builder.v1.block.type.WoodTypeBuilder;
import net.fabricmc.fabric.api.object.builder.v1.entity.FabricTrackedDataRegistry;
import net.fabricmc.fabric.api.particle.v1.FabricParticleTypes;
import net.minecraft.*;
import net.minecraft.block.*;
import net.minecraft.recipe.*;
import org.jetbrains.annotations.Nullable;

import java.util.Optional;
import java.util.Set;
import java.util.function.*;

/**
 * A helper class to streamline object registration.
 *
 * <p>Use {@link #create(String)} to instantiate a {@link RegistryHelper} for a given namespace.
 * For best practice, store the registry helper as a static constant that can be accessed across different index classes.
 *
 * <p>If your mod or your mod's dependencies add new registries,
 * an implementation of this class or {@link RegistryHelperImpl} can be made, with methods to support them.
 */
public interface RegistryHelper {

    /**
     * Creates an instance of the default implementation of {@link RegistryHelper}.
     */
    static RegistryHelper create(String namespace) {
        return new RegistryHelperImpl(namespace);
    }

    /**
     * @param name the Identifier's path
     * @return an {@link class_2960} with the {@link RegistryHelper}'s namespace
     */
    class_2960 id(String name);

    /**
     * @param registryKey the registry key of the registry in the root registry
     * @param name the path of the registry key's value
     * @return the registry key for a value of the registry specified by {@code registryKey}.
     * @param <T> the type of the registry key
     */
    default <T> class_5321<T> key(class_5321<? extends class_2378<T>> registryKey, String name) {
        return class_5321.method_29179(registryKey, this.id(name));
    }

    /**
     * Registers a value to the given registry.
     * @param registry the static registry to register {@code value} to
     * @param name the path of the registered value's identifier
     * @param value the value to register
     * @return {@code value}, now registered
     */
    default <T, V extends T> V register(class_2378<T> registry, String name, V value) {
        return class_2378.method_10230(registry, this.id(name), value);
    }

    /**
     * Registers a value to the given registry.
     * @param registry the static registry to register {@code value} to
     * @param name the path of the registered value's identifier
     * @param value the value to register
     * @return registered {@code value} wrapped in a {@link class_6880.class_6883}
     */
    default <T> class_6880.class_6883<T> registerReference(class_2378<T> registry, String name, T value) {
        return class_2378.method_47985(registry, this.id(name), value);
    }

    /**
     * Creates a registry queue. See {@link RegistryQueue} for more info.
     * @param registry the registry that the registry queue uses
     * @return the registry queue
     * @param <T> the type of the registry
     */
    default <T> RegistryQueue<T> createQueue(class_2378<T> registry) {
        return new RegistryQueueImpl<>(registry, this);
    }

    /**
     * @param name the name of the item
     * @param item the item to register
     * @return the registered item
     * @param <T> the type of the item
     * @apiNote Prioritize the other {@code item} methods over this one,
     * as they handle passing a required {@link class_5321} to the {@link class_1792.class_1793}.
     */
    default <T extends class_1792> T item(String name, T item) {
        return this.register(class_7923.field_41178, name, item);
    }

    /**
     *
     * @param name the name of the item
     * @param settings the item's settings
     * @param factory the factory to instantiate an {@link class_1792} of type {@code <T>} with the given settings
     * @return the registered item created by the {@code factory} with {@code settings}
     * @param <T> the type of the item
     */
    default <T extends class_1792> T item(String name, class_1792.class_1793 settings, Function<class_1792.class_1793, T> factory) {
        return this.item(name, factory.apply(settings.method_63686(this.key(class_7924.field_41197, name))));
    }

    /**
     * Registers an item of type {@link class_1792}.
     * @param name the name of the item
     * @param settings the item's settings
     * @return the registered item with {@code settings}
     */
    default class_1792 item(String name, class_1792.class_1793 settings) {
        return this.item(name, settings, class_1792::new);
    }

    /**
     * Registers an item of type {@link class_1792} with the default item settings.
     * @param name the name of the item
     * @return the registered item
     */
    default class_1792 item(String name) {
        return this.item(name, new class_1792.class_1793());
    }

    /**
     * Registers a block item.
     * @param name the name of the item
     * @param block the block that the block item will place
     * @param settings the item's settings
     * @param factory the factory to instantiate a {@link class_1747} of type {@code <T>} with the given settings
     * @return the registered block item created by the {@code factory} with {@code settings}
     * @param <T> the type of the block item
     */
    default <T extends class_1747> T item(String name, class_2248 block, class_1792.class_1793 settings, BiFunction<class_2248, class_1792.class_1793, T> factory) {
        return this.item(name, settings.method_63685(), itemSettings -> factory.apply(block, itemSettings));
    }

    /**
     * Registers an item of type {@link class_1747}.
     * @param name the name of the item
     * @param block the block that the block item will place
     * @return the registered block item
     */
    default class_1747 item(String name, class_2248 block) {
        return this.item(name, block, new class_1792.class_1793(), class_1747::new);
    }

    /**
     * Registers an item of type {@link class_1826}.
     * @param name the name of the item
     * @param entityType the entity type to be spawned by the spawn egg
     * @return the registered spawn egg item
     */
    default class_1826 item(String name, class_1299<?> entityType) {
        return this.item(name, new class_1792.class_1793().method_72499(entityType), class_1826::new);
    }

    /**
     *
     * @param name the name of the block
     * @param block the block to register
     * @return the registered block
     * @param <T> the type of the block
     * @apiNote Prioritize the other {@code block} methods over this one,
     * as they handle passing a required {@link class_5321} to the {@link class_4970.class_2251}.
     */
    default <T extends class_2248> T block(String name, T block) {
        return this.register(class_7923.field_41175, name, block);
    }

    /**
     * @param name the name of the block
     * @param settings the block's settings
     * @param factory the factory to instantiate a {@link class_2248} of type {@code <T>} with the given settings
     * @return the registered block created by the {@code factory} with {@code settings}
     * @param <T> the type of the block
     */
    default <T extends class_2248> T block(String name, class_4970.class_2251 settings, Function<class_4970.class_2251, T> factory) {
        return this.block(name, factory.apply(settings.method_63500(this.key(class_7924.field_41254, name))));
    }

    /**
     * Registers a block of type {@link class_2248}.
     * @param name the name of the block
     * @param settings the block's settings
     * @return the registered block with {@code settings}
     */
    default class_2248 block(String name, class_4970.class_2251 settings) {
        return this.block(name, settings, class_2248::new);
    }

    /**
     * Registers a {@link class_1747} alongside the block.
     * Use {@link class_2248#method_8389()} to get the instance of the item.
     * @param name the name of the block and item
     * @param block the block to register, and the block that the block item will place
     * @return the registered block
     * @param <T> the type of the block
     * @apiNote Prioritize the other {@code blockWithItem} methods over this one,
     * as they handle passing a required {@link class_5321} to the {@link class_4970.class_2251}.
     */
    default <T extends class_2248> T blockWithItem(String name, T block) {
        return this.registerBlockItem(name, this.block(name, block));
    }

    /**
     * Registers a {@link class_1747} alongside a block.
     * Use {@link class_2248#method_8389()} to get the instance of the item.
     * @param name the name of the block and item
     * @param settings the block's settings
     * @param factory the factory to instantiate a {@link class_2248} of type {@code <T>} with the given settings
     * @return the registered block created by the {@code factory} with {@code settings}
     * @param <T> the type of the block
     */
    default <T extends class_2248> T blockWithItem(String name, class_4970.class_2251 settings, Function<class_4970.class_2251, T> factory) {
        return this.registerBlockItem(name, this.block(name, settings, factory));
    }

    /**
     * Registers a {@link class_1747} alongside a block.
     * Use {@link class_2248#method_8389()} to get the instance of the item.
     * @param name the name of the block and item
     * @param settings the block's settings
     * @return the registered block with {@code settings}
     */
    default class_2248 blockWithItem(String name, class_4970.class_2251 settings) {
        return this.registerBlockItem(name, this.block(name, settings));
    }

    /**
     * Registers an entity type given an entity type builder. Use {@link class_1299.class_1300#method_5902(class_1311)},
     * {@link class_1299.class_1300#method_5903(class_1299.class_4049, class_1311)},
     * {@link class_1299.class_1300#createLiving(EntityType.EntityFactory, SpawnGroup, UnaryOperator)},
     * or {@link class_1299.class_1300#createMob(EntityType.EntityFactory, SpawnGroup, UnaryOperator)}
     * to create a builder.
     * @param name the name of the entity type
     * @param builder the entity type builder
     * @return the built and registered entity type
     * @param <T> the type of the entity
     */
    default <T extends class_1297> class_1299<T> entity(String name, class_1299.class_1300<T> builder) {
        return this.register(class_7923.field_41177, name, builder.method_5905(this.key(class_7924.field_41266, name)));
    }

    /**
     * Registers a block entity type given a block entity type builder.
     * Use {@link FabricBlockEntityTypeBuilder#create(FabricBlockEntityTypeBuilder.Factory, class_2248...)}
     * to create a builder.
     * @param name the name of the block entity type
     * @param builder the block entity type builder
     * @return the built and registered block entity type
     * @param <T> the type of the block entity
     */
    default <T extends class_2586> class_2591<T> blockEntity(String name, FabricBlockEntityTypeBuilder<T> builder) {
        return this.register(class_7923.field_41181, name, builder.build());
    }

    /**
     * @param name the name of the sound event
     * @param soundEvent the sound event to register
     * @return the registered sound event
     */
    default class_3414 sound(String name, class_3414 soundEvent) {
        return this.register(class_7923.field_41172, name, soundEvent);
    }

    /**
     * Registers a {@link class_3414} without a specified range.
     * @param name the name of the sound event
     * @return the registered sound event
     */
    default class_3414 sound(String name) {
        return this.sound(name, class_3414.method_47908(this.id(name)));
    }

    /**
     * Registers a {@link class_3414} wrapped in a {@link class_6880.class_6883}.
     * @param name the name of the sound event
     * @param soundEvent the sound event to register
     * @return the registered sound event
     */
    default class_6880.class_6883<class_3414> soundReference(String name, class_3414 soundEvent) {
        return this.registerReference(class_7923.field_41172, name, soundEvent);
    }

    /**
     * Registers a {@link class_3414} wrapped in a {@link class_6880.class_6883} without a specified range.
     * @param name the name of the sound event
     * @return the registered sound event
     */
    default class_6880.class_6883<class_3414> soundReference(String name) {
        return this.soundReference(name, class_3414.method_47908(this.id(name)));
    }

    /**
     * Creates a {@link class_2498} with its break, step, place, hit, and fall sounds automatically registered.
     * These individual sounds can be accessed with {@link class_2498#method_10595()}, {@link class_2498#method_10594()}, etc.
     * @param name the base name of the sound events of the block sound group
     * @param volume the volume of the block sound group's sounds
     * @param pitch the pitch of the block sound group's sounds
     * @return the block sound group, containing all registered sound events
     */
    default class_2498 blockSoundGroup(String name, float volume, float pitch) {
        return new class_2498(
                volume,
                pitch,
                this.sound("block." + name + ".break"),
                this.sound("block." + name + ".step"),
                this.sound("block." + name + ".place"),
                this.sound("block." + name + ".hit"),
                this.sound("block." + name + ".fall")
        );
    }

    /**
     * Registers a data component type, which is used by items and block entities.
     * @param name the name of the component type
     * @param componentType the component type to register
     * @return the registered component type
     * @param <T> the type of the component type
     */
    default <T> class_9331<T> dataComponent(String name, class_9331<T> componentType) {
        return this.register(class_7923.field_49658, name, componentType);
    }

    /**
     * Registers a data component type, which is used by items and block entities.
     * Use {@link class_9331#method_57873()} to create a builder.
     * @param name the name of the component type
     * @param builder the component type's builder
     * @return the registered and built component type
     * @param <T> the type of the component type
     */
    default <T> class_9331<T> dataComponent(String name, class_9331.class_9332<T> builder) {
        return this.dataComponent(name, builder.method_57880());
    }

    /**
     * Registers a data component type, which is used by items and block entities.
     * @param name the name of the component type
     * @param builderOperator the operator that configures the component type builder
     * @return the registered and built component type
     * @param <T> the type of the component type
     */
    default <T> class_9331<T> dataComponent(String name, UnaryOperator<class_9331.class_9332<T>> builderOperator) {
        return this.dataComponent(name, builderOperator.apply(class_9331.method_57873()));
    }

    /**
     * Registers a registry entry data component type, which is used by items and block entities,
     * based on the entry codec and packet codec of a {@link RegistryCodecGroup}.
     * @param name the name of the component type
     * @param registryCodecGroup the registry codec group which provides the codec and packet codec for the component type
     * @return the registered and built component type
     * @param <T> the registry entry type of the component type
     */
    default <T> class_9331<class_6880<T>> dataComponent(String name, RegistryCodecGroup<T> registryCodecGroup) {
        return this.dataComponent(name, builder ->
                builder.method_57881(registryCodecGroup.entryCodec()).method_57882(registryCodecGroup.packetCodec()).method_59871()
        );
    }

    /**
     * Registers an enchantment component type, which is used by enchantments.
     * @param name the name of the component type
     * @param componentType the component type to register
     * @return the registered component type
     * @param <T> the type of the component type
     */
    default <T> class_9331<T> enchantmentComponent(String name, class_9331<T> componentType) {
        return this.register(class_7923.field_51832, name, componentType);
    }

    /**
     * Registers an enchantment component type, which is used by enchantments.
     * Use {@link class_9331#method_57873()} to create a builder.
     * @param name the name of the component type
     * @param builder the component type's builder
     * @return the registered and built component type
     * @param <T> the type of the component type
     */
    default <T> class_9331<T> enchantmentComponent(String name, class_9331.class_9332<T> builder) {
        return this.enchantmentComponent(name, builder.method_57880());
    }

    /**
     * Registers an enchantment component type, which is used by enchantments.
     * @param name the name of the component type
     * @param builderOperator the operator that configures the component type builder
     * @return the registered and built component type
     * @param <T> the type of the component type
     */
    default <T> class_9331<T> enchantmentComponent(String name, UnaryOperator<class_9331.class_9332<T>> builderOperator) {
        return this.enchantmentComponent(name, builderOperator.apply(class_9331.method_57873()));
    }

    /**
     * @param name the name of the particle type
     * @param particleType the particle type to register
     * @return the registered particle type
     * @param <E> the type of the particle type's particle effect
     * @param <T> the type of the particle type
     */
    default <E extends class_2394, T extends class_2396<E>> T particle(String name, T particleType) {
        return this.register(class_7923.field_41180, name, particleType);
    }

    /**
     * Registers a simple particle type, which has an effect with no parameters.
     * @param name the name of the particle type
     * @param alwaysSpawn {@code true} to always spawn the particle regardless of distance.
     * @return the registered particle type
     */
    default class_2400 particle(String name, boolean alwaysSpawn) {
        return this.particle(name, FabricParticleTypes.simple(alwaysSpawn));
    }

    /**
     * Registers a simple particle type, which has an effect with no parameters.
     * This particle will not spawn depending on distance.
     * @param name the name of the particle type
     * @return the registered particle type
     */
    default class_2400 particle(String name) {
        return this.particle(name, false);
    }

    /**
     * Registers a complex particle type, which typically has an effect with parameters.
     * @param name the name of the particle type
     * @param alwaysSpawn {@code true} to always spawn the particle regardless of distance.
     * @param codec the codec for serialization
     * @param packetCodec the packet codec for network serialization
     * @return the registered particle type
     * @param <T> the type of the particle type's particle effect
     */
    default <T extends class_2394> class_2396<T> particle(String name, boolean alwaysSpawn, MapCodec<T> codec, class_9139<? super class_9129, T> packetCodec) {
        return this.particle(name, FabricParticleTypes.complex(alwaysSpawn, codec, packetCodec));
    }

    /**
     * Registers a complex particle type, which typically has an effect with parameters.
     * This particle will not spawn depending on distance.
     * @param name the name of the particle type
     * @param codec the codec for serialization
     * @param packetCodec the packet codec for network serialization
     * @return the registered particle type
     * @param <T> the type of the particle type's particle effect
     */
    default <T extends class_2394> class_2396<T> particle(String name, MapCodec<T> codec, class_9139<? super class_9129, T> packetCodec) {
        return this.particle(name, false, codec, packetCodec);
    }

    /**
     * Registers a complex particle type, which typically has an effect with parameters.
     * @param name the name of the particle type
     * @param alwaysSpawn {@code true} to always spawn the particle regardless of distance.
     * @param codecGetter a function which, given the particle type, returns the codec for serialization
     * @param packetCodecGetter a function which, given the particle type, returns the packet codec for network serialization
     * @return the registered particle type
     * @param <T> the type of the particle type's particle effect
     */
    default <T extends class_2394> class_2396<T> particle(String name, boolean alwaysSpawn, Function<class_2396<T>, MapCodec<T>> codecGetter, Function<class_2396<T>, class_9139<? super class_9129, T>> packetCodecGetter) {
        return this.particle(name, FabricParticleTypes.complex(alwaysSpawn, codecGetter, packetCodecGetter));
    }

    /**
     * Registers a complex particle type, which typically has an effect with parameters.
     * This particle will not spawn depending on distance.
     * @param name the name of the particle type
     * @param codecGetter a function which, given the particle type, returns the codec for serialization
     * @param packetCodecGetter a function which, given the particle type, returns the packet codec for network serialization
     * @return the registered particle type
     * @param <T> the type of the particle type's particle effect
     */
    default <T extends class_2394> class_2396<T> particle(String name, Function<class_2396<T>, MapCodec<T>> codecGetter, Function<class_2396<T>, class_9139<? super class_9129, T>> packetCodecGetter) {
        return this.particle(name, false, codecGetter, packetCodecGetter);
    }

    /**
     * @param name the name of the item group
     * @param itemGroup the item group to register
     * @return the registered item group
     */
    default class_1761 itemGroup(String name, class_1761 itemGroup) {
        return this.register(class_7923.field_44687, name, itemGroup);
    }

    /**
     * Use {@link FabricItemGroup#builder()}
     * @param name the name of the item group
     * @param builder the builder of the item group to register
     * @return the registered and built item group
     */
    default class_1761 itemGroup(String name, class_1761.class_7913 builder) {
        return this.itemGroup(name, builder.method_47324());
    }

    /**
     * Registers a {@link class_5712} wrapped in a {@link class_6880.class_6883}.
     * @param name the name of the game event
     * @param gameEvent the game event to register
     * @return the registered game event
     */
    default class_6880.class_6883<class_5712> gameEvent(String name, class_5712 gameEvent) {
        return this.registerReference(class_7923.field_41171, name, gameEvent);
    }

    /**
     * Registers a {@link class_5712} wrapped in a {@link class_6880.class_6883}.
     * @param name the name of the game event
     * @param radius the radius of the game event
     * @return the registered game event
     */
    default class_6880.class_6883<class_5712> gameEvent(String name, int radius) {
        return this.gameEvent(name, new class_5712(radius));
    }

    /**
     * Registers a {@link class_5712} wrapped in a {@link class_6880.class_6883} with a default radius of 16.
     * @param name the name of the game event
     * @return the registered game event
     */
    default class_6880.class_6883<class_5712> gameEvent(String name) {
        return this.gameEvent(name, 16);
    }

    /**
     * @param name the name of the fluid
     * @param fluid the fluid to register
     * @return the registered fluid
     * @param <T> the type of the fluid
     */
    default <T extends class_3611> T fluid(String name, T fluid) {
        return this.register(class_7923.field_41173, name, fluid);
    }

    /**
     * Registers a {@link class_1291} wrapped in a {@link class_6880.class_6883}.
     * @param name the name of the status effect
     * @param statusEffect the status effect to register
     * @return the registered status effect
     */
    default class_6880.class_6883<class_1291> statusEffect(String name, class_1291 statusEffect) {
        return this.registerReference(class_7923.field_41174, name, statusEffect);
    }

    /**
     * Registers a {@link class_1291} wrapped in a {@link class_6880.class_6883}.
     * @param name the name of the status effect
     * @param category the category of the status effect
     * @param color the color of the status effect in RGB format
     * @return the registered status effect
     */
    default class_6880.class_6883<class_1291> statusEffect(String name, class_4081 category, int color) {
        return this.statusEffect(name, new class_1291(category, color));
    }

    /**
     * Registers a {@link class_1291} wrapped in a {@link class_6880.class_6883}.
     * @param name the name of the status effect
     * @param category the category of the status effect
     * @param color the color of the status effect in RGB format
     * @param particleEffect the particle spawned by the status effect
     * @return the registered status effect
     */
    default class_6880.class_6883<class_1291> statusEffect(String name, class_4081 category, int color, class_2394 particleEffect) {
        return this.statusEffect(name, new class_1291(category, color, particleEffect));
    }

    /**
     * Registers a {@link class_1291} wrapped in a {@link class_6880.class_6883}.
     * @param name the name of the status effect
     * @param category the category of the status effect
     * @param color the color of the status effect in RGB format
     * @param statusEffectOperator operator to apply changes to the status effect after registration
     * @return the registered status effect
     */
    default class_6880.class_6883<class_1291> statusEffect(String name, class_4081 category, int color, UnaryOperator<class_1291> statusEffectOperator) {
        return this.statusEffect(name, statusEffectOperator.apply(new class_1291(category, color)));
    }

    /**
     * Registers a {@link class_1291} wrapped in a {@link class_6880.class_6883}.
     * @param name the name of the status effect
     * @param category the category of the status effect
     * @param color the color of the status effect in RGB format
     * @param particleEffect the particle spawned by the status effect
     * @param statusEffectOperator operator to apply changes to the status effect after registration
     * @return the registered status effect
     */
    default class_6880.class_6883<class_1291> statusEffect(String name, class_4081 category, int color, class_2394 particleEffect, UnaryOperator<class_1291> statusEffectOperator) {
        return this.statusEffect(name, statusEffectOperator.apply(new class_1291(category, color, particleEffect)));
    }

    /**
     * Registers a {@link class_1842} wrapped in a {@link class_6880.class_6883}.
     * @param name the name of the potion
     * @param potion the potion to register
     * @return the registered potion
     */
    default class_6880.class_6883<class_1842> potion(String name, class_1842 potion) {
        return this.registerReference(class_7923.field_41179, name, potion);
    }

    /**
     * Registers a {@link class_1842} wrapped in a {@link class_6880.class_6883}.
     * @param name the name of the potion
     * @param statusEffectInstances an array of status effect instances that the potion contains
     * @return the registered potion
     * @apiNote This method sets the base name of the potion to {@code name}.
     * Use {@link #potion(String, class_1842)} to register "long" and "strong" variants of a potion,
     * since these should use the original potion's base name.
     */
    default class_6880.class_6883<class_1842> potion(String name, class_1293... statusEffectInstances) {
        return this.potion(name, new class_1842(name, statusEffectInstances));
    }

    /**
     * @param name the name of the custom stat
     * @param formatter the formatting of the custom stat
     * @return the registered custom stat
     */
    default class_2960 customStat(String name, class_3446 formatter) {
        class_2960 stat = this.register(class_7923.field_41183, name, this.id(name));
        class_3468.field_15419.method_14955(stat, formatter);
        return stat;
    }

    /**
     * Registers a custom stat with default formatting
     * @param name the name of the custom stat
     * @return the registered custom stat
     */
    default class_2960 customStat(String name) {
        return this.customStat(name, class_3446.field_16975);
    }

    /**
     * @param name the name of the stat type
     * @param statType the stat type to register
     * @return the registered stat type
     * @param <T> the type of stat
     */
    default <T> class_3448<T> stat(String name, class_3448<T> statType) {
        return this.register(class_7923.field_41193, name, statType);
    }

    /**
     * @param name the name of the stat type
     * @param registry the stat type's registry
     * @return the registered stat type
     * @param <T> the type of stat
     */
    default <T> class_3448<T> stat(String name, class_2378<T> registry) {
        class_2561 text = class_2561.method_43471(class_156.method_646("stat_type", this.id(name)));
        return this.stat(name, new class_3448<>(registry, text));
    }

    /**
     * @param name the name of the rule test type
     * @param type the rule test type to register
     * @return the registered rule test type
     * @param <T> the type of rule test
     */
    default <T extends class_3825> class_3827<T> ruleTest(String name, class_3827<T> type) {
        return this.register(class_7923.field_41185, name, type);
    }

    /**
     * @param name the name of the rule test type
     * @param codec the codec of the rule test
     * @return the registered rule test type
     * @param <T> the type of the rule test
     */
    default <T extends class_3825> class_3827<T> ruleTest(String name, MapCodec<T> codec) {
        return this.ruleTest(name, () -> codec);
    }

    /**
     * @param name the name of the rule block entity modifier type
     * @param type the rule block entity modifier type to register
     * @return the registered rule block entity modifier type
     * @param <T> the type of the rule block entity modifier
     */
    default <T extends class_8248> class_8249<T> ruleBlockEntityModifier(String name, class_8249<T> type) {
        return this.register(class_7923.field_43381, name, type);
    }

    /**
     * @param name the name of the rule block entity modifier type
     * @param codec the codec of the rule block entity modifier
     * @return the registered rule block entity modifier type
     * @param <T> the type of the rule block entity modifier
     */
    default <T extends class_8248> class_8249<T> ruleBlockEntityModifier(String name, MapCodec<T> codec) {
        return this.ruleBlockEntityModifier(name, () -> codec);
    }

    /**
     * @param name the name of the pos rule test type
     * @param type the pos rule test type to register
     * @return the registered pos rule test type
     * @param <T> the type of pos rule test
     */
    default <T extends class_4995> class_4996<T> posRuleTest(String name, class_4996<T> type) {
        return this.register(class_7923.field_41186, name, type);
    }

    /**
     * @param name the name of the pos rule test type
     * @param codec the codec of the pos rule test
     * @return the registered pos rule test type
     * @param <T> the type of the pos rule test
     */
    default <T extends class_4995> class_4996<T> posRuleTest(String name, MapCodec<T> codec) {
        return this.posRuleTest(name, () -> codec);
    }

    /**
     * @param name the name of the screen handler type
     * @param type the screen handler type to register
     * @return the registered screen handler type
     * @param <T> the type of the screen handler
     */
    default <T extends class_1703> class_3917<T> screenHandler(String name, class_3917<T> type) {
        return this.register(class_7923.field_41187, name, type);
    }

    /**
     * Registers a {@link class_3917} with {@link class_7701#field_40182} feature flags.
     * @param name the name of the screen handler type
     * @param factory the screen handler type's factory
     * @return the registered screen handler type
     * @param <T> the type of the screen handler
     */
    default <T extends class_1703> class_3917<T> screenHandler(String name, class_3917.class_3918<T> factory) {
        return this.screenHandler(name, new class_3917<>(factory, class_7701.field_40182));
    }

    /**
     * @param name the name of the recipe type
     * @return the registered recipe type
     * @param <T> the type of recipe
     */
    default <T extends class_1860<?>> class_3956<T> recipeType(String name) {
        return this.register(class_7923.field_41188, name, new RecipeTypeImpl<>(name));
    }

    /**
     * @param name the name of the recipe serializer
     * @param serializer the
     * @return the registered recipe serializer
     * @param <T> the type of recipe
     * @param <S> the type of the recipe serializer
     */
    default <T extends class_1860<?>, S extends class_1865<T>> S recipeSerializer(String name, S serializer) {
        return this.register(class_7923.field_41189, name, serializer);
    }

    /**
     * @param name the name of the recipe serializer
     * @param factory the factory for a special crafting recipe
     * @return the registered special recipe serializer
     * @param <T> the type of crafting recipe
     */
    default <T extends class_3955> class_1852.class_1866<T> recipeSerializer(String name, class_1852.class_1866.class_7711<T> factory) {
        return this.recipeSerializer(name, new class_1852.class_1866<>(factory));
    }

    /**
     * @param name the name of the recipe serializer
     * @param factory the factory for a cooking recipe
     * @param defaultCookingTime the default cooking time of the recipe serializer
     * @return the registered cooking recipe serializer
     * @param <T> the type of cooking recipe
     */
    default <T extends class_1874> class_1874.class_10285<T> recipeSerializer(String name, class_1874.class_3958<T> factory, int defaultCookingTime) {
        return this.recipeSerializer(name, new class_1874.class_10285<>(factory, defaultCookingTime));
    }

    /**
     * @param name the name of the entity attribute
     * @param attribute the entity attribute to register
     * @return the registered entity attribute
     */
    default class_1320 attribute(String name, class_1320 attribute) {
        return this.register(class_7923.field_41190, name, attribute);
    }

    /**
     * @param name the name of the position source type
     * @param type the position source type to register
     * @return the registered position source type
     * @param <S> the type of position source
     * @param <T> the type of the position source type
     */
    default <S extends class_5716, T extends class_5717<S>> T positionSource(String name, T type) {
        return this.register(class_7923.field_41191, name, type);
    }

    /**
     * @param name the name of the argument serializer
     * @param clazz the class of the argument type
     * @param serializer the argument serializer to register
     * @return the registered argument serializer
     * @param <A> the type of argument type
     * @param <T> the type of argument type properties
     * @param <S> the type of the argument serializer
     */
    default <A extends ArgumentType<?>, T extends class_2314.class_7217<A>, S extends class_2314<A, T>> S argumentSerializer(String name, Class<? extends A> clazz, S serializer) {
        ArgumentTypeRegistry.registerArgumentType(this.id(name), clazz, serializer);
        return serializer;
    }

    /**
     * Registers a {@link class_3854} with namespaced {@code name} as its name.
     * @param name the name of the villager type
     * @return the registered villager type
     */
    default class_3854 villagerType(String name) {
        return this.register(class_7923.field_41194, name, new class_3854());
    }

    /**
     * @param name the name of the villager profession
     * @param profession the villager profession to register
     * @return the registered villager profession
     */
    default class_3852 villagerProfession(String name, class_3852 profession) {
        return this.register(class_7923.field_41195, name, profession);
    }

    /**
     * @param name the name of the villager profession
     * @param heldWorkstation the predicate that returns {@code true} if a point of interest is a held workstation
     * @param acquirableWorkstation the predicate that returns {@code true} if a point of interest is an acquirable workstation
     * @param gatherableItems the immutable set of items that the villager type can gather
     * @param secondaryJobSites the immutable set of secondary job sites
     * @param workSound the sound event played when the villager works, or null
     * @return the registered villager profession
     */
    default class_3852 villagerProfession(String name,
                                                  Predicate<class_6880<class_4158>> heldWorkstation,
                                                  Predicate<class_6880<class_4158>> acquirableWorkstation,
                                                  ImmutableSet<class_1792> gatherableItems,
                                                  ImmutableSet<class_2248> secondaryJobSites,
                                                  @Nullable class_3414 workSound) {
        class_2960 id = this.id(name);
        class_2561 text = class_2561.method_43471("entity." + id.method_12836() + ".villager." + id.method_12832());
        return this.villagerProfession(name, new class_3852(text, heldWorkstation, acquirableWorkstation, gatherableItems, secondaryJobSites, workSound));
    }

    /**
     * @param name the name of the villager profession
     * @param heldWorkstation the registry key of the held workstation point of interest type
     * @param gatherableItems the immutable set of items that the villager type can gather
     * @param secondaryJobSites the immutable set of secondary job sites
     * @param workSound the sound event played when the villager works, or null
     * @return the registered villager profession
     */
    default class_3852 villagerProfession(String name,
                                                  class_5321<class_4158> heldWorkstation,
                                                  ImmutableSet<class_1792> gatherableItems,
                                                  ImmutableSet<class_2248> secondaryJobSites,
                                                  @Nullable class_3414 workSound) {
        Predicate<class_6880<class_4158>> predicate = entry -> entry.method_40225(heldWorkstation);
        return this.villagerProfession(name, predicate, predicate, gatherableItems, secondaryJobSites, workSound);
    }

    /**
     * @param name the name of the villager profession
     * @param heldWorkstation the predicate that returns {@code true} if a point of interest is a held workstation
     * @param acquirableWorkstation the predicate that returns {@code true} if a point of interest is an acquirable workstation
     * @param workSound the sound event played when the villager works, or null
     * @return the registered villager profession
     */
    default class_3852 villagerProfession(String name,
                                                  Predicate<class_6880<class_4158>> heldWorkstation,
                                                  Predicate<class_6880<class_4158>> acquirableWorkstation,
                                                  @Nullable class_3414 workSound) {
        return this.villagerProfession(name, heldWorkstation, acquirableWorkstation, ImmutableSet.of(), ImmutableSet.of(), workSound);
    }

    /**
     * @param name the name of the villager profession
     * @param heldWorkstation the registry key of the held workstation point of interest type
     * @param workSound the sound event played when the villager works, or null
     * @return the registered villager profession
     */
    default class_3852 villagerProfession(String name, class_5321<class_4158> heldWorkstation, @Nullable class_3414 workSound) {
        Predicate<class_6880<class_4158>> predicate = entry -> entry.method_40225(heldWorkstation);
        return this.villagerProfession(name, predicate, predicate, workSound);
    }

    /**
     * @param name the name of the point of interest type
     * @param type the point of interest type to register
     * @return the registry key for the registered point of interest type
     */
    default class_5321<class_4158> poi(String name, class_4158 type) {
        class_6880<class_4158> entry = this.registerReference(class_7923.field_41128, name, type);
        class_7477.method_43990(entry, type.comp_815());
        return this.key(class_7924.field_41212, name);
    }

    /**
     * @param name the name of the point of interest type
     * @param states the set of block states that constitute the point of interest type
     * @param ticketCount the ticket count of the point of interest type
     * @param searchDistance the search distance of the point of interest type
     * @return the registered point of interest type
     */
    default class_5321<class_4158> poi(String name, Set<class_2680> states, int ticketCount, int searchDistance) {
        return this.poi(name, new class_4158(states, ticketCount, searchDistance));
    }

    /**
     * @param name the name of the point of interest type
     * @param block the block whose block states constitute the point of interest type
     * @param ticketCount the ticket count of the point of interest type
     * @param searchDistance the search distance of the point of interest type
     * @return the registered point of interest type
     */
    default class_5321<class_4158> poi(String name, class_2248 block, int ticketCount, int searchDistance) {
        return this.poi(name, class_7477.method_43988(block), ticketCount, searchDistance);
    }

    /**
     * Registers a {@link class_4158} with a default ticket count and search distance of 1.
     * @param name the name of the point of interest type
     * @param states the set of block states that constitute the point of interest type
     * @return the registered point of interest type
     */
    default class_5321<class_4158> poi(String name, Set<class_2680> states) {
        return this.poi(name, new class_4158(states, 1, 1));
    }

    /**
     * Registers a {@link class_4158} with a default ticket count and search distance of 1.
     * @param name the name of the point of interest type
     * @param block the block whose block states constitute the point of interest type
     * @return the registered point of interest type
     */
    default class_5321<class_4158> poi(String name, class_2248 block) {
        return this.poi(name, class_7477.method_43988(block), 1, 1);
    }

    /**
     * @param name the name of the memory module type
     * @param type the memory module type to register
     * @return the registered memory module type
     * @param <T> the type of the memory module type
     */
    default <T> class_4140<T> memoryModule(String name, class_4140<T> type) {
        return this.register(class_7923.field_41129, name, type);
    }

    /**
     * Registers a non-persistent {@link class_4140}.
     * @param name the name of the memory module type
     * @return the registered memory module type
     * @param <T> the type of the memory module type
     */
    default <T> class_4140<T> memoryModule(String name) {
        return this.memoryModule(name, new class_4140<>(Optional.empty()));
    }

    /**
     * Registers a persistent {@link class_4140}.
     * @param name the name of the memory module type
     * @param codec the codec of the memory module type's value
     * @return the registered memory module type
     * @param <T> the type of the memory module type
     */
    default <T> class_4140<T> memoryModule(String name, Codec<T> codec) {
        return this.memoryModule(name, new class_4140<>(Optional.of(codec)));
    }

    /**
     * @param name the name of the sensor type
     * @param type the sensor type to register
     * @return the registered sensor type
     * @param <T> the type of sensor
     */
    default <T extends class_4148<?>> class_4149<T> sensor(String name, class_4149<T> type) {
        return this.register(class_7923.field_41130, name, type);
    }

    /**
     * @param name the name of the sensor type
     * @param supplier the supplier for an instance of the sensor, usually a method reference to the sensor's constructor
     * @return the registered sensor type
     * @param <T> the type of sensor
     */
    default <T extends class_4148<?>> class_4149<T> sensor(String name, Supplier<T> supplier) {
        return this.sensor(name, new class_4149<>(supplier));
    }

    /**
     * @param name the name of the schedule
     * @param schedule the schedule to register
     * @return the registered schedule
     */
    default class_4170 schedule(String name, class_4170 schedule) {
        return this.register(class_7923.field_41131, name, schedule);
    }

    /**
     * @param name the name of the schedule
     * @param builderOperator operator for building the schedule
     * @return the registered schedule
     */
    default class_4170 schedule(String name, UnaryOperator<class_4171> builderOperator) {
        return this.schedule(name, builderOperator.apply(new class_4171(new class_4170())).method_19220());
    }

    /**
     * @param name the name of the activity
     * @param activity the activity to register
     * @return the registered activity
     */
    default class_4168 activity(String name, class_4168 activity) {
        return this.register(class_7923.field_41132, name, activity);
    }

    /**
     * Registers an {@link class_4168} with {@code name} for its name parameter.
     * @param name the name of the activity
     * @return the registered activity
     */
    default class_4168 activity(String name) {
        return this.activity(name, new class_4168(name));
    }

    /**
     * @param name the name of the loot pool entry type
     * @param type the loot pool entry type to register
     * @return the registered loot pool entry type
     */
    default class_5338 lootPoolEntry(String name, class_5338 type) {
        return this.register(class_7923.field_41133, name, type);
    }

    /**
     * @param name the name of the loot pool entry type
     * @param codec the codec of the loot pool entry
     * @return the registered loot pool entry type
     */
    default class_5338 lootPoolEntry(String name, MapCodec<? extends class_79> codec) {
        return this.lootPoolEntry(name, new class_5338(codec));
    }

    /**
     * @param name the name of the loot function type
     * @param type the loot function type to register
     * @return the registered loot function type
     * @param <T> the type of loot function
     */
    default <T extends class_117> class_5339<T> lootFunction(String name, class_5339<T> type) {
        return this.register(class_7923.field_41134, name, type);
    }

    /**
     * @param name the name of the loot function type
     * @param codec the codec of the loot function
     * @return the registered loot function type
     * @param <T> the type of loot function
     */
    default <T extends class_117> class_5339<T> lootFunction(String name, MapCodec<T> codec) {
        return this.lootFunction(name, new class_5339<>(codec));
    }

    /**
     * @param name the name of the loot condition type
     * @param type the loot condition type to register
     * @return the registered loot condition type
     */
    default class_5342 lootCondition(String name, class_5342 type) {
        return this.register(class_7923.field_41135, name, type);
    }

    /**
     * @param name the name of the loot condition type
     * @param codec the codec of the loot condition
     * @return the registered loot condition type
     */
    default class_5342 lootCondition(String name, MapCodec<? extends class_5341> codec) {
        return this.lootCondition(name, new class_5342(codec));
    }

    /**
     * @param name the name of the loot number provider type
     * @param type the loot number provider type to register
     * @return the registered loot number provider type
     */
    default class_5657 lootNumberProvider(String name, class_5657 type) {
        return this.register(class_7923.field_41136, name, type);
    }

    /**
     * @param name the name of the loot number provider type
     * @param codec the codec of the loot number provider
     * @return the registered loot number provider type
     */
    default class_5657 lootNumberProvider(String name, MapCodec<? extends class_5658> codec) {
        return this.lootNumberProvider(name, new class_5657(codec));
    }

    /**
     * @param name the name of the loot NBT provider type
     * @param type the loot NBT provider type to register
     * @return the registered loot NBT provider type
     */
    default class_5650 lootNbtProvider(String name, class_5650 type) {
        return this.register(class_7923.field_41137, name, type);
    }

    /**
     * @param name the name of the loot NBT provider type
     * @param codec the codec of the loot NBT provider
     * @return the registered loot NBT provider type
     */
    default class_5650 lootNbtProvider(String name, MapCodec<? extends class_5651> codec) {
        return this.lootNbtProvider(name, new class_5650(codec));
    }

    /**
     * @param name the name of the loot score provider type
     * @param type the loot score provider type to register
     * @return the registered loot score provider type
     */
    default class_5669 lootScoreProvider(String name, class_5669 type) {
        return this.register(class_7923.field_41138, name, type);
    }

    /**
     * @param name the name of the loot score provider type
     * @param codec the codec of the loot score provider
     * @return the registered loot score provider type
     */
    default class_5669 lootScoreProvider(String name, MapCodec<? extends class_5670> codec) {
        return this.lootScoreProvider(name, new class_5669(codec));
    }

    /**
     * @param name the name of the float provider type
     * @param type the float provider type to register
     * @return the registered float provider type
     * @param <T> the type of float provider
     */
    default <T extends class_5863> class_5864<T> floatProvider(String name, class_5864<T> type) {
        return this.register(class_7923.field_41139, name, type);
    }

    /**
     * @param name the name of the float provider type
     * @param codec the codec of the float provider
     * @return the registered float provider type
     * @param <T> the type of float provider
     */
    default <T extends class_5863> class_5864<T> floatProvider(String name, MapCodec<T> codec) {
        return this.floatProvider(name, () -> codec);
    }

    /**
     * @param name the name of the int provider type
     * @param type the int provider type to register
     * @return the registered int provider type
     * @param <T> the type of int provider
     */
    default <T extends class_6017> class_6018<T> intProvider(String name, class_6018<T> type) {
        return this.register(class_7923.field_41140, name, type);
    }

    /**
     * @param name the name of the int provider type
     * @param codec the codec of the int provider
     * @return the registered int provider type
     * @param <T> the type of int provider
     */
    default <T extends class_6017> class_6018<T> intProvider(String name, MapCodec<T> codec) {
        return this.intProvider(name, () -> codec);
    }

    /**
     * @param name the name of the height provider type
     * @param type the height provider type to register
     * @return the registered height provider type
     * @param <T> the type of height provider
     */
    default <T extends class_6122> class_6123<T> heightProvider(String name, class_6123<T> type) {
        return this.register(class_7923.field_41141, name, type);
    }

    /**
     * @param name the name of the height provider type
     * @param codec the codec of the height provider
     * @return the registered height provider type
     * @param <T> the type of height provider
     */
    default <T extends class_6122> class_6123<T> heightProvider(String name, MapCodec<T> codec) {
        return this.heightProvider(name, () -> codec);
    }

    /**
     * @param name the name of the block predicate type
     * @param type the block predicate type to register
     * @return the registered block predicate type
     * @param <T> the type of block predicate
     */
    default <T extends class_6646> class_6647<T> blockPredicate(String name, class_6647<T> type) {
        return this.register(class_7923.field_41142, name, type);
    }

    /**
     * @param name the name of the block predicate type
     * @param codec the codec of the block predicate
     * @return the registered block predicate type
     * @param <T> the type of block predicate
     */
    default <T extends class_6646> class_6647<T> blockPredicate(String name, MapCodec<T> codec) {
        return this.blockPredicate(name, () -> codec);
    }

    /**
     * @param name the name of the carver
     * @param carver the carver to register
     * @return the registered carver
     * @param <T> the type of the carver config
     * @param <C> the type of the carver
     */
    default <T extends class_5871, C extends class_2939<T>> C carver(String name, C carver) {
        return this.register(class_7923.field_41143, name, carver);
    }

    /**
     * @param name the name of the feature
     * @param feature the feature to register
     * @return the registered feature
     * @param <T> the type of the feature config
     * @param <F> the type of the feature
     */
    default <T extends class_3037, F extends class_3031<T>> F feature(String name, F feature) {
        return this.register(class_7923.field_41144, name, feature);
    }

    /**
     * @param name the name of the structure placement type
     * @param type the structure placement type to register
     * @return the registered structure placement type
     * @param <T> the type of structure placement
     */
    default <T extends class_6874> class_6875<T> structurePlacement(String name, class_6875<T> type) {
        return this.register(class_7923.field_41145, name, type);
    }

    /**
     * @param name the name of the structure placement type
     * @param codec the codec of the structure placement
     * @return the registered structure placement type
     * @param <T> the type of structure placement
     */
    default <T extends class_6874> class_6875<T> structurePlacement(String name, MapCodec<T> codec) {
        return this.structurePlacement(name, () -> codec);
    }

    /**
     * @param name the name of the structure piece type
     * @param type the structure piece type to register
     * @return the registered structure piece type
     * @param <T> the type of the structure piece type
     * @see #simpleStructurePiece(String, class_3773.class_6615)
     * @see #managerAwareStructurePiece(String, class_3773.class_6616)
     */
    default <T extends class_3773> T structurePiece(String name, T type) {
        return this.register(class_7923.field_41146, name, type);
    }

    /**
     * @param name the name of the structure piece type
     * @param type the simple structure piece type to register
     * @return the registered structure piece type
     * @param <T> the type of the simple structure piece type
     */
    default <T extends class_3773.class_6615> T simpleStructurePiece(String name, T type) {
        return this.register(class_7923.field_41146, name, type);
    }

    /**
     * @param name the name of the structure piece type
     * @param type the manager aware structure piece type to register
     * @return the registered structure piece type
     * @param <T> the type of the manager aware structure piece type
     */
    default <T extends class_3773.class_6616> T managerAwareStructurePiece(String name, T type) {
        return this.register(class_7923.field_41146, name, type);
    }

    /**
     * @param name the name of the structure type
     * @param type the structure type to register
     * @return the registered structure type
     * @param <T> the type of structure
     */
    default <T extends class_3195> class_7151<T> structure(String name, class_7151<T> type) {
        return this.register(class_7923.field_41147, name, type);
    }

    /**
     * @param name the name of the structure type
     * @param codec the codec of the structure
     * @return the registered structure type
     * @param <T> the type of structure
     */
    default <T extends class_3195> class_7151<T> structure(String name, MapCodec<T> codec) {
        return this.structure(name, () -> codec);
    }

    /**
     * @param name the name of the placement modifier type
     * @param type the placement modifier type to register
     * @return the registered placement modifier type
     * @param <T> the type of placement modifier
     */
    default <T extends class_6797> class_6798<T> placementModifier(String name, class_6798<T> type) {
        return this.register(class_7923.field_41148, name, type);
    }

    /**
     * @param name the name of the placement modifier type
     * @param codec the codec of the placemenet modifier
     * @return the registered placement modifier type
     * @param <T> the type of placement modifier
     */
    default <T extends class_6797> class_6798<T> placementModifier(String name, MapCodec<T> codec) {
        return this.placementModifier(name, () -> codec);
    }

    /**
     * @param name the name of the block state provider type
     * @param type the block state provider type to register
     * @return the registered block state provider type
     * @param <T> the type of block state provider
     */
    default <T extends class_4651> class_4652<T> blockStateProvider(String name, class_4652<T> type) {
        return this.register(class_7923.field_41149, name, type);
    }

    /**
     * @param name the name of the block state provider type
     * @param codec the codec of the block state provider
     * @return the registered block state provider type
     * @param <T> the type of block state provider
     */
    default <T extends class_4651> class_4652<T> blockStateProvider(String name, MapCodec<T> codec) {
        return this.blockStateProvider(name, new class_4652<>(codec));
    }

    /**
     * @param name the name of the foliage placer type
     * @param type the foliage placer type to register
     * @return the registered foliage placer type
     * @param <T> the type of foliage placer
     */
    default <T extends class_4647> class_4648<T> foliagePlacer(String name, class_4648<T> type) {
        return this.register(class_7923.field_41150, name, type);
    }

    /**
     * @param name the name of the foliage placer type
     * @param codec the codec of the foliage placer
     * @return the registered foliage placer type
     * @param <T> the type of foliage placer
     */
    default <T extends class_4647> class_4648<T> foliagePlacer(String name, MapCodec<T> codec) {
        return this.foliagePlacer(name, new class_4648<>(codec));
    }

    /**
     * @param name the name of the trunk placer type
     * @param type the trunk placer
     * @return the registered trunk placer type
     * @param <T> the type of trunk placer
     */
    default <T extends class_5141> class_5142<T> trunkPlacer(String name, class_5142<T> type) {
        return this.register(class_7923.field_41151, name, type);
    }

    /**
     * @param name the name of the trunk placer type
     * @param codec the codec of the trunk placer
     * @return the registered trunk placer type
     * @param <T> the type of trunk placer
     */
    default <T extends class_5141> class_5142<T> trunkPlacer(String name, MapCodec<T> codec) {
        return this.trunkPlacer(name, new class_5142<>(codec));
    }

    /**
     * @param name the name of the root placer type
     * @param type the root placer type to register
     * @return the registered root placer type
     * @param <T> the type of root placer
     */
    default <T extends class_7387> class_7388<T> rootPlacer(String name, class_7388<T> type) {
        return this.register(class_7923.field_41152, name, type);
    }

    /**
     * @param name the name of the root placer type
     * @param codec the codec of the root placer
     * @return the registered root placer type
     * @param <T> the type of root placer
     */
    default <T extends class_7387> class_7388<T> rootPlacer(String name, MapCodec<T> codec) {
        return this.rootPlacer(name, new class_7388<>(codec));
    }

    /**
     * @param name the name of the tree decorator type
     * @param type the tree decorator type to register
     * @return the registered tree decorator type
     * @param <T> the type of tree decorator
     */
    default <T extends class_4662> class_4663<T> treeDecorator(String name, class_4663<T> type) {
        return this.register(class_7923.field_41153, name, type);
    }

    /**
     * @param name the name of the tree decorator type
     * @param codec the codec of the tree decorator
     * @return the registered tree decorator type
     * @param <T> the type of tree decorator
     */
    default <T extends class_4662> class_4663<T> treeDecorator(String name, MapCodec<T> codec) {
        return this.treeDecorator(name, new class_4663<>(codec));
    }

    /**
     * @param name the name of the feature size type
     * @param type the feature size type to register
     * @return the registered feature size type
     * @param <T> the type of feature size
     */
    default <T extends class_5201> class_5202<T> featureSize(String name, class_5202<T> type) {
        return this.register(class_7923.field_41155, name, type);
    }

    /**
     * @param name the name of the feature size type
     * @param codec the codec of the feature size
     * @return the registered feature size type
     * @param <T> the type of feature size
     */
    default <T extends class_5201> class_5202<T> featureSize(String name, MapCodec<T> codec) {
        return this.featureSize(name, new class_5202<>(codec));
    }

    /**
     * @param name the name of the biome source
     * @param codec the codec of the biome source
     * @return the registered biome source codec
     * @param <T> the type of biome source
     */
    default <T extends class_1966> MapCodec<T> biomeSource(String name, MapCodec<T> codec) {
        return this.register(class_7923.field_41156, name, codec);
    }

    /**
     * @param name the name of the chunk generator
     * @param codec the codec of the chunk generator
     * @return the registered chunk generator codec
     * @param <T> the type of chunk generator
     */
    default <T extends class_2794> MapCodec<T> chunkGenerator(String name, MapCodec<T> codec) {
        return this.register(class_7923.field_41157, name, codec);
    }

    /**
     * @param name the name of the material condition
     * @param codec the codec of the material condition
     * @return the registered material condition codec
     * @param <T> the type of material condition
     */
    default <T extends class_6686.class_6693> MapCodec<T> materialCondition(String name, MapCodec<T> codec) {
        return this.register(class_7923.field_41158, name, codec);
    }

    /**
     * @param name the name of the material rule
     * @param codec the codec of the material rule
     * @return the registered material rule codec
     * @param <T> the type of material rule
     */
    default <T extends class_6686.class_6708> MapCodec<T> materialRule(String name, MapCodec<T> codec) {
        return this.register(class_7923.field_41159, name, codec);
    }

    /**
     * @param name the name of the density function
     * @param codec the codec of the density function
     * @return the registered density function codec
     * @param <T> the type of density function
     */
    default <T extends class_6910> MapCodec<T> densityFunction(String name, MapCodec<T> codec) {
        return this.register(class_7923.field_41160, name, codec);
    }

    /**
     * @param name the name of the block type
     * @param codec the codec of the block type
     * @return the registered block type codec
     * @param <T> the type of block type
     */
    default <T extends class_2248> MapCodec<T> blockType(String name, MapCodec<T> codec) {
        return this.register(class_7923.field_46591, name, codec);
    }

    /**
     * @param name the name of the structure processor type
     * @param type the structure processor type to register
     * @return the registered structure processor type
     * @param <T> the type of structure processor
     */
    default <T extends class_3491> class_3828<T> structureProcessor(String name, class_3828<T> type) {
        return this.register(class_7923.field_41161, name, type);
    }

    /**
     * @param name the name of the structure processor type
     * @param codec the codec of the structure processor
     * @return the registered structure processor type
     * @param <T> the type of structure processor
     */
    default <T extends class_3491> class_3828<T> structureProcessor(String name, MapCodec<T> codec) {
        return this.structureProcessor(name, () -> codec);
    }

    /**
     * @param name the name of the structure pool element type
     * @param type the structure pool element type to register
     * @return the registered structure pool element type
     * @param <T> the type of structure pool element
     */
    default <T extends class_3784> class_3816<T> structurePoolElement(String name, class_3816<T> type) {
        return this.register(class_7923.field_41162, name, type);
    }

    /**
     * @param name the name of the structure pool element type
     * @param codec the codec of the structure pool element
     * @return the registered structure pool element type
     * @param <T> the type of structure pool element
     */
    default <T extends class_3784> class_3816<T> structurePoolElement(String name, MapCodec<T> codec) {
        return this.structurePoolElement(name, () -> codec);
    }

    /**
     * @param name the name of the pool alias binding
     * @param codec the codec of the pool alias binding
     * @return the registered pool alias binding codec
     * @param <T> the type of pool alias binding
     */
    default <T extends class_8889> MapCodec<T> poolAliasBinding(String name, MapCodec<T> codec) {
        return this.register(class_7923.field_46912, name, codec);
    }

    /**
     * @param name the name of the decorated pot pattern
     * @param pattern the decorated pot pattern to register
     * @return the registered decorated pot pattern
     */
    default class_9766 decoratedPotPattern(String name, class_9766 pattern) {
        return this.register(class_7923.field_42940, name, pattern);
    }

    /**
     * Registers a {@link class_9766} using {@code name} as the pattern's asset id.
     * @param name the name of the decorated pot pattern
     * @return the registered decorated pot pattern
     */
    default class_9766 decoratedPotPattern(String name) {
        return this.decoratedPotPattern(name, new class_9766(this.id(name)));
    }

    /**
     * @param name the name of the criterion
     * @param criterion the criterion to register
     * @return the registered criterion
     * @param <T> the type of the criterion
     */
    default <T extends class_179<?>> T criterion(String name, T criterion) {
        return this.register(class_7923.field_47496, name, criterion);
    }

    /**
     * @param name the name of the number format type
     * @param type the number format type to register
     * @return the registered number format type
     * @param <T> the type of number format
     */
    default <T extends class_9022> class_9023<T> numberFormat(String name, class_9023<T> type) {
        return this.register(class_7923.field_47555, name, type);
    }

    /**
     * @param name the name of the entity sub-predicate
     * @param codec the codec of the entity sub-predicate
     * @return the registered entity sub-predicate codec
     * @param <T> the type of entity sub-predicate
     */
    default <T extends class_7376> MapCodec<T> entitySubPredicate(String name, MapCodec<T> codec) {
        return this.register(class_7923.field_49911, name, codec);
    }

    /**
     * @param name the name of the data component predicate
     * @param codec the codec of the data component predicate
     * @return the registered data component predicate type
     * @param <T> the type of data component predicate
     */
    default <T extends class_9360> class_9360.class_8745<T> dataComponentPredicate(String name, Codec<T> codec) {
        return this.register(class_7923.field_56404, name, new class_9360.class_8745<>(codec));
    }

    /**
     * Registers a {@link class_9428} wrapped in a {@link class_6880.class_6883}.
     * @param name the name of the map decoration type
     * @param type the map decoration type to register
     * @return the registered map decoration type
     */
    default class_6880.class_6883<class_9428> mapDecoration(String name, class_9428 type) {
        return this.registerReference(class_7923.field_50078, name, type);
    }

    /**
     * Registers a {@link class_9428} wrapped in a {@link class_6880.class_6883}.
     * {@code name} is used as the map decoration type's asset id.
     * @param name the name of the map decoration type
     * @param showOnItemFrame {@code true} for showing the map decoration on item frames
     * @param mapColor the color of the map decoration in RGB format
     * @param explorationMapElement {@code true} disallows the map from being expanded in a cartography table
     * @param trackCount {@code true} for tracking the amount of this map decoration on a map
     * @return the registered map decoration type
     */
    default class_6880.class_6883<class_9428> mapDecoration(String name, boolean showOnItemFrame, int mapColor, boolean explorationMapElement, boolean trackCount) {
        return this.mapDecoration(name, new class_9428(this.id(name), showOnItemFrame, mapColor, explorationMapElement, trackCount));
    }

    /**
     * Registers a {@link class_9428} wrapped in a {@link class_6880.class_6883}.
     * {@code mapColor} is defaulted to white and {@code explorationMapElement} is defaulted to {@code false}.
     * @param name the name of the map decoration type
     * @param showOnItemFrame {@code true} for showing the map decoration on item frames
     * @param trackCount {@code true} for tracking the amount of this map decoration on a map
     * @return the registered map decoration type
     */
    default class_6880.class_6883<class_9428> mapDecoration(String name, boolean showOnItemFrame, boolean trackCount) {
        return this.mapDecoration(name, showOnItemFrame, -1, false, trackCount);
    }

    /**
     * @param name the name of the enchantment level based value
     * @param codec the codec of the enchantment level based value
     * @return the registered enchantment level based value codec
     * @param <T> the type of enchantment level based value
     */
    default <T extends class_9704> MapCodec<T> enchantmentLevelBasedValue(String name, MapCodec<T> codec) {
        return this.register(class_7923.field_51833, name, codec);
    }

    /**
     * @param name the name of the enchantment entity effect
     * @param codec the codec of the enchantment entity effect
     * @return the registered enchantment entity effect codec
     * @param <T> the type of enchantment entity effect
     */
    default <T extends class_9721> MapCodec<T> enchantmentEntityEffect(String name, MapCodec<T> codec) {
        return this.register(class_7923.field_51834, name, codec);
    }

    /**
     * @param name the name of the enchantment location based effect
     * @param codec the codec of the enchantment location based effect
     * @return the registered enchantment location based effect codec
     * @param <T> the type of enchantment location based effect
     */
    default <T extends class_9722> MapCodec<T> enchantmentLocationBasedEffect(String name, MapCodec<T> codec) {
        return this.register(class_7923.field_51835, name, codec);
    }

    /**
     * @param name the name of the enchantment value effect
     * @param codec the codec of the enchantment value effect
     * @return the registered enchantment value effect codec
     * @param <T> the type of enchantment value effect
     */
    default <T extends class_9723> MapCodec<T> enchantmentValueEffect(String name, MapCodec<T> codec) {
        return this.register(class_7923.field_51836, name, codec);
    }

    /**
     * @param name the name of the enchantment provider
     * @param codec the codec of the enchantment provider
     * @return the registered enchantment provider codec
     * @param <T> the type of enchantment provider
     */
    default <T extends class_9741> MapCodec<T> enchantmentProvider(String name, MapCodec<T> codec) {
        return this.register(class_7923.field_51837, name, codec);
    }

    /**
     * @param name the name of the consume effect type
     * @param codec the codec of the consume effect for serialization
     * @param packetCodec the packet codec of the consume effect for network serialization
     * @return the registered consume effect type
     * @param <T> the type of consume effect
     */
    default <T extends class_10134> class_10134.class_10135<T> consumeEffect(String name, MapCodec<T> codec, class_9139<class_9129, T> packetCodec) {
        return this.register(class_7923.field_53967, name, new class_10134.class_10135<>(codec, packetCodec));
    }

    /**
     * @param name the name of the recipe display serializer
     * @param codec the codec of the recipe display for serialization
     * @param packetCodec the packet codec of the recipe display for network serialization
     * @return the registered recipe display serializer
     * @param <T> the type of recipe display
     */
    default <T extends class_10295> class_10295.class_10296<T> recipeDisplay(String name, MapCodec<T> codec, class_9139<class_9129, T> packetCodec) {
        return this.register(class_7923.field_54874, name, new class_10295.class_10296<>(codec, packetCodec));
    }

    /**
     * @param name the name of the slot display serializer
     * @param codec the codec of the slot display for serialization
     * @param packetCodec the packet codec of the slot display for network serialization
     * @return the registered slot display serializer
     * @param <T> the type of slot display
     */
    default <T extends class_10302> class_10302.class_10312<T> slotDisplay(String name, MapCodec<T> codec, class_9139<class_9129, T> packetCodec) {
        return this.register(class_7923.field_54873, name, new class_10302.class_10312<>(codec, packetCodec));
    }

    /**
     * @param name the name of the recipe book category
     * @param category the recipe book category to register
     * @return the registered recipe book category
     */
    default class_10355 recipeBookCategory(String name, class_10355 category) {
        return this.register(class_7923.field_54927, name, category);
    }

    /**
     * @param name the name of the recipe book category
     * @return the registered recipe book category
     */
    default class_10355 recipeBookCategory(String name) {
        return this.recipeBookCategory(name, new class_10355());
    }

    /**
     * @param name the name of the ticket type
     * @param expiryTicks the duration of chunk tickets of this type
     * @param flags packed int flags of the ticket type, such as {@link class_3230#field_61096},
     *              {@link class_3230#field_61097}, {@link class_3230#field_61098},
     *              {@link class_3230#field_61099}, and {@link class_3230#field_61100}
     * @return the registered ticket type
     */
    default class_3230 ticketType(String name, long expiryTicks, int flags) {
        return this.register(class_7923.field_55881, name, new class_3230(expiryTicks, flags));
    }

    /**
     * @param name the name of the test environment definition
     * @param codec the codec of the test environment definition
     * @return the registered test environment definition codec
     * @param <T> the type of test environment definition
     */
    default <T extends class_10665> MapCodec<T> testEnvironmentDefinition(String name, MapCodec<T> codec) {
        return this.register(class_7923.field_56154, name, codec);
    }

    /**
     * @param name the name of the test instance
     * @param codec the codec of the test instance
     * @return the registered test instance codec
     * @param <T> the type of test instance
     */
    default <T extends class_10660> MapCodec<T> testInstance(String name, MapCodec<T> codec) {
        return this.register(class_7923.field_56155, name, codec);
    }

    /**
     * @param name the name of the test function
     * @param testFunction the test function to register
     * @return the registered test function
     */
    default Consumer<class_4516> testFunction(String name, Consumer<class_4516> testFunction) {
        return this.register(class_7923.field_56156, name, testFunction);
    }

    /**
     * @param name the name of the spawn condition
     * @param codec the codec of the spawn condition
     * @return the registered spawn condition codec
     * @param <T> the type of spawn condition
     */
    default <T extends class_10699> MapCodec<T> spawnCondition(String name, MapCodec<T> codec) {
        return this.register(class_7923.field_56403, name, codec);
    }

    /**
     * @param name the name of the dialog
     * @param codec the codec of the dialog
     * @return the registered dialog codec
     * @param <T> the type of dialog
     */
    default <T extends class_11419> MapCodec<T> dialog(String name, MapCodec<T> codec) {
        return this.register(class_7923.field_60812, name, codec);
    }

    /**
     * @param name the name of the dialog body
     * @param codec the codec of the dialog body
     * @return the registered dialog body codec
     * @param <T> the type of dialog body
     */
    default <T extends class_11432> MapCodec<T> dialogBody(String name, MapCodec<T> codec) {
        return this.register(class_7923.field_60815, name, codec);
    }

    /**
     * @param name the name of the dialog action
     * @param codec the codec of the dialog action
     * @return the registered dialog action codec
     * @param <T> the type of dialog action
     */
    default <T extends class_11521> MapCodec<T> dialogAction(String name, MapCodec<T> codec) {
        return this.register(class_7923.field_61038, name, codec);
    }

    /**
     * @param name the name of the input control
     * @param codec the codec of the input control
     * @return the registered input control codec
     * @param <T> the type of input control
     */
    default <T extends class_11438> MapCodec<T> inputControl(String name, MapCodec<T> codec) {
        return this.register(class_7923.field_60814, name, codec);
    }

    /**
     * @param name the name of the debug subscription
     * @param type the debug subscription type to register
     * @return the registered debug subscription
     * @param <T> the type of debug subscription
     */
    default <T> class_12017<T> debugSubscription(String name, class_12017<T> type) {
        return this.register(class_7923.field_62996, name, type);
    }

    /**
     * @param name the name of the debug subscription
     * @return the registered debug subscription
     * @param <T> the type of debug subscription
     */
    default <T> class_12017<T> debugSubscription(String name) {
        return this.debugSubscription(name, new class_12017<>(null));
    }

    /**
     * @param name the name of the debug subscription
     * @param packetCodec the packet codec of the debug subscription
     * @return the registered debug subscription
     * @param <T> the type of debug subscription
     */
    default <T> class_12017<T> debugSubscription(String name, @Nullable class_9139<? super class_9129, T> packetCodec) {
        return this.debugSubscription(name, new class_12017<>(packetCodec));
    }

    /**
     * @param name the name of the debug subscription
     * @param packetCodec the packet codec of the debug subscription
     * @param expiry the expiry of the debug subscription
     * @return the registered debug subscription
     * @param <T> the type of debug subscription
     */
    default <T> class_12017<T> debugSubscription(String name, @Nullable class_9139<? super class_9129, T> packetCodec, int expiry) {
        return this.debugSubscription(name, new class_12017<>(packetCodec, expiry));
    }

    /**
     * @param name the name of the incoming rpc method
     * @param builder the builder for the incoming rpc method
     * @return the registered incoming rpc method
     * @param <T> the type of incoming rpc method
     */
    default <T extends class_11796> T incomingRpcMethod(String name, class_11796.class_11797<T> builder) {
        class_2960 id = this.id(name);
        return this.register(class_7923.field_62490, name, builder.method_73631());
    }

    /**
     * @param name the name of the outgoing rpc method
     * @param builder the builder for the outgoing rpc method
     * @return the registered outgoing rpc method
     * @param <T> the type of outgoing rpc method
     */
    default <T extends class_11808<?, ?>> T outgoingRpcMethod(String name, class_11808.class_11811<T> builder) {
        class_2960 id = this.id(name);
        return this.register(class_7923.field_62491, name, builder.method_73695());
    }

    /**
     * @param name the name of the custom style attribute
     * @param codec the codec of the custom style attribute
     * @param cache {@code true} if codec decode results should be cached
     * @return the registered custom style attribute
     * @param <T> the type of the custom style attribute
     */
    default <T> CustomStyleAttribute<T> customStyleAttribute(String name, Codec<T> codec, boolean cache) {
        return this.register(ZineRegistries.CUSTOM_STYLE_ATTRIBUTES, name, new CustomStyleAttribute<>(codec, cache));
    }

    /**
     * @param name the name of the tracked data handler
     * @param trackedDataHandler the tracked data handler to register
     * @return the registered tracked data handler
     * @param <T> the type of tracked data
     */
    default <T> class_2941<T> trackedDataHandler(String name, class_2941<T> trackedDataHandler) {
        FabricTrackedDataRegistry.register(this.id(name), trackedDataHandler);
        return trackedDataHandler;
    }

    /**
     * @param name the name of the tracked data handler
     * @param codec the packet codec of the tracked data handler
     * @return the registered tracked data handler
     * @param <T> the type of tracked data
     */
    default <T> class_2941<T> trackedDataHandler(String name, class_9139<? super class_9129, T> codec) {
        return this.trackedDataHandler(name, class_2941.method_56031(codec));
    }

    /**
     * @param name the name of the text content
     * @param codec the codec of the text content
     * @return the registered text content codec
     * @param <T> the type of text content
     */
    default <T extends class_7417> MapCodec<T> textContent(String name, MapCodec<T> codec) {
        TextUtil.registerTextContent(this.id(name), codec);
        return codec;
    }

    /**
     * @param name the name of the text object contents
     * @param codec the codec of the text object contents
     * @return the registered text object contents codec
     * @param <T> the type of text object contents
     */
    default <T extends class_11724> MapCodec<T> textObjectContents(String name, MapCodec<T> codec) {
        TextUtil.registerTextObjectContents(this.id(name), codec);
        return codec;
    }

    /**
     * @param name the name of the nbt data source
     * @param codec the codec of the nbt data source
     * @return the registered nbt data source codec
     * @param <T> the type of nbt data source
     */
    default <T extends class_7419> MapCodec<T> nbtDataSource(String name, MapCodec<T> codec) {
        TextUtilImpl.registerNbtDataSource(this.id(name), codec);
        return codec;
    }

    /**
     * @param name the name of the block set type
     * @param builder the builder for the block set type
     * @return the registered block set type
     */
    default class_8177 blockSetType(String name, BlockSetTypeBuilder builder) {
        return builder.register(this.id(name));
    }

    /**
     * @param name the name of the wood type
     * @param builder the builder for the wood type
     * @param blockSetType the block set type for the wood type
     * @return the registered wood type
     */
    default class_4719 woodType(String name, WoodTypeBuilder builder, class_8177 blockSetType) {
        return builder.register(this.id(name), blockSetType);
    }

    private <T extends class_2248> T registerBlockItem(String name, T block) {
        this.item(name, block);
        return block;
    }
}
