package com.eightsidedsquare.zine.common.util.codec;

import com.eightsidedsquare.zine.common.entity.SpawnReasonIds;
import com.mojang.datafixers.util.Either;
import com.mojang.serialization.Codec;
import com.mojang.serialization.DataResult;
import com.mojang.serialization.MapCodec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import org.apache.commons.lang3.mutable.*;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import net.minecraft.class_11368;
import net.minecraft.class_156;
import net.minecraft.class_2248;
import net.minecraft.class_2379;
import net.minecraft.class_238;
import net.minecraft.class_2769;
import net.minecraft.class_2960;
import net.minecraft.class_3730;
import net.minecraft.class_5251;
import net.minecraft.class_5699;
import net.minecraft.class_7923;

public final class CodecUtil {

    public static final Codec<Character> CHARACTER = Codec.string(1, 1).xmap(string -> string.charAt(0), String::valueOf);
    public static final Codec<class_2379> EULER_ANGLE = RecordCodecBuilder.create(instance -> instance.group(
            Codec.FLOAT.optionalFieldOf("pitch", 0f).forGetter(class_2379::comp_3776),
            Codec.FLOAT.optionalFieldOf("yaw", 0f).forGetter(class_2379::comp_3777),
            Codec.FLOAT.optionalFieldOf("roll", 0f).forGetter(class_2379::comp_3778)
    ).apply(instance, class_2379::new));
    public static final Codec<Integer> INT_STRING = Codec.STRING.comapFlatMap(
            string -> {
                int value;
                try {
                    value = Integer.parseInt(string);
                } catch (NumberFormatException e) {
                    return DataResult.error(() -> "Failed to parse int from " + string);
                }
                return DataResult.success(value);
            },
            String::valueOf
    );
    public static final Codec<class_2248> BLOCK = class_7923.field_41175.method_39673();
    public static final Codec<Integer> COLOR = AlternativeCodec.create(
            class_5699.field_54067,
            class_5251.field_39242,
            color -> Optional.of(class_5251.method_27717(color)),
            textColor -> textColor.method_27716() | 0xff000000
    );
    public static final Codec<class_3730> SPAWN_REASON = SpawnReasonIds.IDS.method_65323(class_2960.field_25139);
    public static final Codec<class_238> CODEC = Codec.DOUBLE.listOf().comapFlatMap(
            vertices -> class_156.method_33141(vertices, 6)
                    .map(list -> new class_238(list.getFirst(), list.get(1), list.get(2), list.get(3), list.get(4), list.get(5))),
            box -> List.of(box.field_1323, box.field_1322, box.field_1321, box.field_1320, box.field_1325, box.field_1324)
    );
    public static final Codec<MutableBoolean> MUTABLE_BOOLEAN = Codec.BOOL.xmap(MutableBoolean::new, MutableBoolean::booleanValue);
    public static final Codec<MutableByte> MUTABLE_BYTE = Codec.BYTE.xmap(MutableByte::new, MutableByte::byteValue);
    public static final Codec<MutableShort> MUTABLE_SHORT = Codec.SHORT.xmap(MutableShort::new, MutableShort::shortValue);
    public static final Codec<MutableInt> MUTABLE_INT = Codec.INT.xmap(MutableInt::new, MutableInt::intValue);
    public static final Codec<MutableLong> MUTABLE_LONG = Codec.LONG.xmap(MutableLong::new, MutableLong::longValue);
    public static final Codec<MutableFloat> MUTABLE_FLOAT = Codec.FLOAT.xmap(MutableFloat::new, MutableFloat::floatValue);
    public static final Codec<MutableDouble> MUTABLE_DOUBLE = Codec.DOUBLE.xmap(MutableDouble::new, MutableDouble::doubleValue);

    /**
     * Creates a list codec that can deserialize single elements as a list,
     * and serialize lists of size 1 as a single element.
     * @param codec the codec of a single element
     * @param <A> the type of the list's element
     */
    public static <A> Codec<List<A>> listCodec(Codec<A> codec) {
        return Codec.either(codec, codec.listOf()).xmap(
                either -> either.map(List::of, list -> list),
                list -> list.size() == 1 ? Either.left(list.getFirst()) : Either.right(list)
        );
    }

    /**
     * Creates a list codec using {@link #listCodec(Codec)} with validation to prevent empty lists.
     * @param codec the codec of a single element
     * @param <A> the type of the list's element
     */
    public static <A> Codec<List<A>> nonEmptyListCodec(Codec<A> codec) {
        return listCodec(codec).validate(list -> list.isEmpty() ? DataResult.error(() -> "Empty list") : DataResult.success(list));
    }

    public static <O, T> Codec<O> codec(RecordCodecBuilder<O, T> field, Function<T, O> function) {
        return RecordCodecBuilder.create(instance -> instance.group(field).apply(instance, function));
    }

    public static <O, T> MapCodec<O> mapCodec(RecordCodecBuilder<O, T> field, Function<T, O> function) {
        return RecordCodecBuilder.mapCodec(instance -> instance.group(field).apply(instance, function));
    }

    public static Codec<class_2248> blockCodecWithProperties(class_2769<?>... properties) {
        return BLOCK.validate(block -> {
            Collection<class_2769<?>> blockProperties = block.method_9595().method_11659();
            for (class_2769<?> property : properties) {
                if(!blockProperties.contains(property)) {
                    return DataResult.error(() -> block + " does not contain property " + property);
                }
            }
            return DataResult.success(block);
        });
    }

    public static Codec<class_2248> blockCodecWithPropertiesOf(class_2248 block) {
        return blockCodecWithProperties(block.method_9595().method_11659().toArray(new class_2769<?>[0]));
    }

    public static <T> void readToList(class_11368 view, String key, List<T> list, Codec<? extends Collection<? extends T>> codec) {
        list.clear();
        view.method_71426(key, codec).ifPresent(list::addAll);
    }

    public static <K, V> void readToMap(class_11368 view, String key, Map<K, V> map, Codec<? extends Map<? extends K, ? extends V>> codec) {
        map.clear();
        view.method_71426(key, codec).ifPresent(map::putAll);
    }

    public static <T> Codec<MutableObject<T>> mutable(Codec<T> codec) {
        return codec.xmap(MutableObject::new, MutableObject::getValue);
    }

    private CodecUtil() {
    }
}
