package com.eightsidedsquare.zine.common.util.codec;

import com.eightsidedsquare.zine.common.util.network.PacketCodecUtil;
import com.google.common.collect.ImmutableList;
import com.mojang.serialization.Codec;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.function.BiConsumer;
import java.util.function.Function;
import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_2338;
import net.minecraft.class_2520;
import net.minecraft.class_2680;
import net.minecraft.class_4844;
import net.minecraft.class_5699;
import net.minecraft.class_9129;
import net.minecraft.class_9135;
import net.minecraft.class_9139;
import net.minecraft.class_9323;

public class DataHelperImpl<T> implements DataHelper<T> {

    private final List<DataHelper<T>> fields;

    public DataHelperImpl(List<DataHelper<T>> fields) {
        this.fields = fields;
    }

    @Override
    public void read(class_11368 view, T object) {
        for (DataHelper<T> field : this.fields) {
            field.read(view, object);
        }
    }

    @Override
    public void write(class_11372 view, T object) {
        for (DataHelper<T> field : this.fields) {
            field.write(view, object);
        }
    }

    @Override
    public <I extends class_9129> void read(I buf, T object) {
        for (DataHelper<T> field : this.fields) {
            field.read(buf, object);
        }
    }

    @Override
    public <I extends class_9129> void write(I buf, T object) {
        for (DataHelper<T> field : this.fields) {
            field.write(buf, object);
        }
    }

    static class BuilderImpl<T> implements Builder<T> {

        private final ImmutableList.Builder<DataHelper<T>> fields = ImmutableList.builder();

        private Builder<T> add(DataHelper<T> field) {
            this.fields.add(field);
            return this;
        }

        @Override
        public <F> FieldBuilder<F, T> field(@Nullable Codec<F> codec, @Nullable class_9139<? super class_9129, F> packetCodec, String key) {
            return (defaultValue, getter, setter) ->
                    this.add(new Field<>(codec, packetCodec, key, defaultValue, getter, setter));
        }

        @Override
        public <F> NullableFieldBuilder<F, T> nullableField(@Nullable Codec<F> codec, @Nullable class_9139<? super class_9129, F> packetCodec, String key) {
            return (defaultValue, getter, setter) -> this.field(
                            codec == null ? null : class_5699.method_57155(codec),
                            packetCodec == null ? null : class_9135.method_56382(packetCodec.method_56430()),
                            key
                    )
                    .apply(
                            t -> Optional.ofNullable(defaultValue.apply(t)),
                            t -> Optional.ofNullable(getter.apply(t)),
                            (t, f) -> setter.accept(t, f.orElse(null))
                    );
        }

        @Override
        public <F, L extends Collection<F>> ListFieldBuilder<F, L, T> listField(@Nullable Codec<L> codec, @Nullable class_9139<? super class_9129, L> packetCodec, String key) {
            return getter -> this.add(new ListField<>(codec, packetCodec, key, getter));
        }

        @Override
        public <F> ListFieldBuilder<F, List<F>, T> listFieldOf(@Nullable Codec<F> codec, @Nullable class_9139<? super class_9129, F> packetCodec, String key) {
            return this.listField(
                    codec == null ? null : codec.listOf(),
                    packetCodec == null ? null : class_9135.method_56376(ArrayList::new, packetCodec),
                    key
            );
        }

        @Override
        public <K, V, M extends Map<K, V>> MapFieldBuilder<K, V, M, T> mapField(@Nullable Codec<M> codec, @Nullable class_9139<? super class_9129, M> packetCodec, String key) {
            return getter -> this.add(new MapField<>(codec, packetCodec, key, getter));
        }

        @Override
        public <K, V> MapFieldBuilder<K, V, Map<K, V>, T> mapFieldOf(@Nullable Codec<K> keyCodec, @Nullable Codec<V> elementCodec, @Nullable class_9139<? super class_9129, K> keyPacketCodec, @Nullable class_9139<? super class_9129, V> elementPacketCodec, String key) {
            return this.mapField(
                    keyCodec == null || elementCodec == null ? null : Codec.unboundedMap(keyCodec, elementCodec),
                    keyPacketCodec == null || elementPacketCodec == null ? null : class_9135.method_56377(HashMap::new, keyPacketCodec, elementPacketCodec),
                    key
            );
        }

        @Override
        public FieldBuilder<Boolean, T> booleanField(String key) {
            return this.field(Codec.BOOL, class_9135.field_48547, key);
        }

        @Override
        public FieldBuilder<Byte, T> byteField(String key) {
            return this.field(Codec.BYTE, class_9135.field_48548, key);
        }

        @Override
        public FieldBuilder<Short, T> shortField(String key) {
            return this.field(Codec.SHORT, class_9135.field_48549, key);
        }

        @Override
        public FieldBuilder<Integer, T> intField(String key) {
            return this.field(Codec.INT, class_9135.field_48550, key);
        }

        @Override
        public FieldBuilder<Long, T> longField(String key) {
            return this.field(Codec.LONG, class_9135.field_48551, key);
        }

        @Override
        public FieldBuilder<Float, T> floatField(String key) {
            return this.field(Codec.FLOAT, class_9135.field_48552, key);
        }

        @Override
        public FieldBuilder<Double, T> doubleField(String key) {
            return this.field(Codec.DOUBLE, class_9135.field_48553, key);
        }

        @Override
        public FieldBuilder<String, T> stringField(String key) {
            return this.field(Codec.STRING, class_9135.field_48554, key);
        }

        @Override
        public FieldBuilder<class_2520, T> nbtElementField(String key) {
            return this.field(class_5699.field_60980, class_9135.field_48555, key);
        }

        @Override
        public FieldBuilder<UUID, T> uuidField(String key) {
            return this.field(class_4844.field_25122, class_4844.field_48453, key);
        }

        @Override
        public FieldBuilder<class_2338, T> blockPosField(String key) {
            return this.field(class_2338.field_25064, class_2338.field_48404, key);
        }

        @Override
        public FieldBuilder<class_2680, T> blockStateField(String key) {
            return this.field(class_2680.field_24734, PacketCodecUtil.BLOCK_STATE, key);
        }

        @Override
        public FieldBuilder<class_9323, T> componentMapField(String key) {
            return this.field(class_9323.field_50234, PacketCodecUtil.COMPONENT_MAP, key);
        }

        @Override
        public DataHelper<T> build() {
            return new DataHelperImpl<>(this.fields.build());
        }
    }

    static abstract class AbstractField<F, T> implements DataHelper<T> {
        @Nullable final Codec<F> codec;
        @Nullable final class_9139<? super class_9129, F> packetCodec;
        final String key;
        final Function<T, F> getter;

        AbstractField(@Nullable Codec<F> codec, @Nullable class_9139<? super class_9129, F> packetCodec, String key, Function<T, F> getter) {
            if(codec == null && packetCodec == null) {
                throw new IllegalArgumentException("Both codec and packet codec cannot be null for field " + key);
            }
            this.codec = codec;
            this.packetCodec = packetCodec;
            this.key = key;
            this.getter = getter;
        }

        abstract void read(class_11368 view, Codec<F> codec, T object);

        abstract void write(class_11372 view, Codec<F> codec, T object);

        abstract <I extends class_9129> void read(I buf, class_9139<? super class_9129, F> packetCodec, T object);

        abstract <I extends class_9129> void write(I buf, class_9139<? super class_9129, F> packetCodec, T object);

        @Override
        public final void read(class_11368 view, T object) {
            if(this.codec != null) {
                this.read(view, this.codec, object);
            }
        }

        @Override
        public final void write(class_11372 view, T object) {
            if(this.codec != null) {
                this.write(view, this.codec, object);
            }
        }

        @Override
        public final <I extends class_9129> void read(I buf, T object) {
            if(this.packetCodec != null) {
                this.read(buf, this.packetCodec, object);
            }
        }

        @Override
        public final <I extends class_9129> void write(I buf, T object) {
            if(this.packetCodec != null) {
                this.write(buf, this.packetCodec, object);
            }
        }
    }

    static final class Field<F, T> extends AbstractField<F, T> {
        private final Function<T, F> defaultValueGetter;
        private final BiConsumer<T, F> setter;

        Field(@Nullable Codec<F> codec, @Nullable class_9139<? super class_9129, F> packetCodec, String key, Function<T, F> defaultValueGetter, Function<T, F> getter, BiConsumer<T, F> setter) {
            super(codec, packetCodec, key, getter);
            this.defaultValueGetter = defaultValueGetter;
            this.setter = setter;
        }

        @Override
        void read(class_11368 view, Codec<F> codec, T object) {
            this.setter.accept(object, view.method_71426(this.key, codec).orElse(this.defaultValueGetter.apply(object)));
        }

        @Override
        void write(class_11372 view, Codec<F> codec, T object) {
            view.method_71468(this.key, codec, this.getter.apply(object));
        }

        @Override
        <I extends class_9129> void read(I buf, class_9139<? super class_9129, F> packetCodec, T object) {
            this.setter.accept(object, packetCodec.decode(buf));
        }

        @Override
        <I extends class_9129> void write(I buf, class_9139<? super class_9129, F> packetCodec, T object) {
            packetCodec.encode(buf, this.getter.apply(object));
        }
    }

    static final class ListField<F, L extends Collection<F>, T> extends AbstractField<L, T> {

        ListField(@Nullable Codec<L> codec, @Nullable class_9139<? super class_9129, L> packetCodec, String key, Function<T, L> getter) {
            super(codec, packetCodec, key, getter);
        }

        @Override
        void read(class_11368 view, Codec<L> codec, T object) {
            L list = this.getter.apply(object);
            list.clear();
            view.method_71426(this.key, codec).ifPresent(list::addAll);
        }

        @Override
        void write(class_11372 view, Codec<L> codec, T object) {
            view.method_71468(this.key, codec, this.getter.apply(object));
        }

        @Override
        <I extends class_9129> void read(I buf, class_9139<? super class_9129, L> packetCodec, T object) {
            L list = this.getter.apply(object);
            list.clear();
            list.addAll(packetCodec.decode(buf));
        }

        @Override
        <I extends class_9129> void write(I buf, class_9139<? super class_9129, L> packetCodec, T object) {
            packetCodec.encode(buf, this.getter.apply(object));
        }
    }

    static final class MapField<K, V, M extends Map<K, V>, T> extends AbstractField<M, T> {

        MapField(@Nullable Codec<M> codec, @Nullable class_9139<? super class_9129, M> packetCodec, String key, Function<T, M> getter) {
            super(codec, packetCodec, key, getter);
        }

        @Override
        void read(class_11368 view, Codec<M> codec, T object) {
            M map = this.getter.apply(object);
            map.clear();
            view.method_71426(this.key, codec).ifPresent(map::putAll);
        }

        @Override
        void write(class_11372 view, Codec<M> codec, T object) {
            view.method_71468(this.key, codec, this.getter.apply(object));
        }

        @Override
        <I extends class_9129> void read(I buf, class_9139<? super class_9129, M> packetCodec, T object) {
            M map = this.getter.apply(object);
            map.clear();
            map.putAll(packetCodec.decode(buf));
        }

        @Override
        <I extends class_9129> void write(I buf, class_9139<? super class_9129, M> packetCodec, T object) {
            packetCodec.encode(buf, this.getter.apply(object));
        }

    }
}
