/*
 * Decompiled with CFR 0.152.
 */
package brightspark.asynclocator;

import java.io.IOException;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.lang.invoke.CallSite;
import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class SparkConfig {
    private static final String CATEGORY_START = "#>";
    private static final String CATEGORY_END = "<#";
    private static final String COMMENT = "#";
    private static final String COMMENT_REGEX = "#\\s*";
    private static final String INTERNAL_COMMENT = "#~";
    private static final String INTERNAL_COMMENT_REGEX = "#~\\s*";
    private static final String INTERNAL_COMMENT_SEPARATOR = ", ";
    private static final String INTERNAL_COMMENT_SEPARATOR_REGEX = "\\s*,\\s*";
    private static final String DEFAULT = "default:";
    private static final String MIN = "min:";
    private static final String MAX = "max:";
    private static final String EQUALS = "=";
    private static final String NEW_LINE = System.lineSeparator();
    private static final String NEW_LINE_REGEX = "\\n";
    private static final Set<Class<?>> VALID_TYPES = Set.of(Boolean.TYPE, Boolean.class, Integer.TYPE, Integer.class, Float.TYPE, Float.class, String.class);
    private static final Logger LOG = LoggerFactory.getLogger(SparkConfig.class);

    public static void read(Path path, Class<?> configClass) throws IOException, IllegalAccessException {
        if (Files.notExists(path, new LinkOption[0])) {
            return;
        }
        Map<EntryKey, Entry> entries = SparkConfig.readEntries(path);
        List<ConfigField> configFields = SparkConfig.getConfigFields(configClass);
        for (ConfigField configField : configFields) {
            Field field = configField.field;
            String name = configField.getName();
            Entry entry = entries.get(new EntryKey(name, configField.getCategoryName()));
            if (entry != null) {
                Class<?> type = field.getType();
                Object newValue = SparkConfig.parseValueToType(entry.value, type);
                String internalComment = entry.internalComment;
                if (!internalComment.isBlank()) {
                    String[] parts;
                    String defaultValue = null;
                    String min = null;
                    String max = null;
                    for (String part : parts = internalComment.split(INTERNAL_COMMENT_SEPARATOR_REGEX)) {
                        int colonIndex = part.indexOf(":");
                        String partValue = part.substring(colonIndex + 1).trim();
                        if (part.startsWith(DEFAULT)) {
                            defaultValue = partValue;
                            continue;
                        }
                        if (part.startsWith(MIN)) {
                            min = partValue;
                            continue;
                        }
                        if (part.startsWith(MAX)) {
                            max = partValue;
                            continue;
                        }
                        LOG.warn("Invalid part of internal comment: " + part);
                    }
                    if (newValue == null && defaultValue != null) {
                        newValue = SparkConfig.parseValueToType(defaultValue, type);
                    }
                    if (SparkConfig.isLessThan(min, newValue)) {
                        throw new IllegalStateException("Config '" + name + " value '" + String.valueOf(newValue) + "' is less than the minimum of " + min);
                    }
                    if (SparkConfig.isGreaterThan(max, newValue)) {
                        throw new IllegalStateException("Config '" + name + " value '" + String.valueOf(newValue) + "' is greater than the maximum of " + max);
                    }
                }
                field.set(null, newValue);
                continue;
            }
            LOG.warn("Config '{}' has no value in file!", (Object)name);
        }
    }

    private static Map<EntryKey, Entry> readEntries(Path path) throws IOException {
        String internalComment = "";
        ArrayList<String> comments = new ArrayList<String>();
        HashMap<EntryKey, Entry> entries = new HashMap<EntryKey, Entry>();
        String category = null;
        for (String line : Files.readAllLines(path)) {
            if (line.isBlank() || line.startsWith("##")) continue;
            if (line.startsWith(CATEGORY_START) && line.endsWith(CATEGORY_END)) {
                category = line.substring(2, line.length() - 2).trim();
                continue;
            }
            if (line.startsWith(INTERNAL_COMMENT)) {
                internalComment = line.replaceFirst(INTERNAL_COMMENT_REGEX, "");
                continue;
            }
            if (line.startsWith(COMMENT)) {
                comments.add(line.replaceFirst(COMMENT_REGEX, ""));
                continue;
            }
            String[] parts = line.split(EQUALS, 2);
            if (parts.length != 2) {
                LOG.warn("Invalid config line -> '" + line + "'");
                continue;
            }
            String comment = comments.isEmpty() ? "" : String.join((CharSequence)NEW_LINE, comments);
            String name = parts[0].trim();
            String value = parts[1].trim();
            entries.put(new EntryKey(name, category), new Entry(name, value, category, comment, internalComment));
        }
        return entries;
    }

    public static void write(Path path, Class<?> configClass) throws IOException, IllegalAccessException {
        StringBuilder sb = new StringBuilder();
        ConfigCategory lastConfigCategory = null;
        List<ConfigField> configFields = SparkConfig.getConfigFields(configClass);
        for (ConfigField configField : configFields) {
            StringBuilder internalComment;
            StringBuilder comment;
            if (configField.category != lastConfigCategory) {
                lastConfigCategory = configField.category;
                sb.append((CharSequence)lastConfigCategory.getCategory());
            }
            if ((comment = configField.getComment()) != null) {
                sb.append((CharSequence)comment);
            }
            if ((internalComment = configField.getInternalComment()) != null) {
                sb.append((CharSequence)internalComment);
            }
            sb.append((CharSequence)configField.getConfig());
        }
        Files.writeString(path, (CharSequence)sb.toString(), new OpenOption[0]);
    }

    private static List<ConfigField> getConfigFields(Class<?> configClass) {
        LOG.debug("Getting config fields from class " + configClass.getName());
        ArrayList<ConfigField> configs = new ArrayList<ConfigField>();
        for (Field field : configClass.getDeclaredFields()) {
            ConfigField config = SparkConfig.createConfigField(null, field);
            if (config == null) continue;
            configs.add(config);
        }
        for (AnnotatedElement annotatedElement : configClass.getDeclaredClasses()) {
            if (Modifier.isPrivate(((Class)annotatedElement).getModifiers())) continue;
            Category category = ((Class)annotatedElement).isAnnotationPresent(Category.class) ? ((Class)annotatedElement).getAnnotation(Category.class) : null;
            ConfigCategory configCategory = new ConfigCategory((Class<?>)annotatedElement, category);
            for (Field field : ((Class)annotatedElement).getDeclaredFields()) {
                ConfigField config = SparkConfig.createConfigField(configCategory, field);
                if (config == null) continue;
                configs.add(config);
            }
        }
        LOG.debug("Got " + configs.size() + " configs");
        return configs;
    }

    private static ConfigField createConfigField(ConfigCategory configCategory, Field field) {
        int modifiers = field.getModifiers();
        if (Modifier.isPublic(modifiers) && Modifier.isStatic(modifiers)) {
            Class<?> type = field.getType();
            if (!VALID_TYPES.contains(type)) {
                LOG.warn("The config '{}' value type {} is not supported!", (Object)field.getName(), (Object)type.getName());
                return null;
            }
            Config config = field.isAnnotationPresent(Config.class) ? field.getAnnotation(Config.class) : null;
            ConfigField configField = new ConfigField(field, config, configCategory);
            LOG.debug("Found config: {}", (Object)configField);
            return configField;
        }
        return null;
    }

    private static Object parseValueToType(String value, Class<?> type) {
        if (value.isBlank() && !type.isPrimitive()) {
            return null;
        }
        if (value.isBlank() && (type == Integer.TYPE || type == Float.TYPE)) {
            return 0;
        }
        if (type == Boolean.TYPE || type == Boolean.class) {
            return Boolean.parseBoolean(value);
        }
        if (type == Integer.TYPE || type == Integer.class) {
            return Integer.parseInt(value);
        }
        if (type == Float.TYPE || type == Float.class) {
            return Float.valueOf(Float.parseFloat(value));
        }
        if (type == String.class) {
            return value;
        }
        throw new IllegalStateException("Config value '" + value + "' cannot be parsed to type " + type.getName());
    }

    private static boolean isLessThan(String min, Object value) {
        if (min == null || !(value instanceof Number)) {
            return false;
        }
        if (value instanceof Integer) {
            return (float)((Integer)value).intValue() < Float.parseFloat(min);
        }
        if (value instanceof Float) {
            return ((Float)value).floatValue() < Float.parseFloat(min);
        }
        return false;
    }

    private static boolean isGreaterThan(String max, Object value) {
        if (max == null || !(value instanceof Number)) {
            return false;
        }
        if (value instanceof Integer) {
            return (float)((Integer)value).intValue() > Float.parseFloat(max);
        }
        if (value instanceof Float) {
            return ((Float)value).floatValue() > Float.parseFloat(max);
        }
        return false;
    }

    private SparkConfig() {
    }

    private record ConfigField(Field field, Config config, ConfigCategory category) {
        String getName() {
            return this.config != null && !this.config.value().isBlank() ? this.config().value() : this.field.getName();
        }

        String getCategoryName() {
            return this.category != null ? this.category.getName() : null;
        }

        StringBuilder getComment() {
            if (this.config == null || this.config.comment().isBlank()) {
                return null;
            }
            String[] commentLines = this.config.comment().split(SparkConfig.NEW_LINE_REGEX);
            StringBuilder sb = new StringBuilder();
            for (String commentLine : commentLines) {
                sb.append(SparkConfig.COMMENT).append(" ").append(commentLine).append(NEW_LINE);
            }
            return sb;
        }

        StringBuilder getInternalComment() throws IllegalAccessException {
            boolean hasMax;
            boolean hasDefault = !this.field.getType().isPrimitive() && this.field.get(null) == null;
            boolean hasMin = this.config != null && this.config.min() != Float.MIN_NORMAL;
            boolean bl = hasMax = this.config != null && this.config.max() != Float.MAX_VALUE;
            if (!(hasDefault || hasMin || hasMax)) {
                return null;
            }
            StringBuilder sb = new StringBuilder();
            sb.append(SparkConfig.INTERNAL_COMMENT).append(" ");
            ArrayList<CallSite> internalCommentParts = new ArrayList<CallSite>();
            if (hasDefault) {
                internalCommentParts.add((CallSite)((Object)("default: " + String.valueOf(this.field.get(null)))));
            }
            if (hasMin) {
                internalCommentParts.add((CallSite)((Object)("min: " + this.config.min())));
            }
            if (hasMax) {
                internalCommentParts.add((CallSite)((Object)("max: " + this.config.max())));
            }
            sb.append(String.join((CharSequence)SparkConfig.INTERNAL_COMMENT_SEPARATOR, internalCommentParts)).append(NEW_LINE);
            return sb;
        }

        StringBuilder getConfig() throws IllegalAccessException {
            String name = this.getName();
            Object rawValue = this.field.get(null);
            String value = rawValue == null ? "" : rawValue.toString();
            StringBuilder sb = new StringBuilder();
            sb.append(name).append(" ").append(SparkConfig.EQUALS).append(" ").append(value).append(NEW_LINE).append(NEW_LINE);
            return sb;
        }
    }

    private record EntryKey(String name, String category) {
    }

    private record Entry(String name, String value, String category, String comment, String internalComment) {
    }

    private record ConfigCategory(Class<?> clazz, Category category) {
        String getName() {
            return this.category != null && !this.category.value().isBlank() ? this.category().value() : this.clazz.getSimpleName();
        }

        StringBuilder getCategory() {
            String name = this.getName();
            String border = SparkConfig.COMMENT.repeat(name.length() + 6);
            StringBuilder sb = new StringBuilder();
            sb.append(border).append(NEW_LINE);
            sb.append(SparkConfig.CATEGORY_START).append(" ").append(name).append(" ").append(SparkConfig.CATEGORY_END).append(NEW_LINE);
            sb.append(border).append(NEW_LINE).append(NEW_LINE);
            return sb;
        }
    }

    @Retention(value=RetentionPolicy.RUNTIME)
    @Target(value={ElementType.TYPE})
    public static @interface Category {
        public String value() default "";
    }

    @Retention(value=RetentionPolicy.RUNTIME)
    @Target(value={ElementType.FIELD})
    public static @interface Config {
        public String value() default "";

        public String comment() default "";

        public float min() default 1.1754944E-38f;

        public float max() default 3.4028235E38f;
    }
}

