package com.codex.onlyvanadv;

import com.codex.onlyvanadv.config.OvaConfig;
import net.fabricmc.api.ModInitializer;
import net.fabricmc.fabric.api.event.lifecycle.v1.ServerLifecycleEvents;
import net.minecraft.class_2960;
import net.minecraft.class_2989;
import net.minecraft.class_3298;
import net.minecraft.class_3300;
import net.minecraft.class_8781;
import net.minecraft.server.MinecraftServer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;

/**
 * Conservative filter that preserves vanilla advancements and UI layout.
 * - Never removes vanilla roots/chains: keeps all entries in the "minecraft" namespace
 *   whose parent chain also stays within the kept set.
 * - Removes all non-minecraft advancements (mods/datapacks) and any minecraft entries
 *   that depend on removed parents (prevents UI stacking/orphans).
 * - Runs only at server start and after successful data pack reload. No per-tick logic.
 */
public class OnlyVanillaAdvancements implements ModInitializer {
    public static final String MOD_ID = "only_vanilla_advancements";
    public static final Logger LOGGER = LoggerFactory.getLogger(MOD_ID);

    @Override
    public void onInitialize() {
        ServerLifecycleEvents.SERVER_STARTED.register(this::filterAdvancements);
        ServerLifecycleEvents.END_DATA_PACK_RELOAD.register((server, rm, success) -> {
            if (success) filterAdvancements(server);
        });
    }

    private void filterAdvancements(MinecraftServer server) {
        try {
            class_2989 loader = server.method_3851();
            var manager = loader.method_53646();
            // Load or create TOML config with comments
            OvaConfig cfg = OvaConfig.loadOrCreate();

            // Snapshot all nodes and ids
            Map<class_2960, class_8781> nodes = new HashMap<>();
            Set<class_2960> allIds = new HashSet<>();
            for (class_8781 pa : manager.method_53693()) {
                class_2960 id = pa.method_53649().comp_1919();
                nodes.put(id, pa);
                allIds.add(id);
            }

            // Build keep/remove sets from config
            Set<String> keepMods = new HashSet<>(cfg.kept_mods);
            Set<String> removeMods = new HashSet<>(cfg.removed_mods);
            Set<class_2960> keepIds = parseIdSet(cfg.kept_advancements);
            Set<class_2960> removeIds = parseIdSet(cfg.removed_advancements);

            // Expand kept ids upward to include parents if requested
            if (cfg.keep_parent_advancements) {
                Deque<class_2960> stack = new ArrayDeque<>(keepIds);
                while (!stack.isEmpty()) {
                    class_2960 id = stack.pop();
                    class_8781 pa = nodes.get(id);
                    if (pa == null) continue;
                    class_8781 parent = pa.method_53651();
                    if (parent == null) continue;
                    class_2960 pid = parent.method_53649().comp_1919();
                    if (!keepIds.contains(pid)) { keepIds.add(pid); stack.push(pid); }
                }
            }

            class_3300 rm = server.method_34864();
            // Origin-aware removal set
            Set<class_2960> toRemove = new HashSet<>();
            for (class_2960 id : allIds) {
                boolean isKept = keepIds.contains(id) || keepMods.contains(id.method_12836());
                boolean isMinecraft = "minecraft".equals(id.method_12836());
                if (isKept) continue; // never remove kept

                if (!isMinecraft) {
                    // Non-vanilla default: remove unless explicitly kept
                    toRemove.add(id);
                    continue;
                }

                // minecraft:* -> check if effective resource comes from built-in "vanilla" pack
                boolean isTrueVanilla = isEffectiveVanilla(rm, id);

                if (!isTrueVanilla) {
                    // This is an override from a datapack/mod; remove unless explicitly kept
                    if (!(keepIds.contains(id) || keepMods.contains("minecraft"))) {
                        toRemove.add(id);
                    }
                }
            }

            // Expand kept ids with their parent chain if configured
            // Prune orphans: remove anything whose parent is not present after rules
            boolean changed;
            do {
                changed = false;
                for (class_2960 id : new HashSet<>(allIds)) {
                    if (toRemove.contains(id)) continue;
                    // never prune vanilla or explicitly kept entries, even if orphaned
                    if ("minecraft".equals(id.method_12836()) || keepMods.contains(id.method_12836()) || keepIds.contains(id)) {
                        continue;
                    }
                    class_8781 pa = nodes.get(id);
                    if (pa == null) continue;
                    class_8781 parent = pa.method_53651();
                    if (parent == null) continue; // root ok
                    class_2960 pid = parent.method_53649().comp_1919();
                    if (toRemove.contains(pid)) {
                        toRemove.add(id);
                        changed = true;
                    }
                }
            } while (changed);

            if (!toRemove.isEmpty()) {
                manager.method_713(toRemove);
                server.method_3760().method_14571().forEach(p -> p.method_14236().method_12886(loader));
                LOGGER.info("OnlyVanillaAdvancements: removed {} advancements (after rules).", toRemove.size());
            } else {
                LOGGER.info("OnlyVanillaAdvancements: no removals needed.");
            }
        } catch (Throwable t) {
            LOGGER.error("OnlyVanillaAdvancements: filtering failed", t);
        }
    }

    private String safePackId(class_3298 res) {
        try { return res.method_14480(); } catch (Throwable t) { return ""; }
    }

    private boolean isEffectiveVanilla(class_3300 rm, class_2960 advId) {
        try {
            String resPath = "advancements/" + advId.method_12832() + ".json";
            class_2960 resId = class_2960.method_60655("minecraft", resPath);
            java.util.Optional<class_3298> opt = rm.method_14486(resId);
            if (opt.isEmpty()) {
                // If we can’t resolve, fail-open and keep (avoid nuking vanilla)
                return true;
            }
            String pid = safePackId(opt.get());
            return isBuiltinPack(pid);
        } catch (Throwable t) {
            // Fail-open as vanilla to avoid accidental removal
            return true;
        }
    }

    private boolean isBuiltinPack(String pid) {
        String p = pid == null ? "" : pid.toLowerCase();
        // Accept common built-in identifiers defensively across launcher variants
        return p.equals("vanilla") || p.equals("minecraft") || p.contains("builtin") || p.contains("default");
    }

    private Set<class_2960> parseIdSet(Collection<String> raw) {
        Set<class_2960> out = new HashSet<>();
        if (raw == null) return out;
        for (String s : raw) {
            if (s == null || s.isBlank()) continue;
            class_2960 id = class_2960.method_12829(s.trim());
            if (id != null) out.add(id);
        }
        return out;
    }
}
