package rege.rege.minecraftmod.searchworldenhance.util;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map.Entry;
import java.util.function.BiPredicate;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import net.minecraft.class_2847;
import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import rege.rege.minecraftmod.NoMc;
import rege.rege.misc.searchworldenhance.util.BoolItem;
import rege.rege.misc.searchworldenhance.util.EnumItem;
import rege.rege.misc.searchworldenhance.util.NumberItem;
import rege.rege.misc.searchworldenhance.util.RegexItem;

public abstract class SearchKeyRegistry {
    private static final ArrayList<@NotNull String> TRUE_STRINGS =
    new ArrayList<>(5);
    private static final ArrayList<@NotNull String> FALSE_STRINGS =
    new ArrayList<>(5);
    private static final HashMap<@NotNull String, @NotNull BiPredicate<
        @NotNull RegexItem, class_2847
    >> REGEX_REGISTRIES = new HashMap<>();
    private static final HashMap<@NotNull String, @NotNull BiPredicate<
        @NotNull NumberItem<Long>, class_2847
    >> LONG_REGISTRIES = new HashMap<>();
    private static final HashMap<@NotNull String, Entry<
        Class<Enum<?>>, @NotNull BiPredicate<@NotNull EnumItem<?>, class_2847>
    >> ENUM_REGISTRIES = new HashMap<>();
    private static final HashMap<@NotNull String, @NotNull BiPredicate<
        @NotNull BoolItem, class_2847
    >> BOOL_REGISTRIES = new HashMap<>();

    @NoMc
    public static String validateKey(@NotNull String key) {
        if (key.isEmpty()) {
            throw new IllegalArgumentException("Keys must be not empty");
        }
        if (key.indexOf('-') != -1 || key.indexOf(' ') != -1 ||
            key.indexOf('=') != -1 || key.indexOf('!') != -1 ||
            key.indexOf('>') != -1 || key.indexOf('<') != -1 ||
            key.indexOf(',') != -1 || key.indexOf('/') != -1 ||
            key.indexOf('"') != -1 || key.indexOf('\\') != -1) {
            throw new IllegalArgumentException(
                "Keys cannot contain space and -=!><,/\"\\"
            );
        }
        return key;
    }

    public static void registerRegex(@NotNull String key, @NotNull BiPredicate<
        @NotNull RegexItem, class_2847
    > predicate) {
        REGEX_REGISTRIES.put(validateKey(key), predicate);
    }

    public static void registerLong(@NotNull String key, @NotNull BiPredicate<
        @NotNull NumberItem<Long>, class_2847
    > predicate) {
        LONG_REGISTRIES.put(validateKey(key), predicate);
    }

    @SuppressWarnings({"rawtypes", "unchecked"})
    public static <T extends Enum<T>> void registerEnum(
        @NotNull String key, @NotNull Class<T> clazz,
        @NotNull BiPredicate<@NotNull EnumItem<T>, class_2847> predicate
    ) {
        final Entry<Class<T>, BiPredicate<EnumItem<T>, class_2847>> ENTRY =
        new Entry<Class<T>, BiPredicate<EnumItem<T>, class_2847>>() {
            @Override
            @Contract(pure = true)
            @NotNull
            public Class<T> getKey() {
                return clazz;
            }

            @Override
            @Contract(pure = true)
            @NotNull
            public BiPredicate<EnumItem<T>, class_2847> getValue() {
                return predicate;
            }

            @Override
            @Contract("_ -> fail")
            public BiPredicate<EnumItem<T>, class_2847> setValue(
                BiPredicate<EnumItem<T>, class_2847>
                enumItemclass2847BiPredicate
            ) {
                throw new UnsupportedOperationException();
            }
        };
        ENUM_REGISTRIES.put(validateKey(key), (Entry)ENTRY);
    }

    public static void registerBool(@NotNull String key, @NotNull BiPredicate<
        @NotNull BoolItem, class_2847
    > predicate) {
        BOOL_REGISTRIES.put(validateKey(key), predicate);
    }

    @SuppressWarnings({"rawtypes", "unchecked"})
    public static @Nullable Predicate<class_2847>
    evaluate(@NotNull String str) throws IndexOutOfBoundsException {
        int ptr = 0;
        Predicate<class_2847> result = null;
        while (true) {
            StringBuilder sb = new StringBuilder();
            while ("- =!><,/\"\\".indexOf(str.charAt(ptr)) == -1) {
                sb.append(str.charAt(ptr));
                ptr++;
            }
            final String KEY = sb.toString();
            try {
                validateKey(KEY);
            } catch (IllegalArgumentException e) {
                return null;
            }
            if (REGEX_REGISTRIES.containsKey(KEY)) {
                boolean negated = false;
                boolean ignoreCase = true;
                boolean isRegex = false;
                boolean end = false;
                while (!end) {
                    switch ("=!-/".indexOf(str.charAt(ptr))) {
                        case 0:
                            end = true;
                            break;
                        case 1:
                            negated = !negated;
                            break;
                        case 2:
                            ignoreCase = false;
                            break;
                        case 3:
                            isRegex = true;
                            break;
                        default: return null;
                    }
                    ptr++;
                }
                if (str.charAt(ptr) == '"') {
                    ptr++;
                    sb = new StringBuilder();
                    boolean escaped = false;
                    while (true) {
                        char ach = str.charAt(ptr);
                        if (escaped) {
                            escaped = false;
                            sb.append(ach);
                        } else if (ach == '\\') {
                            escaped = true;
                        } else if (ach == '"') {
                            ptr++;
                            break;
                        }
                        ptr++;
                    }
                    final String PATTERN = sb.toString();
                    if (isRegex) {
                        try {
                            Pattern.compile(PATTERN);
                        } catch (PatternSyntaxException e) {
                            return null;
                        }
                    }
                    final RegexItem ITEM =
                    new RegexItem(PATTERN, ignoreCase, isRegex, negated);
                    Predicate<class_2847> predicate =
                    entry -> REGEX_REGISTRIES.get(KEY).test(ITEM, entry);
                    result =
                    (result == null) ? predicate : result.and(predicate);
                    if (ptr >= str.length()) {
                        return result;
                    }
                    if (str.charAt(ptr) != ',') {
                        return null;
                    }
                    ptr++;
                } else {
                    sb = new StringBuilder();
                    while (true) {
                        try {
                            char ach = str.charAt(ptr);
                            if (ach == ',') {
                                ptr++;
                                break;
                            } else {
                                sb.append(ach);
                            }
                            ptr++;
                        } catch (IndexOutOfBoundsException e) {
                            break;
                        }
                    }
                    final String PATTERN = sb.toString();
                    if (isRegex) {
                        try {
                            Pattern.compile(PATTERN);
                        } catch (PatternSyntaxException e) {
                            return null;
                        }
                    }
                    final RegexItem ITEM =
                    new RegexItem(PATTERN, ignoreCase, isRegex, negated);
                    Predicate<class_2847> predicate =
                    entry -> REGEX_REGISTRIES.get(KEY).test(ITEM, entry);
                    result =
                    (result == null) ? predicate : result.and(predicate);
                }
            } else if (LONG_REGISTRIES.containsKey(KEY)) {
                char mark = ' ';
                boolean hasEq = false;
                switch (str.charAt(ptr)) {
                    case '=':
                        mark = '=';
                        break;
                    case '>':
                        mark = '>';
                        break;
                    case '<':
                        mark = '<';
                        break;
                    case '!':
                        mark = '!';
                        break;
                    default: return null;
                }
                ptr++;
                if (str.charAt(ptr) == '=') {
                    hasEq = true;
                    ptr++;
                }
                sb = new StringBuilder();
                while (true) {
                    try {
                        char ach = str.charAt(ptr);
                        if (ach == ',') {
                            ptr++;
                            break;
                        } else {
                            sb.append(ach);
                        }
                        ptr++;
                    } catch (IndexOutOfBoundsException e) {
                        break;
                    }
                }
                try {
                    final long PATTERN = Long.parseLong(sb.toString());
                    final NumberItem<Long> ITEM = new NumberItem<>(
                        PATTERN, mark == '!' || mark == '<',
                        mark == '=' || (mark != '!' && hasEq),
                        mark == '!' || mark == '>'
                    );
                    Predicate<class_2847> predicate =
                    entry -> LONG_REGISTRIES.get(KEY).test(ITEM, entry);
                    result =
                    (result == null) ? predicate : result.and(predicate);
                } catch (NumberFormatException e) {
                    return null;
                }
            } else if (ENUM_REGISTRIES.containsKey(KEY)) {
                boolean negated = false;
                if (str.charAt(ptr) == '!') {
                    negated = true;
                    ptr++;
                }
                if (str.charAt(ptr) != '=') {
                    return null;
                }
                ptr++;
                sb = new StringBuilder();
                while (true) {
                    try {
                        char ach = str.charAt(ptr);
                        if (ach == ',') {
                            ptr++;
                            break;
                        } else {
                            sb.append(ach);
                        }
                        ptr++;
                    } catch (IndexOutOfBoundsException e) {
                        break;
                    }
                }
                try {
                    final String VALUE = sb.toString();
                    final Enum<?> PATTERN =
                    "null".equals(VALUE) ? null : Enum.valueOf(
                        (Class)ENUM_REGISTRIES.get(KEY).getKey(), VALUE
                    );
                    final EnumItem<?> ITEM =
                    new EnumItem<>((Enum)PATTERN, negated);
                    Predicate<class_2847> predicate =
                    entry -> ENUM_REGISTRIES.get(KEY).getValue()
                             .test(ITEM, entry);
                    result =
                    (result == null) ? predicate : result.and(predicate);
                } catch (IllegalArgumentException e) {
                    return null;
                }
            } else if (BOOL_REGISTRIES.containsKey(KEY)) {
                if (str.charAt(ptr) != '=') {
                    return null;
                }
                ptr++;
                sb = new StringBuilder();
                while (true) {
                    try {
                        char ach = str.charAt(ptr);
                        if (ach == ',') {
                            ptr++;
                            break;
                        } else {
                            sb.append(ach);
                        }
                        ptr++;
                    } catch (IndexOutOfBoundsException e) {
                        break;
                    }
                }
                final String VALUE = sb.toString().toLowerCase();
                if (TRUE_STRINGS.contains(VALUE)) {
                    final BoolItem ITEM = new BoolItem(true);
                    Predicate<class_2847> predicate =
                    entry -> BOOL_REGISTRIES.get(KEY).test(ITEM, entry);
                    result =
                    (result == null) ? predicate : result.and(predicate);
                } else if (FALSE_STRINGS.contains(VALUE)) {
                    final BoolItem ITEM = new BoolItem(false);
                    Predicate<class_2847> predicate =
                    entry -> BOOL_REGISTRIES.get(KEY).test(ITEM, entry);
                    result =
                    (result == null) ? predicate : result.and(predicate);
                } else {
                    return null;
                }
            } else {
                return null;
            }
            if (ptr >= str.length()) {
                return result;
            }
        }
    }

    @Contract("-> fail")
    private SearchKeyRegistry() {
        throw new UnsupportedOperationException();
    }

    static {
        TRUE_STRINGS.add("true");
        TRUE_STRINGS.add("yes");
        TRUE_STRINGS.add("t");
        TRUE_STRINGS.add("y");
        TRUE_STRINGS.add("1");
        FALSE_STRINGS.add("false");
        FALSE_STRINGS.add("no");
        FALSE_STRINGS.add("f");
        FALSE_STRINGS.add("n");
        FALSE_STRINGS.add("0");
    }
}
