package com.codex.realseasons.server;

import com.codex.realseasons.RealSeasonsSharedData;
import com.codex.realseasons.calendar.RealSeasonsCalendarService;
import com.codex.realseasons.calendar.RealSeasonsCalendarService.RealSeasonsCalendarSnapshot;
import com.codex.realseasons.config.RealSeasonsCommonConfig;
import net.fabricmc.fabric.api.event.lifecycle.v1.ServerLifecycleEvents;
import net.fabricmc.fabric.api.event.lifecycle.v1.ServerTickEvents;
import net.minecraft.class_1928;
import net.minecraft.class_1937;
import net.minecraft.class_3218;
import net.minecraft.class_5321;
import net.minecraft.server.MinecraftServer;
import sereneseasons.api.SSGameRules;
import sereneseasons.init.ModConfig;
import sereneseasons.season.SeasonHandler;
import sereneseasons.season.SeasonSavedData;
import sereneseasons.season.SeasonTime;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Supplier;

/**
 * Keeps Serene Seasons' world data aligned with the real-world calendar.
 */
public final class RealSeasonsSeasonSynchronizer {
    private final Supplier<RealSeasonsCommonConfig> configSupplier;
    private final RealSeasonsCalendarService calendarService;
    private final RealSeasonsSeasonStateStore stateStore;
    private final RealSeasonsDisplayLinker displayLinker;

    private final Map<class_5321<class_1937>, Integer> lastBroadcastTicks = new HashMap<>();
    private final Set<class_5321<class_1937>> disabledCycleRule = new HashSet<>();

    public RealSeasonsSeasonSynchronizer(Supplier<RealSeasonsCommonConfig> configSupplier,
                                         RealSeasonsCalendarService calendarService,
                                         RealSeasonsSeasonStateStore stateStore) {
        this.configSupplier = Objects.requireNonNull(configSupplier, "configSupplier");
        this.calendarService = Objects.requireNonNull(calendarService, "calendarService");
        this.stateStore = Objects.requireNonNull(stateStore, "stateStore");
        this.displayLinker = new RealSeasonsDisplayLinker();
    }

    public void register() {
        ServerLifecycleEvents.SERVER_STARTED.register(server -> syncAll(server, true));
        ServerLifecycleEvents.SERVER_STOPPED.register(server -> resetCachedState());
        ServerTickEvents.END_SERVER_TICK.register(server -> syncAll(server, false));
    }

    public void resetCachedState() {
        lastBroadcastTicks.clear();
        disabledCycleRule.clear();
    }

    private void syncAll(MinecraftServer server, boolean forceBroadcast) {
        RealSeasonsCommonConfig config = configSupplier.get();
        // Ensure Serene Seasons' UI day count tracks our cadence (MONTH/WEEK/DAY)
        displayLinker.maybeApply(server, config);
        RealSeasonsCalendarSnapshot snapshot = calendarService.snapshot(config);
        SeasonTime desiredState = snapshot.seasonTime();
        stateStore.update(desiredState);

        for (class_3218 level : server.method_3738()) {
            if (!ModConfig.seasons.isDimensionWhitelisted(level.method_27983())) {
                continue;
            }

            disableSeasonCycle(level);
            applySeasonState(level, desiredState, forceBroadcast);
        }
    }

    private void disableSeasonCycle(class_3218 level) {
        if (disabledCycleRule.contains(level.method_27983())) {
            return;
        }

        class_1928.class_4310 rule = level.method_64395().method_20746(SSGameRules.RULE_DOSEASONCYCLE);
        if (rule != null && rule.method_20753()) {
            rule.method_20758(false, level.method_8503());
            RealSeasonsSharedData.LOGGER.info("Disabled doSeasonCycle game rule in {} to allow real-time season syncing.", level.method_27983().method_29177());
        }
        disabledCycleRule.add(level.method_27983());
    }

    private void applySeasonState(class_3218 level, SeasonTime desiredState, boolean forceBroadcast) {
        SeasonSavedData savedData = SeasonHandler.getSeasonSavedData(level);
        if (savedData == null) {
            return;
        }

        int targetTick = desiredState.getSeasonCycleTicks();
        boolean changed = savedData.seasonCycleTicks != targetTick;
        if (changed) {
            savedData.seasonCycleTicks = targetTick;
            savedData.method_80();
        }

        SeasonHandler.prevServerSeasonCycleTicks.put(level.method_27983(), targetTick);

        Integer previous = lastBroadcastTicks.get(level.method_27983());
        if (forceBroadcast || changed || previous == null || previous != targetTick) {
            SeasonHandler.sendSeasonUpdate(level);
            lastBroadcastTicks.put(level.method_27983(), targetTick);
        }
    }
}
