package net.litetex.capes.handler;

import java.io.IOException;
import java.net.Proxy;
import java.net.ProxySelector;
import java.net.SocketAddress;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.mojang.authlib.GameProfile;

import net.litetex.capes.Capes;
import net.litetex.capes.config.AnimatedCapesHandling;
import net.litetex.capes.handler.textures.DefaultTextureResolver;
import net.litetex.capes.handler.textures.TextureResolver;
import net.litetex.capes.provider.CapeProvider;
import net.litetex.capes.provider.ResolvedTextureInfo;
import net.litetex.capes.util.CapeProviderTextureAsset;
import net.minecraft.class_1011;
import net.minecraft.class_1043;
import net.minecraft.class_1060;
import net.minecraft.class_12079;
import net.minecraft.class_2960;
import net.minecraft.class_310;


@SuppressWarnings({"checkstyle:MagicNumber", "PMD.GodClass"})
public class PlayerCapeHandler
{
	private static final Logger LOG = LoggerFactory.getLogger(PlayerCapeHandler.class);
	
	private final Capes capes;
	private final GameProfile profile;
	private Optional<TextureProvider> optTextureProvider = Optional.empty();
	private boolean hasElytraTexture = true;
	
	public PlayerCapeHandler(final Capes capes, final GameProfile profile)
	{
		this.capes = capes;
		this.profile = profile;
	}
	
	public Optional<TextureProvider> capeTextureProvider()
	{
		return this.optTextureProvider;
	}
	
	public class_12079.class_12081 getCape()
	{
		final TextureProvider textureProvider = this.optTextureProvider.orElse(null);
		if(textureProvider != null)
		{
			return textureProvider.texture();
		}
		return null;
	}
	
	public void resetCape()
	{
		this.optTextureProvider = Optional.empty();
		this.hasElytraTexture = true;
	}
	
	public boolean trySetCape(final CapeProvider capeProvider)
	{
		final String profileApiUrl = capeProvider.getBaseUrl(this.profile);
		if(profileApiUrl == null)
		{
			return false;
		}

		try
		{
			final HttpClient.Builder clientBuilder = this.createBuilder();

			// Request the profile API (not the texture yet)
			final HttpRequest.Builder requestBuilder = HttpRequest.newBuilder(URI.create(profileApiUrl))
				.timeout(Duration.ofSeconds(10))
				.header("User-Agent", "CP");

			final ResolvedTextureInfo resolvedTextureInfo =
				capeProvider.resolveTexture(clientBuilder, requestBuilder, this.profile);
			if(resolvedTextureInfo == null)
			{
				return false;
			}

			byte[] imageBytes = resolvedTextureInfo.imageBytes();
			String textureUrlForLog = profileApiUrl;

			// If the resolved texture info is a UrlTextureInfo, fetch the image from the URL
			if (resolvedTextureInfo instanceof ResolvedTextureInfo.UrlTextureInfo urlInfo && urlInfo.textureURL() != null) {
				textureUrlForLog = urlInfo.textureURL();
				try {
					HttpClient imageClient = clientBuilder.build();
					HttpRequest imageRequest = HttpRequest.newBuilder(URI.create(urlInfo.textureURL()))
						.timeout(Duration.ofSeconds(10))
						.header("User-Agent", "CP")
						.build();
					HttpResponse<byte[]> imageResponse = imageClient.send(imageRequest, HttpResponse.BodyHandlers.ofByteArray());
					if (imageResponse.statusCode() / 100 == 2) {
						imageBytes = imageResponse.body();
					} else {
						LOG.warn("Failed to fetch cape image[url='{}',profileId='{}',statusCode='{}']", urlInfo.textureURL(), this.profile.id(), imageResponse.statusCode());
						return false;
					}
				} catch (Exception e) {
					LOG.warn("Failed to fetch cape image[url='{}',profileId='{}']", urlInfo.textureURL(), this.profile.id(), e);
					return false;
				}
			}

			if(imageBytes == null)
			{
				return false;
			}

			if(this.isCapeBlocked(capeProvider, imageBytes))
			{
				return false;
			}

			final TextureResolver textureResolver = this.capes.getAllTextureResolvers()
				.getOrDefault(resolvedTextureInfo.textureResolverId(), DefaultTextureResolver.INSTANCE);

			final AnimatedCapesHandling animatedCapesHandling = this.animatedCapesHandling();
			if(textureResolver.animated() && animatedCapesHandling == AnimatedCapesHandling.OFF)
			{
				return false;
			}

			this.optTextureProvider = this.registerTexturesAndGetProvider(
				this.determineTexturesToRegister(
					textureResolver,
					imageBytes,
					animatedCapesHandling == AnimatedCapesHandling.FROZEN,
					textureUrlForLog));

			return this.optTextureProvider.isPresent();
		}
		catch(final InterruptedException iex)
		{
			LOG.warn("Got interrupted[url='{}',profileId='{}']", profileApiUrl, this.profile.id(), iex);
			Thread.currentThread().interrupt();
		}
		catch(final Exception ex)
		{
			LOG.warn("Failed to process texture[url='{}',profileId='{}']", profileApiUrl, this.profile.id(), ex);
		}

		this.resetCape();
		return false;
	}
	
	private HttpClient.Builder createBuilder()
	{
		final HttpClient.Builder clientBuilder = HttpClient.newBuilder()
			.connectTimeout(Duration.ofSeconds(10));
		final Proxy proxy = class_310.method_1551().method_1487();
		if(proxy != null)
		{
			clientBuilder.proxy(new ProxySelector()
			{
				@Override
				public List<Proxy> select(final URI uri)
				{
					return List.of(proxy);
				}
				
				@Override
				public void connectFailed(final URI uri, final SocketAddress sa, final IOException ioe)
				{
					// Ignore
				}
			});
		}
		return clientBuilder;
	}
	
	private boolean isCapeBlocked(final CapeProvider provider, final byte[] imageBytes)
	{
		final Set<Integer> blockedCapeHashes = this.capes.blockedProviderCapeHashes().get(provider);
		if(blockedCapeHashes == null)
		{
			return false;
		}
		
		return blockedCapeHashes.contains(Arrays.hashCode(imageBytes));
	}
	
	private List<TextureToRegister> determineTexturesToRegister(
		final TextureResolver textureResolver,
		final byte[] imageData,
		final boolean freezeAnimation,
		final String url
	) throws IOException
	{
		final TextureResolver.ResolvedTextureData resolved = textureResolver.resolve(
			imageData,
			freezeAnimation);
		this.hasElytraTexture = !Boolean.FALSE.equals(resolved.hasElytra()); // if null -> default to true
		
		if(resolved instanceof final TextureResolver.DefaultResolvedTextureData defaultResolvedTextureData)
		{
			return List.of(new TextureToRegister(
				identifier(this.uuid().toString()),
				defaultResolvedTextureData.texture()));
		}
		else if(resolved instanceof final TextureResolver.AnimatedResolvedTextureData animatedResolvedTextureData)
		{
			final List<AnimatedNativeImageContainer> textures = animatedResolvedTextureData.textures();
			Stream<AnimatedNativeImageContainer> animatedTextureStream = textures.stream();
			
			if(textures.isEmpty())
			{
				LOG.warn(
					"Received animated texture with no frames[url='{}',profileId='{}']",
					url,
					this.uuid());
				return List.of();
			}
			
			if(freezeAnimation)
			{
				animatedTextureStream = animatedTextureStream.limit(1);
			}
			
			final AtomicInteger counter = new AtomicInteger(0);
			return animatedTextureStream
				.map(c -> new TextureToRegister(
					identifier(this.uuid() + (!freezeAnimation ? "/" + counter.getAndIncrement() : "")),
					c.image(),
					c.delayMs()))
				.toList();
		}
		throw new IllegalStateException("Unexpected ResolvedTextureData: " + resolved.getClass().getSimpleName());
	}
	
	private Optional<TextureProvider> registerTexturesAndGetProvider(
		final List<TextureToRegister> texturesToRegister)
	{
		if(texturesToRegister.isEmpty())
		{
			return Optional.empty();
		}
		
		final class_1060 textureManager = class_310.method_1551().method_1531();
		// Do texturing work NOT on Render thread
		CompletableFuture.runAsync(
				() -> texturesToRegister.forEach(t ->
					textureManager.method_4616(
						t.identifier(),
						new class_1043(t.identifier()::toString, t.image()))),
				class_310.method_1551())
			.exceptionally(ex -> {
				LOG.warn("Failed to register textures", ex);
				return null;
			});
		
		return Optional.of(texturesToRegister.size() == 1
			? new DefaultTextureProvider(texturesToRegister.getFirst().identifier())
			: new AnimatedTextureProvider(texturesToRegister));
	}
	
	record TextureToRegister(
		class_2960 identifier,
		class_1011 image,
		int delayMs
	)
	{
		public TextureToRegister(final class_2960 identifier, final class_1011 image)
		{
			this(identifier, image, 100);
		}
	}
	
	private AnimatedCapesHandling animatedCapesHandling()
	{
		return this.capes.config().getAnimatedCapesHandling();
	}
	
	static class_2960 identifier(final String id)
	{
		return class_2960.method_60655(Capes.MOD_ID, id);
	}
	
	// region Getter
	
	public UUID uuid()
	{
		return this.profile.id();
	}
	
	public boolean hasElytraTexture()
	{
		return this.hasElytraTexture;
	}
	
	// endregion
	
	
	record DefaultTextureProvider(CapeProviderTextureAsset texture) implements TextureProvider
	{
		DefaultTextureProvider(final class_2960 id)
		{
			this(new CapeProviderTextureAsset(id));
		}
		
		@Override
		public boolean dynamicIdentifier()
		{
			return false;
		}
	}
	
	
	static class AnimatedTextureProvider implements TextureProvider
	{
		private final List<IdentifierContainer> identifiers;
		private int lastFrameIndex;
		private long nextFrameTime;
		
		AnimatedTextureProvider(final Collection<TextureToRegister> identifiers)
		{
			this.identifiers = identifiers.stream()
				.map(t -> new IdentifierContainer(
					new CapeProviderTextureAsset(t.identifier()),
					Math.clamp(
						t.delayMs(),
						1,
						// 1min
						60 * 1_000)))
				.toList();
		}
		
		@Override
		public class_12079.class_12081 texture()
		{
			final long time = System.currentTimeMillis();
			if(time > this.nextFrameTime)
			{
				final int thisFrameIndex = (this.lastFrameIndex + 1) % this.identifiers.size();
				this.lastFrameIndex = thisFrameIndex;
				
				final IdentifierContainer ic = this.identifiers.get(thisFrameIndex);
				this.nextFrameTime = time + ic.delay();
				
				return ic.identifier();
			}
			return this.identifiers.get(this.lastFrameIndex).identifier();
		}
		
		@Override
		public boolean dynamicIdentifier()
		{
			return true;
		}
		
		record IdentifierContainer(
			class_12079.class_12081 identifier,
			int delay)
		{
		}
	}
}
