/*
 * Copyright (c) 2025 macuguita
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE
 * OR OTHER DEALINGS IN THE SOFTWARE.
 */

package com.macuguita.backpacks.client.gui;

import com.macuguita.backpacks.client.gui.payload.BackpackInventoryPayload;
import com.macuguita.backpacks.client.gui.slots.BackpackSlot;
import com.macuguita.backpacks.client.gui.slots.CustomSlot;
import com.macuguita.backpacks.common.GuitaBackpacks;
import com.macuguita.backpacks.common.reg.GBItemTags;
import com.macuguita.backpacks.common.utils.BackpackUtils;
import com.macuguita.backpacks.common.utils.EquipmentUtils;
import net.minecraft.class_1263;
import net.minecraft.class_1277;
import net.minecraft.class_1657;
import net.minecraft.class_1661;
import net.minecraft.class_1703;
import net.minecraft.class_1735;
import net.minecraft.class_1799;
import org.jetbrains.annotations.NotNull;

public class BackpackScreenHandler extends class_1703 {

	private static final int VISIBLE_ROWS = 6;
	private static final int SLOT_SIZE = 18;
	private static final int GAP_BETWEEN_BACKPACK_AND_PLAYER = 13;
	private static final int TOP_PADDING = 14;
	private static final int SIDE_PADDING = 7;

	public final class_1263 inventory;
	public final class_1799 backpack;
	public final int slotIndex;
	private final int totalRows;
	private final int backpackStartY;
	private final int playerInventoryStartY;
	private int scrollOffset = 0;

	// TODO: should probably migrate to PropertyDelegates https://wiki.fabricmc.net/tutorial:propertydelegates
	public BackpackScreenHandler(int syncId, class_1661 playerInventory, @NotNull BackpackInventoryPayload buf) {
		this(syncId, playerInventory, new class_1277(buf.backpackSize()), buf.slotIndex());
	}

	public BackpackScreenHandler(int syncId, @NotNull class_1661 playerInventory, class_1263 inventory, int slotIndex) {
		super(GuitaBackpacks.BACKPACK_SCREEN_HANDLER, syncId);
		method_17359(inventory, inventory.method_5439());
		this.slotIndex = slotIndex;
		this.backpack = EquipmentUtils.getBackpackFromSlotIndex(playerInventory.field_7546, slotIndex);
		this.inventory = inventory;
		this.totalRows = (int) Math.ceil((double) inventory.method_5439() / 9.0);

		int visibleRows = Math.min(totalRows, VISIBLE_ROWS);
		int backpackHeight = visibleRows * field_52558;

		this.backpackStartY = TOP_PADDING;
		this.playerInventoryStartY = backpackStartY + backpackHeight + GAP_BETWEEN_BACKPACK_AND_PLAYER;

		inventory.method_5435(playerInventory.field_7546);

		for (int i = 0; i < inventory.method_5439(); i++) {
			this.method_7621(new BackpackSlot(inventory, i, -1000, -1000, backpack, GBItemTags.BACKPACK_BLACKLIST));
		}

		updateSlotPositions();

		for (int row = 0; row < 3; row++) {
			for (int col = 0; col < 9; col++) {
				int index = col + row * 9 + 9;
				this.method_7621(new BackpackSlot(playerInventory, index, SIDE_PADDING + col * field_52558,
						playerInventoryStartY + row * field_52558, backpack));
			}
		}

		int hotbarY = playerInventoryStartY + 3 * field_52558 + 4;
		for (int col = 0; col < 9; col++) {
			this.method_7621(new BackpackSlot(playerInventory, col, SIDE_PADDING + col * field_52558, hotbarY, backpack));
		}
	}

	public void scroll(int direction) {
		int maxScroll = Math.max(0, totalRows - VISIBLE_ROWS);
		int oldOffset = scrollOffset;

		scrollOffset = Math.max(0, Math.min(maxScroll, scrollOffset + direction));
		if (oldOffset != scrollOffset) updateSlotPositions();
	}

	public void updateSlotPositions() {
		int startIndex = scrollOffset * 9;
		int totalSlots = inventory.method_5439();
		int visibleSlots = Math.min(VISIBLE_ROWS * 9, totalSlots - startIndex);

		for (int i = 0; i < visibleSlots; i++) {
			int slotIndex = startIndex + i;
			CustomSlot slot = (CustomSlot) this.field_7761.get(slotIndex);

			int row = i / 9;
			int col = i % 9;

			int x = SIDE_PADDING + col * field_52558;
			int y = backpackStartY + row * field_52558;

			slot.gbackpacks$setX(x);
			slot.gbackpacks$setY(y);
		}

		for (int i = 0; i < totalSlots; i++) {
			if (i < startIndex || i >= startIndex + visibleSlots) {
				CustomSlot slot = (CustomSlot) this.field_7761.get(i);
				slot.gbackpacks$setX(-1000);
				slot.gbackpacks$setY(-1000);
			}
		}
	}

	public int getScrollOffset() {
		return scrollOffset;
	}

	public int getTotalRows() {
		return totalRows;
	}

	public int getPlayerInventoryStartY() {
		return playerInventoryStartY;
	}

	public int getBackpackStartY() {
		return backpackStartY;
	}

	public boolean needsScrolling() {
		return totalRows > VISIBLE_ROWS;
	}

	@Override
	public boolean method_7597(class_1657 player) {
		return true;
	}

	@Override
	public class_1799 method_7601(class_1657 player, int invSlot) {
		class_1735 slot = this.field_7761.get(invSlot);
		if (!slot.method_7681()) return class_1799.field_8037;

		class_1799 originalStack = slot.method_7677();
		class_1799 newStack = originalStack.method_7972();

		int playerInvStartIndex = inventory.method_5439();
		int playerInvEndIndex = this.field_7761.size();

		boolean moved;
		if (invSlot < playerInvStartIndex) {
			moved = this.method_7616(originalStack, playerInvStartIndex, playerInvEndIndex, true);
		} else {
			moved = insertItemIntoBackpack(originalStack);
		}

		if (!moved) return class_1799.field_8037;

		if (originalStack.method_7960()) {
			slot.method_53512(class_1799.field_8037);
		} else {
			slot.method_7668();
		}

		return newStack;
	}

	private boolean insertItemIntoBackpack(@NotNull class_1799 stack) {
		if (stack.method_31573(GBItemTags.BACKPACK_BLACKLIST)) return false;
		for (int i = 0; i < inventory.method_5439(); i++) {
			class_1799 slotStack = inventory.method_5438(i);
			if (!slotStack.method_7960() && class_1799.method_31577(stack, slotStack)) {
				int combined = Math.min(stack.method_7947() + slotStack.method_7947(), slotStack.method_7914());
				int transferred = combined - slotStack.method_7947();
				if (transferred > 0) {
					slotStack.method_7939(combined);
					stack.method_7934(transferred);
					if (stack.method_7960()) return true;
				}
			}
		}

		for (int i = 0; i < inventory.method_5439(); i++) {
			class_1799 slotStack = inventory.method_5438(i);
			if (slotStack.method_7960()) {
				inventory.method_5447(i, stack.method_7972());
				stack.method_7939(0);
				return true;
			}
		}

		return false;
	}
}
