/*
 * Copyright (c) 2025 macuguita
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE
 * OR OTHER DEALINGS IN THE SOFTWARE.
 */

package com.macuguita.backpacks.common.utils;

import com.macuguita.backpacks.common.components.GuitaBackpacksComponents;
import com.macuguita.backpacks.common.item.BackpackItem;
import dev.emi.trinkets.api.TrinketComponent;
import dev.emi.trinkets.api.TrinketInventory;
import dev.emi.trinkets.api.TrinketsApi;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_3545;

public class EquipmentUtils {

	private static final int TRINKET_SLOT_OFFSET = 10000;
	private static final int CUSTOM_EQUIPMENT_SLOT_OFFSET = 20000;

	public static boolean isTrinketsLoaded() {
		return FabricLoader.getInstance().isModLoaded("trinkets");
	}

	public static class_1799 getEquippedBackpack(class_1657 player) {
		if (!isTrinketsLoaded())
			return GuitaBackpacksComponents.EQUIPMENT_COMPONENT.get(player).getBackpack();
		return TrinketsApi.getTrinketComponent(player)
				.map(component -> component.getEquipped(stack -> stack.method_7909() instanceof BackpackItem)
						.stream()
						.findFirst()
						.map(class_3545::method_15441)
						.orElse(class_1799.field_8037))
				.orElse(class_1799.field_8037);
	}

	/**
	 * Gets the slot index of the equipped backpack.
	 * For regular inventory: returns 0-40 (standard inventory slots)
	 * For trinkets: returns TRINKET_SLOT_OFFSET + encoded position
	 * For custom equipment: returns CUSTOM_EQUIPMENT_SLOT_OFFSET
	 *
	 * @param player The player to search
	 * @return The slot index, or -1 if no backpack is equipped
	 */
	public static int getBackpackSlotIndex(class_1657 player) {

		int result = -1;

		if (isTrinketsLoaded()) {
			result = getTrinketBackpackSlotIndex(player);
		} else {

			class_1799 customBackpack = GuitaBackpacksComponents.EQUIPMENT_COMPONENT.get(player).getBackpack();
			if (!customBackpack.method_7960() && customBackpack.method_7909() instanceof BackpackItem) {
				result = CUSTOM_EQUIPMENT_SLOT_OFFSET;
			}
		}

		if (result != -1) return result;

		for (int i = 0; i < player.method_31548().method_5439(); i++) {
			class_1799 stack = player.method_31548().method_5438(i);
			if (stack.method_7909() instanceof BackpackItem) {
				return i;
			}
		}
		return -1;
	}

	private static int getTrinketBackpackSlotIndex(class_1657 player) {
		TrinketComponent trinketComponent = TrinketsApi.getTrinketComponent(player).orElse(null);
		if (trinketComponent == null) {
			return -1;
		}

		int groupIndex = 0;
		for (var groupEntry : trinketComponent.getGroups().entrySet()) {
			String groupId = groupEntry.getKey();
			int slotTypeIndex = 0;

			for (var slotEntry : groupEntry.getValue().getSlots().entrySet()) {
				String slotId = slotEntry.getKey();
				TrinketInventory trinketInv = trinketComponent.getInventory().get(groupId).get(slotId);

				for (int i = 0; i < trinketInv.method_5439(); i++) {
					class_1799 stack = trinketInv.method_5438(i);
					if (stack.method_7909() instanceof BackpackItem) {
						return TRINKET_SLOT_OFFSET + (groupIndex * 1000) + (slotTypeIndex * 100) + i;
					}
				}
				slotTypeIndex++;
			}
			groupIndex++;
		}

		return -1;
	}

	/**
	 * Gets the backpack ItemStack from a slot index.
	 * Handles regular inventory slots, trinket slots, and custom equipment slots.
	 *
	 * @param player The player
	 * @param slotIndex The slot index (from getBackpackSlotIndex)
	 * @return The ItemStack, or ItemStack.EMPTY if not found
	 */
	public static class_1799 getBackpackFromSlotIndex(class_1657 player, int slotIndex) {
		if (slotIndex < 0) {
			return class_1799.field_8037;
		}

		if (slotIndex >= CUSTOM_EQUIPMENT_SLOT_OFFSET) {
			return GuitaBackpacksComponents.EQUIPMENT_COMPONENT.get(player).getBackpack();
		}

		if (slotIndex >= TRINKET_SLOT_OFFSET) {
			if (isTrinketsLoaded()) {
				return getBackpackFromTrinketSlotIndex(player, slotIndex);
			}
			return class_1799.field_8037;
		}

		if (slotIndex < player.method_31548().method_5439()) {
			return player.method_31548().method_5438(slotIndex);
		}

		return class_1799.field_8037;
	}

	private static class_1799 getBackpackFromTrinketSlotIndex(class_1657 player, int slotIndex) {
		TrinketComponent trinketComponent = TrinketsApi.getTrinketComponent(player).orElse(null);
		if (trinketComponent == null) {
			return class_1799.field_8037;
		}

		int encoded = slotIndex - TRINKET_SLOT_OFFSET;
		int targetGroupIndex = encoded / 1000;
		int targetSlotTypeIndex = (encoded % 1000) / 100;
		int targetSlot = encoded % 100;

		int groupIndex = 0;
		for (var groupEntry : trinketComponent.getGroups().entrySet()) {
			if (groupIndex == targetGroupIndex) {
				String groupId = groupEntry.getKey();
				int slotTypeIndex = 0;
				for (var slotEntry : groupEntry.getValue().getSlots().entrySet()) {
					if (slotTypeIndex == targetSlotTypeIndex) {
						String slotId = slotEntry.getKey();
						TrinketInventory trinketInv = trinketComponent.getInventory().get(groupId).get(slotId);
						if (targetSlot < trinketInv.method_5439()) {
							return trinketInv.method_5438(targetSlot);
						}
						return class_1799.field_8037;
					}
					slotTypeIndex++;
				}
			}
			groupIndex++;
		}

		return class_1799.field_8037;
	}
}
