/*
 * Copyright (c) 2025 macuguita
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE
 * OR OTHER DEALINGS IN THE SOFTWARE.
 */

package com.macuguita.backpacks.client.gui.widgets;

import com.macuguita.backpacks.GuitaBackpacks;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.class_10799;
import net.minecraft.class_11909;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_332;
import net.minecraft.class_339;
import net.minecraft.class_3532;
import net.minecraft.class_6381;
import net.minecraft.class_6382;

@Environment(EnvType.CLIENT)
public class ScrollBarWidget extends class_339 {

	private static final class_2960 SCROLLER_TEXTURE = class_2960.method_60656("container/creative_inventory/scroller");
	private static final class_2960 SCROLLER_DISABLED_TEXTURE = class_2960.method_60656("container/creative_inventory/scroller_disabled");
	private static final class_2960 SCROLLER_BACK_TEXTURE = GuitaBackpacks.id("scroll_back");

	public static final int SCROLLER_WIDTH = 12;
	public static final int BACKGROUND_WIDTH = 14;
	public static final int SCROLLER_HEIGHT = 15;

	private final int innerHeight;
	final ScrollCallback callback;
	private float scrollPercent = 0f;
	private boolean scrolling = false;

	public ScrollBarWidget(int x, int y, int height, ScrollCallback callback) {
		super(x, y, BACKGROUND_WIDTH, height + 2, class_2561.method_43473());
		this.callback = callback;
		this.innerHeight = height;
	}

	public ScrollCallback getCallback() {
		return callback;
	}

	@Override
	protected void method_48579(class_332 context, int mouseX, int mouseY, float delta) {
		int x = method_46426();
		int y = method_46427();

		context.method_52706(class_10799.field_56883, SCROLLER_BACK_TEXTURE, x, y, this.field_22758, this.field_22759);

		int scrollerX = x + (this.field_22758 - SCROLLER_WIDTH) / 2;

		int scrollerY;
		if (innerHeight <= SCROLLER_HEIGHT) {
			scrollerY = y + 1;
		} else {
			scrollerY = y + 1 + (int) (scrollPercent * (innerHeight - SCROLLER_HEIGHT));
		}

		class_2960 scrollerTexture = callback.canScroll() ? SCROLLER_TEXTURE : SCROLLER_DISABLED_TEXTURE;
		context.method_52706(class_10799.field_56883, scrollerTexture, scrollerX, scrollerY, SCROLLER_WIDTH, SCROLLER_HEIGHT);
	}

	@Override
	public boolean method_25402(class_11909 click, boolean doubled) {
		if (!callback.canScroll() || click.method_74245() != 0) {
			return super.method_25402(click, doubled);
		}

		if (!doubled) {
			this.scrolling = true;

			int currentScrollerY;
			if (innerHeight <= SCROLLER_HEIGHT) {
				currentScrollerY = method_46427() + 1;
			} else {
				currentScrollerY = method_46427() + 1 + (int) (scrollPercent * (innerHeight - SCROLLER_HEIGHT));
			}

			if (click.comp_4799() >= currentScrollerY && click.comp_4799() <= currentScrollerY + SCROLLER_HEIGHT) {
				return true;
			}

			updateScroll(click.comp_4799(), true);
			return true;
		}

		return super.method_25402(click, doubled);
	}

	@Override
	public boolean method_25403(class_11909 click, double offsetX, double offsetY) {
		if (this.scrolling && callback.canScroll()) {
			updateScroll(click.comp_4799(), false);
			return true;
		}
		return super.method_25403(click, offsetX, offsetY);
	}

	@Override
	public boolean method_25406(class_11909 click) {
		if (click.method_74245() == 0) {
			this.scrolling = false;
		}
		return super.method_25406(click);
	}

	@Override
	public boolean method_25401(double mouseX, double mouseY, double horizontalAmount, double verticalAmount) {
		if (callback.canScroll() && this.method_25405(mouseX, mouseY)) {
			int scrollDirection = verticalAmount > 0 ? -1 : 1;
			callback.onScroll(scrollDirection);
			updateScrollPercent();
			return true;
		}
		return super.method_25401(mouseX, mouseY, horizontalAmount, verticalAmount);
	}

	private void updateScroll(double mouseY, boolean snap) {
		float position;
		if (innerHeight <= SCROLLER_HEIGHT) {
			position = 0f;
		} else {
			float denom = (float) (innerHeight - SCROLLER_HEIGHT);
			position = ((float) mouseY - (float) (method_46427() + 1) - (SCROLLER_HEIGHT / 2.0F)) / denom;
			position = class_3532.method_15363(position, 0.0F, 1.0F);
		}

		int maxOffset = callback.getMaxScrollOffset();
		int targetOffset = Math.round(position * maxOffset);
		int currentOffset = callback.getCurrentScrollOffset();

		if (targetOffset != currentOffset || snap) {
			callback.scrollTo(targetOffset);
			this.scrollPercent = maxOffset > 0 ? (float) targetOffset / maxOffset : 0f;
		}
	}

	public void updateScrollPercent() {
		int maxOffset = callback.getMaxScrollOffset();
		int currentOffset = callback.getCurrentScrollOffset();
		this.scrollPercent = maxOffset > 0 ? (float) currentOffset / maxOffset : 0f;
	}

	@Override
	protected void method_47399(class_6382 builder) {
		builder.method_37034(class_6381.field_33788, class_2561.method_43471("narration.gbackpacks.scroll_bar"));
	}

	/**
	 * Callback interface for scroll bar interactions
	 */
	public interface ScrollCallback {
		/**
		 * Called when scrolling by a delta amount (e.g., mouse wheel)
		 *
		 * @param delta positive or negative integer for scroll direction
		 */
		void onScroll(int delta);

		/**
		 * Called when scrolling to a specific offset (e.g., dragging)
		 *
		 * @param offset the target scroll offset
		 */
		void scrollTo(int offset);

		/**
		 * @return the maximum scroll offset
		 */
		int getMaxScrollOffset();

		/**
		 * @return the current scroll offset
		 */
		int getCurrentScrollOffset();

		/**
		 * @return whether scrolling is enabled
		 */
		boolean canScroll();
	}
}
