/*
 * Copyright (c) 2025 macuguita
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE
 * OR OTHER DEALINGS IN THE SOFTWARE.
 */

package com.macuguita.backpacks.network;

import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;
import net.minecraft.class_156;
import net.minecraft.class_238;
import net.minecraft.class_2960;
import net.minecraft.class_3300;
import net.minecraft.class_3518;
import net.minecraft.class_4013;
import net.minecraft.class_5699;
import com.macuguita.backpacks.GuitaBackpacks;
import com.mojang.serialization.Codec;
import com.mojang.serialization.DataResult;
import com.mojang.serialization.JsonOps;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import org.joml.Vector2i;
import org.joml.Vector3f;

public class BackpacksResourceReloadListener implements class_4013 {

	public static final List<Backpack> BACKPACKS = new ArrayList<>();
	private static final class_2960 BACKPACKS_DIR = GuitaBackpacks.id("backpacks");
	public static final class_2960 ID = GuitaBackpacks.id("backpacks_resource_reload_listener");

	@Override
	public void method_14491(class_3300 manager) {
		BACKPACKS.clear();

		var resources = manager.method_14488(BACKPACKS_DIR.method_12832(), path -> path.method_12832().endsWith(".json"));
		for (var entry : resources.entrySet()) {
			var id = entry.getKey();

			try (var reader = new InputStreamReader(entry.getValue().method_14482())) {
				var json = class_3518.method_15255(reader);

				var result = Backpack.CODEC.parse(JsonOps.INSTANCE, json);
				result.resultOrPartial(error -> GuitaBackpacks.LOGGER.warn("Failed to parse backpack at {}: {}", id, error))
						.ifPresent(BACKPACKS::add);
			} catch (Exception e) {
				GuitaBackpacks.LOGGER.error("Error reading backpack at {}: {}", id, e.getMessage(), e);
			}
		}
	}

	public record Backpack(class_2960 id, String translationKey, Vector2i guiDisplacement, float guiScale, class_238 blockCollisionShape) {

		public static final Codec<Vector2i> VECTOR2I_CODEC =
				Codec.INT.listOf().comapFlatMap(
						list -> class_156.method_33141(list, 2)
								.map(listi -> new Vector2i(listi.getFirst(), listi.get(1))),
						vector2i -> List.of(vector2i.x, vector2i.y)
				);

		public static final Codec<class_238> BOX_CODEC =
				class_5699.field_40723.listOf().comapFlatMap(
						list -> class_156.method_33141(list, 2)
								.map(listv3f -> {
									Vector3f min = listv3f.get(0);
									Vector3f max = listv3f.get(1);
									return new class_238(
											min.x / 16.0, min.y / 16.0, min.z / 16.0,
											max.x / 16.0, max.y / 16.0, max.z / 16.0
									);
								}),
						box -> List.of(
								new Vector3f((float) (box.field_1323 * 16), (float) (box.field_1322 * 16), (float) (box.field_1321 * 16)),
								new Vector3f((float) (box.field_1320 * 16), (float) (box.field_1325 * 16), (float) (box.field_1324 * 16))
						)
				);

		public static final Codec<Backpack> CODEC = RecordCodecBuilder.create(i -> i.group(
				class_2960.field_25139.fieldOf("id").forGetter(Backpack::id),
				Codec.STRING.fieldOf("translation_key").forGetter(Backpack::translationKey),
				VECTOR2I_CODEC.optionalFieldOf("gui_displacement", new Vector2i(0, 0))
						.forGetter(Backpack::guiDisplacement),
				Codec.FLOAT.optionalFieldOf("gui_scale", 1.0f).flatXmap(
						scale -> scale > 0
								? DataResult.success(scale)
								: DataResult.error(() -> "gui_scale must be higher than 0 " + scale),
						DataResult::success
				).forGetter(Backpack::guiScale),
				BOX_CODEC.optionalFieldOf("block_collision_shape", new class_238(0.0, 0.0, 0.0, 1.0, 1.0, 1.0))
						.forGetter(Backpack::blockCollisionShape)
		).apply(i, Backpack::new));
	}
}
