/*
 * Decompiled with CFR 0.152.
 */
package com.sheath.veinminer.logic;

import com.sheath.veinminer.concurrent.TaskExecutor;
import com.sheath.veinminer.config.ConfigService;
import com.sheath.veinminer.config.GeneralConfig;
import com.sheath.veinminer.logic.rules.RuleIndex;
import com.sheath.veinminer.metrics.ServerTpsTracker;
import com.sheath.veinminer.mixin.ExperienceDroppingBlockAccessor;
import com.sheath.veinminer.permission.PermissionService;
import com.sheath.veinminer.player.PlayerSettingsStore;
import com.sheath.veinminer.state.CooldownTracker;
import com.sheath.veinminer.state.KeyStateRegistry;
import com.sheath.veinminer.util.Translations;
import com.sheath.veinminer.visual.ParticleOutlineManager;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Holder;
import net.minecraft.core.Vec3i;
import net.minecraft.core.registries.Registries;
import net.minecraft.network.chat.Component;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.sounds.SoundSource;
import net.minecraft.world.entity.EquipmentSlot;
import net.minecraft.world.entity.ExperienceOrb;
import net.minecraft.world.entity.LivingEntity;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.enchantment.EnchantmentHelper;
import net.minecraft.world.item.enchantment.Enchantments;
import net.minecraft.world.level.ItemLike;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.block.Block;
import net.minecraft.world.level.block.Blocks;
import net.minecraft.world.level.block.DropExperienceBlock;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.storage.loot.LootParams;
import net.minecraft.world.level.storage.loot.parameters.LootContextParams;
import net.minecraft.world.phys.Vec3;

public final class VeinMinerController {
    private static final BlockPos[] NEIGHBOR_OFFSETS = VeinMinerController.buildNeighborOffsets();
    private final ConfigService configService;
    private final TaskExecutor taskExecutor;
    private final PlayerSettingsStore playerSettings;
    private final ServerTpsTracker tpsTracker;
    private final PermissionService permissionService;
    private final KeyStateRegistry keyStates;
    private final CooldownTracker cooldowns;
    private ConfigService.ConfigSnapshot snapshot;
    private RuleIndex ruleIndex;
    private long autosaveCounter = 0L;

    public VeinMinerController(ConfigService configService, TaskExecutor taskExecutor, PlayerSettingsStore playerSettings, ServerTpsTracker tpsTracker, PermissionService permissionService, KeyStateRegistry keyStates, CooldownTracker cooldowns) {
        this.configService = configService;
        this.taskExecutor = taskExecutor;
        this.playerSettings = playerSettings;
        this.tpsTracker = tpsTracker;
        this.permissionService = permissionService;
        this.keyStates = keyStates;
        this.cooldowns = cooldowns;
    }

    public void reloadFromConfig() {
        this.snapshot = this.configService.snapshot();
        this.ruleIndex = RuleIndex.fromSnapshot(this.snapshot);
        this.taskExecutor.configure(this.snapshot.general().threadCount());
        this.permissionService.configure(this.snapshot.general().autoLuckPerms());
    }

    public void onServerStarted() {
        ParticleOutlineManager.register();
        this.reloadFromConfig();
        this.autosaveCounter = 0L;
    }

    public void onServerStopping() {
        this.playerSettings.saveBlocking();
        this.cooldowns.clearAll();
        this.taskExecutor.shutdown();
    }

    public void onServerTick(MinecraftServer server) {
        this.tpsTracker.recordTick(System.nanoTime());
        ++this.autosaveCounter;
        if (this.autosaveCounter >= 6000L) {
            this.autosaveCounter = 0L;
            this.playerSettings.saveAsync();
        }
    }

    public void onPlayerDisconnect(ServerPlayer player) {
        this.keyStates.unregister(player);
        this.cooldowns.clear(player);
        this.playerSettings.drop(player);
    }

    public boolean handleBlockBreak(ServerLevel level, ServerPlayer player, BlockPos pos, BlockState state) {
        int remaining;
        if (this.snapshot == null || this.ruleIndex == null) {
            return true;
        }
        if (!this.snapshot.general().veinminerEnabled()) {
            return true;
        }
        if (!this.permissionService.hasPermission(player, "veinminer.use")) {
            this.notify(player, PlayerSettingsStore.MessageType.PERMISSION, "message.veinminer.no_permission", false, new Object[0]);
            return true;
        }
        boolean activationAttempt = this.isActivationAttempt(player);
        if (!this.playerSettings.isVeinminerEnabled(player)) {
            if (activationAttempt) {
                this.notify(player, PlayerSettingsStore.MessageType.DISABLED, "message.veinminer.disabled", true, new Object[0]);
            }
            return true;
        }
        if (!this.wantsVeinminer(player)) {
            return true;
        }
        ItemStack tool = player.getMainHandItem();
        if (!this.ruleIndex.isToolAllowed(tool)) {
            return true;
        }
        if (state.requiresCorrectToolForDrops() && !player.hasCorrectToolForDrops(state)) {
            return true;
        }
        if (!this.ruleIndex.isBlockAllowed(state, tool)) {
            return true;
        }
        if (this.snapshot.general().cooldown().enabled() && this.cooldowns.isOnCooldown(player, this.snapshot.general().cooldown().seconds())) {
            this.notify(player, PlayerSettingsStore.MessageType.COOLDOWN, "message.veinminer.cooldown", false, new Object[0]);
            return false;
        }
        int reserve = this.snapshot.general().durabilityReserveFor(tool.getMaxDamage());
        int n = remaining = tool.isDamageableItem() ? tool.getMaxDamage() - tool.getDamageValue() : Integer.MAX_VALUE;
        if (this.snapshot.general().checkToolDurability() && tool.isDamageableItem() && remaining <= reserve) {
            this.notify(player, PlayerSettingsStore.MessageType.DURABILITY, "message.veinminer.low_durability", true, new Object[0]);
            return true;
        }
        int blockCap = this.computeBlockCap(tool, remaining, reserve);
        if (blockCap <= 0) {
            return true;
        }
        Holder.Reference silkTouch = level.registryAccess().lookupOrThrow(Registries.ENCHANTMENT).getOrThrow(Enchantments.SILK_TOUCH);
        int silkLevel = EnchantmentHelper.getItemEnchantmentLevel((Holder)silkTouch, (ItemStack)tool);
        ItemStack toolCopy = tool.copy();
        BlockState originState = state;
        CompletableFuture<VeinPlan> future = this.taskExecutor.submitAsync(() -> this.findVein(level, pos, originState, blockCap, silkLevel, toolCopy));
        future.thenAccept(plan -> level.getServer().execute(() -> this.applyPlan(level, player, tool, (VeinPlan)plan, reserve)));
        return false;
    }

    private boolean wantsVeinminer(ServerPlayer player) {
        boolean preferKey;
        boolean bl = preferKey = this.playerSettings.useKeybind(player) && this.keyStates.hasClient(player);
        if (preferKey) {
            if (this.playerSettings.keyToggleMode(player)) {
                return this.playerSettings.isKeyToggleActive(player);
            }
            return this.keyStates.isKeyPressed(player);
        }
        this.playerSettings.resetKeyToggleState(player);
        if (this.snapshot.general().requireCrouch()) {
            return this.playerSettings.updateCrouchToggleState(player, player.isShiftKeyDown());
        }
        this.playerSettings.resetCrouchToggleState(player);
        return true;
    }

    private int computeBlockCap(ItemStack tool, int remaining, int reserve) {
        int baseLimit;
        GeneralConfig.BlockLimitSettings limits = this.snapshot.general().blockLimits();
        if (limits.dynamicMaxBlocks()) {
            double tps = this.tpsTracker.currentTps();
            int range = limits.maxDynamicBlocks() - limits.minBlocks();
            int computed = (int)Math.round(tps / 20.0 * (double)range) + limits.minBlocks();
            baseLimit = Math.max(limits.minBlocks(), Math.min(limits.maxDynamicBlocks(), computed));
        } else {
            baseLimit = limits.maxBlocks();
        }
        if (!this.snapshot.general().checkToolDurability() || !tool.isDamageableItem()) {
            return baseLimit;
        }
        int breakable = Math.max(0, remaining - reserve);
        return Math.min(baseLimit, Math.max(1, breakable));
    }

    private VeinPlan findVein(ServerLevel level, BlockPos origin, BlockState originState, int limit, int silkLevel, ItemStack toolCopy) {
        HashSet<BlockPos> visited = new HashSet<BlockPos>();
        ArrayDeque<BlockPos> queue = new ArrayDeque<BlockPos>();
        visited.add(origin);
        queue.add(origin);
        Block originBlock = originState.getBlock();
        block0: while (!queue.isEmpty() && visited.size() < limit) {
            BlockPos current = (BlockPos)queue.poll();
            for (BlockPos offset : NEIGHBOR_OFFSETS) {
                BlockState neighborState;
                BlockPos neighbor = current.offset((Vec3i)offset);
                if (visited.contains(neighbor) || !(neighborState = level.getBlockState(neighbor)).is(originBlock)) continue;
                visited.add(neighbor);
                queue.add(neighbor);
                if (visited.size() >= limit) continue block0;
            }
        }
        return new VeinPlan(origin, originState, visited, silkLevel, toolCopy);
    }

    private void applyPlan(ServerLevel level, ServerPlayer player, ItemStack actualTool, VeinPlan plan, int reserve) {
        if (plan.blocks().isEmpty()) {
            return;
        }
        boolean checkDurability = this.snapshot.general().checkToolDurability() && actualTool.isDamageableItem();
        int remaining = actualTool.isDamageableItem() ? actualTool.getMaxDamage() - actualTool.getDamageValue() : Integer.MAX_VALUE;
        int breakable = checkDurability ? Math.max(0, remaining - reserve) : plan.blocks().size();
        int limit = Math.min(breakable, plan.blocks().size());
        if (limit <= 0) {
            this.notify(player, PlayerSettingsStore.MessageType.DURABILITY, "message.veinminer.low_durability", true, new Object[0]);
            return;
        }
        int broken = 0;
        int totalXp = 0;
        ArrayList<ItemStack> collectedDrops = new ArrayList<ItemStack>();
        for (BlockPos pos : plan.blocks()) {
            Block block;
            if (broken >= limit) break;
            BlockState state = level.getBlockState(pos);
            if (state.isAir()) continue;
            if (this.playerSettings.isParticlesEnabled(player) && this.snapshot.general().particles().enabled()) {
                GeneralConfig.ParticleSettings particles = this.snapshot.general().particles();
                ParticleOutlineManager.spawnOutline(level, pos, particles.red(), particles.green(), particles.blue(), particles.durationTicks());
            }
            this.mergeDrops(collectedDrops, this.collectDrops(level, pos, state, plan.toolCopy(), plan.silkLevel()));
            if (plan.silkLevel() == 0 && (block = state.getBlock()) instanceof DropExperienceBlock) {
                DropExperienceBlock block2 = (DropExperienceBlock)block;
                totalXp += this.resolveExperience(block2, level);
            }
            level.setBlock(pos, Blocks.AIR.defaultBlockState(), 3);
            ++broken;
        }
        if (checkDurability && broken < plan.blocks().size()) {
            this.notify(player, PlayerSettingsStore.MessageType.DURABILITY, "message.veinminer.limit_durability", true, plan.blocks().size(), broken);
        }
        for (ItemStack drop : collectedDrops) {
            Block.popResource((Level)level, (BlockPos)plan.origin(), (ItemStack)drop);
        }
        if (totalXp > 0) {
            ExperienceOrb.award((ServerLevel)level, (Vec3)Vec3.atCenterOf((Vec3i)plan.origin()), (int)totalXp);
        }
        level.playSound(null, plan.origin(), plan.originState().getSoundType().getBreakSound(), SoundSource.BLOCKS, 1.0f, 1.0f);
        if (actualTool.isDamageableItem()) {
            actualTool.hurtAndBreak(broken, (LivingEntity)player, EquipmentSlot.MAINHAND);
        }
        if (this.snapshot.general().cooldown().enabled() && broken > 0) {
            this.cooldowns.startCooldown(player);
        }
    }

    private List<ItemStack> collectDrops(ServerLevel level, BlockPos pos, BlockState state, ItemStack tool, int silkLevel) {
        if (silkLevel > 0) {
            return List.of(new ItemStack((ItemLike)state.getBlock()));
        }
        LootParams.Builder builder = new LootParams.Builder(level).withParameter(LootContextParams.BLOCK_STATE, (Object)state).withParameter(LootContextParams.ORIGIN, (Object)Vec3.atCenterOf((Vec3i)pos)).withParameter(LootContextParams.TOOL, (Object)tool);
        return state.getDrops(builder);
    }

    private void mergeDrops(List<ItemStack> target, List<ItemStack> newDrops) {
        block0: for (ItemStack drop : newDrops) {
            for (ItemStack existing : target) {
                if (!ItemStack.isSameItemSameComponents((ItemStack)existing, (ItemStack)drop)) continue;
                existing.grow(drop.getCount());
                continue block0;
            }
            target.add(drop.copy());
        }
    }

    private int resolveExperience(DropExperienceBlock block, ServerLevel level) {
        return ((ExperienceDroppingBlockAccessor)block).veinminer$xpRange().sample(level.random);
    }

    private boolean isActivationAttempt(ServerPlayer player) {
        boolean preferKey;
        boolean bl = preferKey = this.playerSettings.useKeybind(player) && this.keyStates.hasClient(player);
        if (preferKey) {
            return this.keyStates.isKeyPressed(player);
        }
        if (this.snapshot.general().requireCrouch()) {
            return player.isShiftKeyDown();
        }
        return true;
    }

    private void notify(ServerPlayer player, PlayerSettingsStore.MessageType type, String translationKey, boolean actionBar, Object ... args) {
        if (!this.playerSettings.isMessageEnabled(player, type)) {
            return;
        }
        Component message = Translations.translate(translationKey, args);
        player.displayClientMessage(message, actionBar);
    }

    private static BlockPos[] buildNeighborOffsets() {
        BlockPos[] offsets = new BlockPos[26];
        int index = 0;
        for (int dx = -1; dx <= 1; ++dx) {
            for (int dy = -1; dy <= 1; ++dy) {
                for (int dz = -1; dz <= 1; ++dz) {
                    if (dx == 0 && dy == 0 && dz == 0) continue;
                    offsets[index++] = new BlockPos(dx, dy, dz);
                }
            }
        }
        return offsets;
    }

    public static final class Permissions {
        public static final String USE = "veinminer.use";
        public static final String RELOAD = "veinminer.reload";

        private Permissions() {
        }
    }

    private record VeinPlan(BlockPos origin, BlockState originState, Set<BlockPos> blocks, int silkLevel, ItemStack toolCopy) {
    }
}

