/*
 * Decompiled with CFR 0.152.
 */
package com.sheath.veinminer.player;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.sheath.veinminer.concurrent.TaskExecutor;
import com.sheath.veinminer.util.Log;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.EnumMap;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import net.minecraft.server.level.ServerPlayer;
import net.neoforged.fml.loading.FMLPaths;

public final class PlayerSettingsStore {
    private static final Gson GSON = new GsonBuilder().setPrettyPrinting().create();
    private final Path dataFile = FMLPaths.CONFIGDIR.get().resolve("Veinminer").resolve("Players").resolve("PlayerData.json");
    private final Map<UUID, PlayerSettings> settings = new ConcurrentHashMap<UUID, PlayerSettings>();
    private final TaskExecutor executor;
    private final Lock writeLock = new ReentrantLock();

    public PlayerSettingsStore(TaskExecutor executor) {
        this.executor = executor;
    }

    public void load() {
        this.ensureParentDirectory();
        if (!Files.exists(this.dataFile, new LinkOption[0])) {
            this.saveBlocking();
            return;
        }
        try (BufferedReader reader = Files.newBufferedReader(this.dataFile, StandardCharsets.UTF_8);){
            JsonElement element = JsonParser.parseReader((Reader)reader);
            if (!element.isJsonObject()) {
                Log.warn("Player data file did not contain a JSON object, resetting", new Object[0]);
                this.saveBlocking();
                return;
            }
            JsonObject root = element.getAsJsonObject();
            this.parseVeinminerStates(root.getAsJsonObject("veinminer"));
            this.parseParticleStates(root.getAsJsonObject("particles"));
            this.parseMessageStates(root.getAsJsonObject("messages"));
            this.parseInputStates(root.getAsJsonObject("input"));
            Log.info("Loaded player preference data for {} players", this.settings.size());
        }
        catch (IOException ex) {
            Log.error("Failed to read player data file", ex);
        }
    }

    public CompletableFuture<Void> saveAsync() {
        return this.executor.submitAsync(() -> {
            this.writeLock.lock();
            try {
                this.ensureParentDirectory();
                try (BufferedWriter writer = Files.newBufferedWriter(this.dataFile, StandardCharsets.UTF_8, new OpenOption[0]);){
                    GSON.toJson((JsonElement)this.buildSerializableJson(), (Appendable)writer);
                }
            }
            catch (IOException ex) {
                Log.error("Failed to save player data", ex);
            }
            finally {
                this.writeLock.unlock();
            }
            return null;
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void saveBlocking() {
        this.writeLock.lock();
        try {
            this.ensureParentDirectory();
            try (BufferedWriter writer = Files.newBufferedWriter(this.dataFile, StandardCharsets.UTF_8, new OpenOption[0]);){
                GSON.toJson((JsonElement)this.buildSerializableJson(), (Appendable)writer);
            }
        }
        catch (IOException ex) {
            Log.error("Failed to save player data", ex);
        }
        finally {
            this.writeLock.unlock();
        }
    }

    public boolean isVeinminerEnabled(ServerPlayer player) {
        return this.settingsFor((UUID)player.getUUID()).veinminerEnabled;
    }

    public void setVeinminerEnabled(ServerPlayer player, boolean enabled) {
        this.settingsFor((UUID)player.getUUID()).veinminerEnabled = enabled;
    }

    public boolean isParticlesEnabled(ServerPlayer player) {
        return this.settingsFor((UUID)player.getUUID()).particlesEnabled;
    }

    public void setParticlesEnabled(ServerPlayer player, boolean enabled) {
        this.settingsFor((UUID)player.getUUID()).particlesEnabled = enabled;
    }

    public boolean isMessageEnabled(ServerPlayer player, MessageType type) {
        return this.settingsFor(player.getUUID()).messageEnabled(type);
    }

    public void setMessageEnabled(ServerPlayer player, MessageType type, boolean enabled) {
        this.settingsFor(player.getUUID()).setMessage(type, enabled);
    }

    public boolean useKeybind(ServerPlayer player) {
        return this.settingsFor((UUID)player.getUUID()).useKeybind;
    }

    public void setUseKeybind(ServerPlayer player, boolean useKeybind) {
        this.settingsFor((UUID)player.getUUID()).useKeybind = useKeybind;
    }

    public boolean keyToggleMode(ServerPlayer player) {
        return this.settingsFor((UUID)player.getUUID()).keyToggleMode;
    }

    public void setKeyToggleMode(ServerPlayer player, boolean toggle) {
        this.settingsFor((UUID)player.getUUID()).keyToggleMode = toggle;
    }

    public boolean isKeyToggleActive(ServerPlayer player) {
        return this.settingsFor((UUID)player.getUUID()).keyToggleActive;
    }

    public void setKeyToggleState(ServerPlayer player, boolean active) {
        this.settingsFor((UUID)player.getUUID()).keyToggleActive = active;
    }

    public boolean flipKeyToggleState(ServerPlayer player) {
        PlayerSettings settings = this.settingsFor(player.getUUID());
        settings.keyToggleActive = !settings.keyToggleActive;
        return settings.keyToggleActive;
    }

    public void resetKeyToggleState(ServerPlayer player) {
        PlayerSettings settings = this.settingsFor(player.getUUID());
        settings.keyToggleActive = false;
    }

    public boolean crouchToggleMode(ServerPlayer player) {
        return this.settingsFor((UUID)player.getUUID()).crouchToggleMode;
    }

    public void setCrouchToggleMode(ServerPlayer player, boolean toggle) {
        this.settingsFor((UUID)player.getUUID()).crouchToggleMode = toggle;
    }

    public boolean isCrouchToggleActive(ServerPlayer player) {
        return this.settingsFor((UUID)player.getUUID()).crouchToggleActive;
    }

    public void setCrouchToggleState(ServerPlayer player, boolean active) {
        this.settingsFor((UUID)player.getUUID()).crouchToggleActive = active;
    }

    public boolean updateCrouchToggleState(ServerPlayer player, boolean sneaking) {
        PlayerSettings settings = this.settingsFor(player.getUUID());
        if (!settings.crouchToggleMode) {
            settings.lastCrouchInput = sneaking;
            return sneaking;
        }
        if (sneaking && !settings.lastCrouchInput) {
            settings.crouchToggleActive = !settings.crouchToggleActive;
        }
        settings.lastCrouchInput = sneaking;
        return settings.crouchToggleActive;
    }

    public boolean lastCrouchInput(ServerPlayer player) {
        return this.settingsFor((UUID)player.getUUID()).lastCrouchInput;
    }

    public void setLastCrouchInput(ServerPlayer player, boolean value) {
        this.settingsFor((UUID)player.getUUID()).lastCrouchInput = value;
    }

    public void resetCrouchToggleState(ServerPlayer player) {
        PlayerSettings settings = this.settingsFor(player.getUUID());
        settings.crouchToggleActive = false;
        settings.lastCrouchInput = false;
    }

    public void drop(ServerPlayer player) {
        this.settings.remove(player.getUUID());
    }

    private void parseVeinminerStates(JsonObject object) {
        if (object == null) {
            return;
        }
        for (Map.Entry entry : object.entrySet()) {
            UUID uuid = PlayerSettingsStore.parseUuid((String)entry.getKey());
            if (uuid == null || !((JsonElement)entry.getValue()).isJsonPrimitive()) continue;
            this.settingsFor((UUID)uuid).veinminerEnabled = ((JsonElement)entry.getValue()).getAsBoolean();
        }
    }

    private void parseParticleStates(JsonObject object) {
        if (object == null) {
            return;
        }
        for (Map.Entry entry : object.entrySet()) {
            UUID uuid = PlayerSettingsStore.parseUuid((String)entry.getKey());
            if (uuid == null || !((JsonElement)entry.getValue()).isJsonPrimitive()) continue;
            this.settingsFor((UUID)uuid).particlesEnabled = ((JsonElement)entry.getValue()).getAsBoolean();
        }
    }

    private void parseMessageStates(JsonObject object) {
        if (object == null) {
            return;
        }
        for (Map.Entry entry : object.entrySet()) {
            UUID uuid = PlayerSettingsStore.parseUuid((String)entry.getKey());
            if (uuid == null || !((JsonElement)entry.getValue()).isJsonObject()) continue;
            PlayerSettings playerSettings = this.settingsFor(uuid);
            JsonObject messageObject = ((JsonElement)entry.getValue()).getAsJsonObject();
            for (MessageType type : MessageType.values()) {
                JsonElement val = messageObject.get(type.id());
                if (val == null || !val.isJsonPrimitive()) continue;
                playerSettings.setMessage(type, val.getAsBoolean());
            }
        }
    }

    private void parseInputStates(JsonObject object) {
        if (object == null) {
            return;
        }
        for (Map.Entry entry : object.entrySet()) {
            JsonElement crouchState;
            JsonElement crouchToggle;
            JsonElement keyState;
            JsonElement keyToggle;
            UUID uuid = PlayerSettingsStore.parseUuid((String)entry.getKey());
            if (uuid == null || !((JsonElement)entry.getValue()).isJsonObject()) continue;
            PlayerSettings settings = this.settingsFor(uuid);
            JsonObject input = ((JsonElement)entry.getValue()).getAsJsonObject();
            JsonElement useKey = input.get("useKeybind");
            if (useKey != null && useKey.isJsonPrimitive()) {
                settings.useKeybind = useKey.getAsBoolean();
            }
            if ((keyToggle = input.get("keyToggle")) != null && keyToggle.isJsonPrimitive()) {
                settings.keyToggleMode = keyToggle.getAsBoolean();
            }
            if ((keyState = input.get("keyToggleState")) != null && keyState.isJsonPrimitive()) {
                settings.keyToggleActive = keyState.getAsBoolean();
            }
            if ((crouchToggle = input.get("crouchToggle")) != null && crouchToggle.isJsonPrimitive()) {
                settings.crouchToggleMode = crouchToggle.getAsBoolean();
            }
            if ((crouchState = input.get("crouchToggleState")) == null || !crouchState.isJsonPrimitive()) continue;
            settings.crouchToggleActive = crouchState.getAsBoolean();
        }
    }

    private PlayerSettings settingsFor(UUID uuid) {
        return this.settings.computeIfAbsent(uuid, __ -> new PlayerSettings());
    }

    private JsonObject buildSerializableJson() {
        JsonObject root = new JsonObject();
        JsonObject veinminerObj = new JsonObject();
        JsonObject particlesObj = new JsonObject();
        JsonObject messagesObj = new JsonObject();
        JsonObject inputObj = new JsonObject();
        for (Map.Entry<UUID, PlayerSettings> entry : this.settings.entrySet()) {
            String key = entry.getKey().toString();
            PlayerSettings value = entry.getValue();
            veinminerObj.addProperty(key, Boolean.valueOf(value.veinminerEnabled));
            particlesObj.addProperty(key, Boolean.valueOf(value.particlesEnabled));
            JsonObject messageObj = new JsonObject();
            for (MessageType type : MessageType.values()) {
                messageObj.addProperty(type.id(), Boolean.valueOf(value.messageEnabled(type)));
            }
            messagesObj.add(key, (JsonElement)messageObj);
            JsonObject input = new JsonObject();
            input.addProperty("useKeybind", Boolean.valueOf(value.useKeybind));
            input.addProperty("keyToggle", Boolean.valueOf(value.keyToggleMode));
            input.addProperty("keyToggleState", Boolean.valueOf(value.keyToggleActive));
            input.addProperty("crouchToggle", Boolean.valueOf(value.crouchToggleMode));
            input.addProperty("crouchToggleState", Boolean.valueOf(value.crouchToggleActive));
            inputObj.add(key, (JsonElement)input);
        }
        root.add("veinminer", (JsonElement)veinminerObj);
        root.add("particles", (JsonElement)particlesObj);
        root.add("messages", (JsonElement)messagesObj);
        root.add("input", (JsonElement)inputObj);
        return root;
    }

    private void ensureParentDirectory() {
        try {
            Files.createDirectories(this.dataFile.getParent(), new FileAttribute[0]);
        }
        catch (IOException ex) {
            Log.error("Failed to create player data directory", ex);
        }
    }

    private static UUID parseUuid(String raw) {
        try {
            return UUID.fromString(raw);
        }
        catch (IllegalArgumentException ex) {
            Log.warn("Invalid UUID '{}' in player data", raw);
            return null;
        }
    }

    private static final class PlayerSettings {
        boolean veinminerEnabled = true;
        boolean particlesEnabled = true;
        boolean useKeybind = true;
        boolean keyToggleMode = false;
        boolean keyToggleActive = false;
        boolean crouchToggleMode = false;
        boolean crouchToggleActive = false;
        boolean lastCrouchInput = false;
        final EnumMap<MessageType, Boolean> messages = new EnumMap(MessageType.class);

        private PlayerSettings() {
        }

        boolean messageEnabled(MessageType type) {
            return this.messages.getOrDefault((Object)type, Boolean.TRUE);
        }

        void setMessage(MessageType type, boolean enabled) {
            this.messages.put(type, enabled);
        }
    }

    public static enum MessageType {
        PERMISSION("permission", "Permission"),
        DISABLED("disabled", "Disabled"),
        COOLDOWN("cooldown", "Cooldown"),
        DURABILITY("durability", "Durability");

        private final String id;
        private final String displayName;

        private MessageType(String id, String displayName) {
            this.id = id;
            this.displayName = displayName;
        }

        public String id() {
            return this.id;
        }

        public String displayName() {
            return this.displayName;
        }
    }
}

