package org.codeberg.zenxarch.zombies.difficulty;

import java.util.function.ToDoubleFunction;
import java.util.stream.Stream;
import net.minecraft.class_1267;
import net.minecraft.class_1299;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_3218;
import net.minecraft.class_3222;
import net.minecraft.class_3468;
import net.minecraft.class_3532;
import net.minecraft.class_4076;
import net.minecraft.class_5134;
import net.minecraft.class_9334;
import org.codeberg.zenxarch.zombies.spawning.ZombieApocalypse;

public abstract class DifficultyCalculations {
  private static final int TICKS_PER_HOUR = 60 * 60 * 20;
  private static final int TICKS_PER_DAY = 20 * 60 * 20;

  private DifficultyCalculations() {
    throw new IllegalStateException("Utility class");
  }

  private static double normalize(double value) {
    return Math.clamp(value, 0.0, 1.0);
  }

  private static double[] getPlayerScore(class_3218 world, class_2338 pos) {
    var players = 0;
    var result = new double[] {0.0, 0.0};
    for (class_3222 player : ZombieApocalypse.players(world)) {
      if (player.method_5707(pos.method_46558()) > 128.0 * 128.0) continue;
      players++;
      result[0] += getPlayerScore(player);
      result[1] += player.method_14248().method_15025(class_3468.field_15403.method_14956(class_1299.field_6051));
    }
    if (players == 0) return result;
    result[0] = normalize(result[0] / players);
    result[1] = normalize(result[1] / (players * 2500));
    return result;
  }

  private static double baseDifficulty(class_3218 world, class_2338 pos) {
    var dayFactor = mapDays(world.method_8407(), getDays(world));
    var moonSize = (double) world.method_30272();
    var timeFactor = normalize(dayFactor * (1.0 + moonSize));
    var playerScore = getPlayerScore(world, pos);
    return timeFactor * 0.1
        + timeFactor * playerScore[0] * 0.5
        + playerScore[0] * 0.3
        + playerScore[1] * 0.1;
  }

  public static double calculateDifficulty(class_3218 world, class_2338 pos) {
    var inhabitedTimeFactor = mapHours(world.method_8407(), getHoursInhabited(world, pos));
    if (inhabitedTimeFactor < 0.0) return 0.0;

    return inhabitedTimeFactor * baseDifficulty(world, pos);
  }

  private static double normalize(double value, double start, double end) {
    return class_3532.method_32854(value, start, end, 0.0, 1.0);
  }

  private static double mapDays(class_1267 difficulty, double days) {
    return switch (difficulty) {
      case field_5801 -> 0.0;
      case field_5805 -> normalize(days, 5.0, 250.0);
      case field_5802 -> normalize(days, 2.0, 150.0);
      case field_5807 -> normalize(days, 0.0, 100.0);
    };
  }

  private static double mapHours(class_1267 difficulty, double hours) {
    var delta =
        switch (difficulty) {
          case field_5801 -> 0.0;
          case field_5805 -> normalize(hours, 8.0, 1.5);
          case field_5802 -> normalize(hours, 20.0, 1.5);
          case field_5807 -> normalize(hours, 35.0, 1.5);
        };
    return class_3532.method_16436(delta, 0.25, 1.0);
  }

  private static double getDays(class_3218 world) {
    return (double) world.method_8532() / TICKS_PER_DAY;
  }

  private static double getHoursInhabited(class_1937 world, class_2338 pos) {
    if (world.method_8393(
        class_4076.method_18675(pos.method_10263()), class_4076.method_18675(pos.method_10260()))) {
      return (double) world.method_8500(pos).method_12033() / TICKS_PER_HOUR;
    }

    return 0.0;
  }

  private static Stream<class_1799> items(class_3222 player) {
    return player.method_31548().method_67533().stream();
  }

  private static double getPlayerScore(class_3222 player) {
    var weaponSpeedScore =
        scoreWeapon(
            player,
            DifficultyCalculations::getDamagePerSecond,
            class_1802.field_8091,
            class_1802.field_22022);
    var weaponDamageScore =
        scoreWeapon(
            player, DifficultyCalculations::getAttackDamage, class_1802.field_8406, class_1802.field_22025);
    var foodScore = items(player).mapToDouble(DifficultyCalculations::scoreFood).sum();
    foodScore = normalize(foodScore);

    var armor =
        player.method_45325(class_5134.field_23724)
            + player.method_45325(class_5134.field_23725)
            + player.method_45325(class_5134.field_23718);

    armor = normalize(armor, 0.0, 32.0);
    return (armor + weaponSpeedScore + weaponDamageScore) * (0.75 / 3.0) + foodScore * 0.25;
  }

  private static double getAttackDamage(class_1799 stack) {
    return ItemAttributeUtils.getAttributeValue(
        class_1299.field_6097, stack, class_5134.field_23721);
  }

  private static double getAttackSpeed(class_1799 stack) {
    return ItemAttributeUtils.getAttributeValue(
        class_1299.field_6097, stack, class_5134.field_23723);
  }

  private static double getDamagePerSecond(class_1799 stack) {
    return getAttackDamage(stack) * getAttackSpeed(stack);
  }

  private static double scoreFood(class_1799 stack) {
    var score = 0.0;
    if (stack.method_57353().method_57832(class_9334.field_50075))
      score = (double) stack.method_58694(class_9334.field_50075).comp_2491() * stack.method_7947();
    return normalize(score, 0.0, 400.0);
  }

  private static boolean isAWeapon(class_1799 stack) {
    if (stack.method_7960()) return false;
    if (stack.method_57353().method_57832(class_9334.field_49636)) return true;
    return stack.method_7942();
  }

  private static double scoreWeapon(
      class_3222 player, ToDoubleFunction<class_1799> scorer, class_1792 minItem, class_1792 maxItem) {
    var baseMin = scorer.applyAsDouble(minItem.method_7854());
    var baseMax = scorer.applyAsDouble(maxItem.method_7854());
    var bestScore =
        items(player)
            .filter(DifficultyCalculations::isAWeapon)
            .mapToDouble(scorer)
            .max()
            .orElse(0.0);
    return normalize(bestScore, baseMin, baseMax);
  }
}
