/*
 * Decompiled with CFR 0.152.
 */
package playerquests.utility.listener;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonMappingException;
import java.io.File;
import java.io.IOException;
import java.nio.file.ClosedWatchServiceException;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.util.HashSet;
import java.util.Set;
import java.util.stream.Stream;
import org.bukkit.Bukkit;
import org.bukkit.entity.Player;
import org.bukkit.event.EventHandler;
import org.bukkit.event.HandlerList;
import org.bukkit.event.Listener;
import org.bukkit.event.server.ServerLoadEvent;
import playerquests.Core;
import playerquests.product.Quest;
import playerquests.utility.ChatUtils;
import playerquests.utility.FileUtils;
import playerquests.utility.singleton.Database;
import playerquests.utility.singleton.PlayerQuests;
import playerquests.utility.singleton.QuestRegistry;

public class ServerListener
implements Listener {
    private WatchService watchService;
    private Thread watchThread;

    public ServerListener() {
        Bukkit.getPluginManager().registerEvents((Listener)this, Core.getPlugin());
    }

    @EventHandler
    public void onLoad(ServerLoadEvent event) {
        Bukkit.getScheduler().runTask(Core.getPlugin(), () -> {
            this.createDirectories();
            this.initializeDatabase();
            this.processQuests();
            Bukkit.getServer().getOnlinePlayers().stream().forEach(player -> QuestRegistry.getInstance().createQuester((Player)player));
            this.startWatchService();
        });
    }

    @EventHandler
    public void onDisable() {
        Bukkit.getServer().getScheduler().cancelTasks(Core.getPlugin());
        HandlerList.unregisterAll();
        PlayerQuests.getInstance().clear();
        this.stopWatchService();
    }

    private void createDirectories() {
        File questsFolder;
        File dataFolder = Core.getPlugin().getDataFolder();
        if (!dataFolder.exists()) {
            dataFolder.mkdir();
            this.sendWelcomeMessage();
        }
        if (!(questsFolder = new File(Core.getPlugin().getDataFolder(), Core.getQuestsPath())).exists()) {
            questsFolder.mkdirs();
        }
        Core.getPlugin().saveResource(Core.getQuestsPath() + "beans-tester-bonus.json", true);
    }

    private void sendWelcomeMessage() {
        ChatUtils.message("Welcome!").target(ChatUtils.MessageTarget.WORLD).type(ChatUtils.MessageType.NOTIF).send();
    }

    private void initializeDatabase() {
        Database.getInstance().init();
    }

    public void processQuests() {
        File questsDir = new File(Core.getPlugin().getDataFolder(), Core.getQuestsPath());
        HashSet<String> allQuests = new HashSet<String>();
        allQuests.addAll(Database.getInstance().getAllQuests());
        try (Stream<Path> paths = Files.walk(questsDir.toPath(), new FileVisitOption[0]);){
            paths.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).filter(path -> path.toString().endsWith(Core.getQuestFileExtension())).forEach(path -> {
                String questName = this.getQuestName((Path)path);
                allQuests.add(questName);
            });
        }
        catch (IOException e) {
            ChatUtils.message("Could not process the quest files directory/path :(. " + String.valueOf(e)).target(ChatUtils.MessageTarget.CONSOLE).style(ChatUtils.MessageStyle.PLAIN).type(ChatUtils.MessageType.ERROR).send();
        }
        this.loadQuests(allQuests, true);
        ChatUtils.message("Finished submitting quests into server: " + String.valueOf(allQuests)).target(ChatUtils.MessageTarget.CONSOLE).style(ChatUtils.MessageStyle.PLAIN).type(ChatUtils.MessageType.NOTIF).send();
    }

    private String getQuestName(Path path) {
        String[] questNameParts = path.toString().replace(Core.getQuestFileExtension(), "").split("/" + Core.getQuestsPath());
        if (questNameParts.length < 1) {
            return null;
        }
        return questNameParts[1];
    }

    private void loadQuests(Set<String> quests, boolean overwrite) {
        quests.forEach(id -> {
            boolean errorOccurred = true;
            try {
                Quest newQuest = Quest.fromJSONString(FileUtils.get(Core.getQuestsPath() + id + Core.getQuestFileExtension()));
                if (newQuest == null) {
                    return;
                }
                if (QuestRegistry.getInstance().getQuest((String)id, false) != null) {
                    return;
                }
                if (overwrite) {
                    newQuest.save();
                }
                errorOccurred = false;
            }
            catch (JsonMappingException e) {
                ChatUtils.message("Could not map JSON string for: " + id + " to the Quest object. " + String.valueOf(e)).target(ChatUtils.MessageTarget.CONSOLE).type(ChatUtils.MessageType.ERROR).send();
            }
            catch (JsonProcessingException e) {
                ChatUtils.message("JSON in quest: " + id + " is malformed. " + String.valueOf(e)).target(ChatUtils.MessageTarget.CONSOLE).type(ChatUtils.MessageType.ERROR).send();
            }
            catch (IOException e) {
                ChatUtils.message("Could not read file: " + id + ".json. " + String.valueOf(e)).target(ChatUtils.MessageTarget.CONSOLE).type(ChatUtils.MessageType.ERROR).send();
            }
            if (errorOccurred) {
                Database.getInstance().removeQuest((String)id);
            }
        });
    }

    private void startWatchService() {
        try {
            this.watchService = FileSystems.getDefault().newWatchService();
            Path questFilesPath = new File(Core.getPlugin().getDataFolder(), Core.getQuestsPath()).toPath();
            questFilesPath.register(this.watchService, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_MODIFY);
            this.watchThread = new Thread(() -> this.watchServiceRuntime());
            this.watchThread.start();
        }
        catch (IOException e) {
            e.printStackTrace();
        }
    }

    private void watchServiceRuntime() {
        ChatUtils.message("Started watching for changes to plugin files.").style(ChatUtils.MessageStyle.PLAIN).type(ChatUtils.MessageType.NOTIF).target(ChatUtils.MessageTarget.CONSOLE).send();
        while (!Thread.currentThread().isInterrupted()) {
            WatchKey key = this.getWatchServiceKey();
            if (key == null) {
                return;
            }
            this.watchServiceDelegation(key);
            boolean valid = key.reset();
            if (valid) continue;
            break;
        }
    }

    private void watchServiceDelegation(WatchKey key) {
        for (WatchEvent<?> event : key.pollEvents()) {
            WatchEvent.Kind<?> kind = event.kind();
            if (kind == StandardWatchEventKinds.OVERFLOW) continue;
            WatchEvent<?> ev = event;
            Path filename = (Path)ev.context();
            if (kind == StandardWatchEventKinds.ENTRY_DELETE) {
                this.handleFileDeletion();
            }
            if (!filename.toString().endsWith(Core.getQuestFileExtension())) continue;
            this.handleFileChange(filename, kind);
        }
    }

    private void handleFileChange(Path filename, WatchEvent.Kind<?> kind) {
        String questName = filename.toString().replace(Core.getQuestFileExtension(), "");
        QuestRegistry questRegistry = Core.getQuestRegistry();
        if (questName == null) {
            return;
        }
        switch (kind.name()) {
            case "ENTRY_CREATE": {
                this.loadQuests(new HashSet<String>(Set.of(questName)), false);
                break;
            }
            case "ENTRY_DELETE": {
                Quest questToDelete = questRegistry.getQuest(questName);
                if (questToDelete == null) {
                    return;
                }
                questRegistry.delete(questToDelete, false, false, false);
                break;
            }
        }
    }

    private void handleFileDeletion() {
        ChatUtils.message("TODO: Figure out if an important file was deleted just now.").target(ChatUtils.MessageTarget.CONSOLE).type(ChatUtils.MessageType.NOTIF).send();
    }

    private WatchKey getWatchServiceKey() {
        try {
            return this.watchService.take();
        }
        catch (InterruptedException | ClosedWatchServiceException e) {
            Thread.currentThread().interrupt();
            ChatUtils.message("Stopped watching for changes to plugin files.").style(ChatUtils.MessageStyle.PLAIN).type(ChatUtils.MessageType.NOTIF).target(ChatUtils.MessageTarget.CONSOLE).send();
            return null;
        }
    }

    private void stopWatchService() {
        if (this.watchService != null) {
            this.watchThread.interrupt();
            try {
                this.watchService.close();
            }
            catch (IOException e) {
                e.printStackTrace();
            }
            this.watchService = null;
            this.watchThread = null;
        }
    }
}

