/*
 * Decompiled with CFR 0.152.
 */
package de.shiewk.blockhistory.v3;

import de.shiewk.blockhistory.v3.BlockHistoryPlugin;
import de.shiewk.blockhistory.v3.StatManager;
import de.shiewk.blockhistory.v3.exception.LowDiskSpaceException;
import de.shiewk.blockhistory.v3.history.BlockHistoryElement;
import de.shiewk.blockhistory.v3.history.BlockHistorySearchCallback;
import de.shiewk.blockhistory.v3.util.BlockHistoryFileNames;
import de.shiewk.blockhistory.v3.util.NamedLoggingThreadFactory;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.Objects;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.bukkit.Location;
import org.bukkit.World;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;

public final class HistoryManager {
    private final Path saveDirectory;
    private final Logger logger;
    private final StatManager statManager;
    private final ThreadPoolExecutor writeExecutor;
    private final ThreadPoolExecutor readExecutor;

    HistoryManager(Logger logger, Path saveDirectory, StatManager statManager) {
        this.saveDirectory = saveDirectory;
        this.logger = logger;
        this.statManager = statManager;
        AtomicInteger threadNumber = new AtomicInteger();
        String threadName = "BlockHistoryIO";
        int threadPriority = 2;
        this.writeExecutor = new ThreadPoolExecutor(1, 1, 1L, TimeUnit.SECONDS, new ArrayBlockingQueue<Runnable>(1000), new NamedLoggingThreadFactory(threadName, threadPriority, logger, "BlockHistory write I/O", threadNumber));
        this.readExecutor = new ThreadPoolExecutor(0, 3, 30L, TimeUnit.SECONDS, new ArrayBlockingQueue<Runnable>(3), new NamedLoggingThreadFactory(threadName, threadPriority, logger, "BlockHistory read I/O", threadNumber));
    }

    public Path getSaveDirectory() {
        return this.saveDirectory;
    }

    void shutdown() {
        this.logger.info("Shutting down I/O executor...");
        long n = System.nanoTime();
        this.readExecutor.shutdown();
        this.writeExecutor.shutdown();
        try {
            if (!this.readExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.logger.warn("Read executor shutdown timed out after {}ms", (Object)((System.nanoTime() - n) / 1000000L));
            }
            if (!this.writeExecutor.awaitTermination(10L, TimeUnit.SECONDS)) {
                this.logger.warn("Read executor shutdown timed out after {}ms", (Object)((System.nanoTime() - n) / 1000000L));
            }
        }
        catch (InterruptedException e) {
            this.logger.warn("Thread interrupted while waiting for shutdown");
        }
        this.logger.info("Shutdown finished ({}ms)", (Object)((System.nanoTime() - n) / 1000000L));
    }

    public void addHistoryElement(BlockHistoryElement element) {
        Objects.requireNonNull(element);
        this.writeExecutor.execute(() -> {
            try {
                byte[] saveData = element.saveData();
                this.writeToDisk(BlockHistoryFileNames.encode(this.saveDirectory, element.getLocation()), saveData);
                this.statManager.bytesWritten(saveData.length);
                this.statManager.elementWritten();
            }
            catch (LowDiskSpaceException e) {
                BlockHistoryPlugin.logger().warn("Free disk space is too low to safely write further history elements: {} bytes", (Object)e.getFreeBytes());
            }
            catch (Exception e) {
                StringWriter strw = new StringWriter();
                e.printStackTrace(new PrintWriter(strw));
                BlockHistoryPlugin.logger().warn("Exception while writing history element:");
                for (String s : strw.toString().split("\n")) {
                    BlockHistoryPlugin.logger().warn(s);
                }
            }
        });
    }

    private void writeToDisk(Path path, byte[] saveData) throws LowDiskSpaceException, IOException {
        long usableDiskSpace = this.getUsableDiskSpace();
        if (usableDiskSpace < 8192L) {
            throw new LowDiskSpaceException(usableDiskSpace);
        }
        Files.createDirectories(path.getParent(), new FileAttribute[0]);
        Files.write(path, saveData, StandardOpenOption.APPEND, StandardOpenOption.CREATE);
    }

    public long getUsableDiskSpace() throws IOException {
        return Files.getFileStore(this.saveDirectory).getUsableSpace();
    }

    public CompletableFuture<Void> searchAsync(BlockHistorySearchCallback callback, World world, int x, int y, int z) {
        return CompletableFuture.runAsync(() -> {
            try {
                this.search(callback, world, x, y, z);
            }
            catch (FileNotFoundException e) {
                callback.onNoFilePresent(e);
            }
            catch (Throwable e) {
                throw new RuntimeException(e);
            }
        }, this.readExecutor);
    }

    private void search(BlockHistorySearchCallback callback, World world, int x, int y, int z) throws IOException {
        Location location = new Location(world, (double)x, (double)y, (double)z);
        Path path = BlockHistoryFileNames.encode(this.saveDirectory, location);
        try (FileInputStream fin = new FileInputStream(path.toFile());){
            DataInputStream dataIn = new DataInputStream(fin);
            try {
                try {
                    while (true) {
                        int b;
                        BlockHistoryElement element;
                        if ((element = BlockHistoryElement.read(b = dataIn.readUnsignedByte(), dataIn, world, HistoryManager.getChunkX(location), HistoryManager.getChunkZ(location))).x() != x || element.y() != y || element.z() != z) {
                            continue;
                        }
                        callback.onElementFound(element);
                    }
                }
                catch (EOFException e) {
                    dataIn.close();
                    return;
                }
            }
            catch (Throwable throwable) {
                try {
                    dataIn.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
        }
    }

    public static int getChunkZ(Location location) {
        int pos = location.getZ() < 0.0 ? (location.getBlockZ() + 1) / 16 - 1 : location.getBlockZ() / 16;
        return pos;
    }

    public static int getChunkX(Location location) {
        int pos = location.getX() < 0.0 ? (location.getBlockX() + 1) / 16 - 1 : location.getBlockX() / 16;
        return pos;
    }

    public CompletableFuture<DiskSpaceApproximationVisitor> approximateDiskSpaceBytes() {
        return CompletableFuture.supplyAsync(() -> {
            try {
                DiskSpaceApproximationVisitor visitor = new DiskSpaceApproximationVisitor();
                Files.walkFileTree(this.saveDirectory, visitor);
                return visitor;
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }, this.readExecutor);
    }

    public static class DiskSpaceApproximationVisitor
    implements FileVisitor<Path> {
        private long bytes = 0L;
        private long directories = 0L;
        private long files = 0L;

        public long getDiskSpaceBytes() {
            return this.bytes;
        }

        public long getDirectoryCount() {
            return this.directories;
        }

        public long getFileCount() {
            return this.files;
        }

        @Override
        @NotNull
        public FileVisitResult preVisitDirectory(Path dir, @NotNull BasicFileAttributes attrs) {
            ++this.directories;
            this.bytes += (long)dir.getFileName().toString().getBytes().length;
            return FileVisitResult.CONTINUE;
        }

        @Override
        @NotNull
        public FileVisitResult visitFile(Path file, @NotNull BasicFileAttributes attrs) {
            ++this.files;
            this.bytes += (long)file.getFileName().toString().getBytes().length;
            this.bytes += file.toFile().length();
            return FileVisitResult.CONTINUE;
        }

        @Override
        @NotNull
        public FileVisitResult visitFileFailed(Path file, @NotNull IOException exc) throws IOException {
            throw exc;
        }

        @Override
        @NotNull
        public FileVisitResult postVisitDirectory(Path dir, @Nullable IOException exc) throws IOException {
            if (exc != null) {
                throw exc;
            }
            return FileVisitResult.CONTINUE;
        }
    }
}

