package com.provismet.CombatPlusCore.items.component;

import com.mojang.serialization.Codec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import com.provismet.lilylib.util.MoreMath;
import net.minecraft.class_1304;
import net.minecraft.class_1322;
import net.minecraft.class_5134;
import net.minecraft.class_5699;
import net.minecraft.class_9129;
import net.minecraft.class_9135;
import net.minecraft.class_9139;
import net.minecraft.class_9285;

/**
 * <p> Item data component that allows data-driven use of melee and dual weapon damage numbers. </p>
 * <p> This component should be present on all melee weapons. </p>
 *
 * @apiNote The damage numbers used here do not affect standard damage dealt. They only affect special damage callbacks from Combat+ mods, such as the parry damage from Dual Swords.
 * @implNote This is applied automatically as part of the {@code .sword()} and {@code .axe()} Item.Settings methods.
 *
 * @param weaponDamage Damage number to use for mainhand callbacks.
 * @param dualDamage Damage number to use for offhand callbacks.
 */
public record MeleeWeaponComponent (float weaponDamage, float dualDamage) {
    public static final MeleeWeaponComponent DEFAULT = new MeleeWeaponComponent(0, 0);

    public static final Codec<MeleeWeaponComponent> CODEC = RecordCodecBuilder.create(
        instance -> instance.group(
            class_5699.field_53754.fieldOf("weapon_damage").forGetter(MeleeWeaponComponent::weaponDamage),
            class_5699.field_53754.fieldOf("dual_damage").forGetter(MeleeWeaponComponent::dualDamage)
        ).apply(instance, MeleeWeaponComponent::new)
    );
    public static final class_9139<class_9129, MeleeWeaponComponent> PACKET_CODEC = class_9139.method_56435(
        class_9135.field_48552,
        MeleeWeaponComponent::weaponDamage,
        class_9135.field_48552,
        MeleeWeaponComponent::dualDamage,
        MeleeWeaponComponent::new
    );

    /**
     * Creates a component with no dual damage.
     *
     * @param damage The mainhand damage.
     * @return A new MeleeWeaponComponent.
     */
    public static MeleeWeaponComponent createMelee (float damage) {
        return new MeleeWeaponComponent(damage, 0);
    }

    /**
     * Creates a component for a dual weapon, setting the offhand damage to a portion of the mainhand damage.
     *
     * @apiNote The dualDamage value will always be at least 0.5.
     * @param damage The mainhand damage.
     * @return A new MeleeWeaponComponent.
     */
    public static MeleeWeaponComponent createDual (float damage) {
        float offhandDamage = MoreMath.roundDownToMultipleFloat(damage / 3.5f, 0.5f);
        offhandDamage = Math.max(offhandDamage, 0.5f);

        return new MeleeWeaponComponent(damage, offhandDamage);
    }

    /**
     * Creates a component from the mainhand and offhand damage attributes.
     *
     * @param attributes Attributes component containing the damage for the new component.
     * @return A new MeleeWeaponComponent.
     */
    public static MeleeWeaponComponent createFromAttributes (class_9285 attributes) {
        float meleeDamage = 0f;
        float dualDamage = 0f;

        for (class_9285.class_9287 entry : attributes.comp_2393()) {
            if (entry.comp_2395() == class_5134.field_23721 && entry.comp_2396().comp_2450() == class_1322.class_1323.field_6328) {
                if (entry.comp_2397().method_57286(class_1304.field_6173)) meleeDamage += (float)entry.comp_2396().comp_2449();
                if (entry.comp_2397().method_57286(class_1304.field_6171)) dualDamage += (float)entry.comp_2396().comp_2449();
            }
        }
        return new MeleeWeaponComponent(meleeDamage, dualDamage);
    }

    /**
     * Creates a component, creating a dual weapon component from the mainhand damage of the attributes provided.
     *
     * @see MeleeWeaponComponent#createDual
     * @param attributes Attribute component containing mainhand damage.
     * @return A new MeleeWeaponComponent.
     */
    public static MeleeWeaponComponent createDualFromAttributes (class_9285 attributes) {
        float meleeDamage = 0f;
        for (class_9285.class_9287 entry : attributes.comp_2393()) {
            if (entry.comp_2395() == class_5134.field_23721 && entry.comp_2396().comp_2450() == class_1322.class_1323.field_6328 && entry.comp_2397().method_57286(class_1304.field_6173)) {
                meleeDamage += (float)entry.comp_2396().comp_2449();
            }
        }
        return MeleeWeaponComponent.createDual(meleeDamage);
    }

    public boolean isDual () {
        return this.dualDamage > 0;
    }
}
