package com.provismet.CombatPlusCore.utility;

import com.provismet.CombatPlusCore.registries.CPCEnchantmentComponentTypes;
import com.provismet.CombatPlusCore.enchantment.effect.CPCEnchantmentEntityEffect;
import com.provismet.CombatPlusCore.enchantment.loot.context.CPCLootContext;
import it.unimi.dsi.fastutil.objects.Object2IntMap;
import org.apache.commons.lang3.mutable.MutableFloat;

import java.util.List;
import net.minecraft.class_1282;
import net.minecraft.class_1297;
import net.minecraft.class_1304;
import net.minecraft.class_1309;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1887;
import net.minecraft.class_1890;
import net.minecraft.class_3218;
import net.minecraft.class_47;
import net.minecraft.class_5819;
import net.minecraft.class_6880;
import net.minecraft.class_9304;
import net.minecraft.class_9331;
import net.minecraft.class_9334;
import net.minecraft.class_9698;
import net.minecraft.class_9699;
import net.minecraft.class_9723;

/**
 * Enchantment helper to apply hooks from Combat+ enchantment components.
 *
 * @see class_1890
 */
public class CPCEnchantmentHelper {
    /**
     * Replaces the original {@link class_1890#method_60120} to provide gamerule compatible damage and multi-item damage bonuses.
     *
     * @param world The world.
     * @param itemStack The enchanted item.
     * @param target The entity that was attacked.
     * @param damageSource The damage source of the attack.
     * @param baseDamage The initial damage amount for the attack.
     * @return The total damage dealt.
     */
    public static float getDamage (class_3218 world, class_1799 itemStack, class_1297 target, class_1282 damageSource, float baseDamage) {
        MutableFloat damage = new MutableFloat();
        damage.add(CPCEnchantmentHelper.modifyValue(CPCEnchantmentComponentTypes.GAMERULE_DAMAGE, world, itemStack, target, damageSource, 0));
        if (target instanceof class_1657) damage.setValue(damage.floatValue() * world.method_64395().method_20746(CPCGameRules.PVP_DAMAGE_MODIFIER).get());
        damage.add(baseDamage);

        CPCEnchantmentHelper.forEachEnchantment((enchantment, level) -> {
            enchantment.comp_349().method_60041(world, level, itemStack, target, damageSource, damage);
        }, itemStack);

        if (damageSource.method_5529() instanceof class_1309 attacker) {
            for (class_1304 slot : class_1304.values()) {
                class_1799 equippedItem = attacker.method_6118(slot);
                if (equippedItem == null || equippedItem.method_7960()) continue;

                CPCEnchantmentHelper.forEachEnchantment((enchantment, level) -> {
                    if (enchantment.comp_349().method_60026(slot))
                        enchantment.comp_349().method_60035(CPCEnchantmentComponentTypes.BONUS_DAMAGE, world, level, itemStack, attacker, damageSource, damage);
                }, equippedItem);
            }
        }

        return damage.floatValue();
    }

    /**
     * Calls enchantment callbacks for charged hits.
     *
     * @param world The world the effect should occur in.
     * @param user The wielder of the item.
     * @param target The entity that was struck.
     * @param slot The equipment slot to trigger callbacks for.
     */
    public static void postChargedHit (class_3218 world, class_1309 user, class_1309 target, class_1304 slot) {
        CPCEnchantmentHelper.forEachEnchantment((class_6880<class_1887> enchantment, int level, class_9699 context) -> {
            for (class_9698<CPCEnchantmentEntityEffect> effect : enchantment.comp_349().method_60034(CPCEnchantmentComponentTypes.POST_CHARGED_ATTACK)) {
                class_47 conditional = CPCLootContext.createDoubleEntity(world, level, user, target, user.method_6118(slot));
                if (effect.method_60006(conditional)) effect.comp_2680().apply(world, level, context, user, target);
            }
        }, user, slot);
    }

    /**
     * Calls enchantment callbacks for critical hits.
     *
     * @param world The world the effect should occur in.
     * @param user The wielder of the item.
     * @param target The entity that was struck.
     * @param slot The equipment slot to trigger callbacks for.
     */
    public static void postCriticalHit (class_3218 world, class_1309 user, class_1309 target, class_1304 slot) {
        CPCEnchantmentHelper.forEachEnchantment((class_6880<class_1887> enchantment, int level, class_9699 context) -> {
            for (class_9698<CPCEnchantmentEntityEffect> effect : enchantment.comp_349().method_60034(CPCEnchantmentComponentTypes.POST_CRITICAL_ATTACK)) {
                class_47 conditional = CPCLootContext.createDoubleEntity(world, level, user, target, user.method_6118(slot));
                if (effect.method_60006(conditional)) effect.comp_2680().apply(world, level, context, user, target);
            }
        }, user, slot);
    }

    /**
     * Calls enchantment callbacks for kills.
     *
     * @param world The world the effect should occur in.
     * @param user The wielder of the item.
     * @param target The entity that was struck.
     * @param slot The equipment slot to trigger callbacks for.
     */
    public static void postKill (class_3218 world, class_1309 user, class_1309 target, class_1304 slot) {
        CPCEnchantmentHelper.forEachEnchantment((class_6880<class_1887> enchantment, int level, class_9699 context) -> {
            for (class_9698<CPCEnchantmentEntityEffect> effect : enchantment.comp_349().method_60034(CPCEnchantmentComponentTypes.POST_KILL)) {
                class_47 conditional = CPCLootContext.createDoubleEntity(world, level, user, target, user.method_6118(slot));
                if (effect.method_60006(conditional)) effect.comp_2680().apply(world, level, context, user, target);
            }
        }, user, slot);
    }

    public static void postBlock (class_3218 world, class_1309 user, class_1297 attacker, class_1304 slot) {
        CPCEnchantmentHelper.forEachEnchantment((class_6880<class_1887> enchantment, int level, class_9699 context) -> {
            for (class_9698<CPCEnchantmentEntityEffect> effect : enchantment.comp_349().method_60034(CPCEnchantmentComponentTypes.POST_BLOCK)) {
                class_47 conditional = CPCLootContext.createReversedDoubleEntity(world, level, user, attacker, user.method_6118(slot));
                if (effect.method_60006(conditional)) effect.comp_2680().apply(world, level, context, user, attacker);
            }
        }, user, slot);
    }

    /**
     * Modifies a value from a given enchantment component type.
     *
     * @param valueType The type of value to modify.
     * @param random Random source.
     * @param itemStack The enchanted item.
     * @param base The original float value.
     * @return The new float value.
     */
    public static float modifyValue (class_9331<class_9723> valueType, class_5819 random, class_1799 itemStack, float base) {
        MutableFloat value = new MutableFloat(base);
        CPCEnchantmentHelper.forEachEnchantment((enchantment, level) -> enchantment.comp_349().method_60506(valueType, random, level, value), itemStack);
        return value.floatValue();
    }

    /**
     * Modifies a value from a given enchantment component type.
     *
     * @param valueType The type of value to modify.
     * @param world The server-side world.
     * @param itemStack The enchanted item.
     * @param base The original float value.
     * @return The new float value.
     */
    public static float modifyValue (class_9331<List<class_9698<class_9723>>> valueType, class_3218 world, class_1799 itemStack, float base) {
        MutableFloat value = new MutableFloat(base);
        CPCEnchantmentHelper.forEachEnchantment((enchantment, level) -> enchantment.comp_349().method_60037(valueType, world, level, itemStack, value), itemStack);
        return value.floatValue();
    }

    /**
     * Modifies a value from a given enchantment component type.
     *
     * @param valueType The type of value to modify.
     * @param world The server-side world.
     * @param itemStack The enchanted item.
     * @param user The owner of the item.
     * @param base The original float value.
     * @return The new float value.
     */
    public static float modifyValue (class_9331<List<class_9698<class_9723>>> valueType, class_3218 world, class_1799 itemStack, class_1297 user, float base) {
        MutableFloat value = new MutableFloat(base);
        CPCEnchantmentHelper.forEachEnchantment((enchantment, level) -> enchantment.comp_349().method_60036(valueType, world, level, itemStack, user, value), itemStack);
        return value.floatValue();
    }

    /**
     * Modifies a value from a given enchantment component type.
     *
     * @param valueType The type of value to modify.
     * @param world The server-side world.
     * @param itemStack The enchanted itemStack.
     * @param user The owner of the item.
     * @param damageSource The associated damage source.
     * @param base The original float value.
     * @return The new float value.
     */
    public static float modifyValue (class_9331<List<class_9698<class_9723>>> valueType, class_3218 world, class_1799 itemStack, class_1297 user, class_1282 damageSource, float base) {
        MutableFloat value = new MutableFloat(base);
        CPCEnchantmentHelper.forEachEnchantment((enchantment, level) -> enchantment.comp_349().method_60035(valueType, world, level, itemStack, user, damageSource, value), itemStack);
        return value.floatValue();
    }

    public static void forEachEnchantment (Consumer consumer, class_1799 item) {
        class_9304 itemEnchantmentsComponent = item.method_58695(class_9334.field_49633, class_9304.field_49385);
        for (Object2IntMap.Entry<class_6880<class_1887>> entry : itemEnchantmentsComponent.method_57539()) {
            consumer.accept(entry.getKey(), entry.getIntValue());
        }
    }

    /**
     * Iterates over all enchantments on an item, activating the consumer for each.
     *
     * @param consumer A consumer / lambda to be called.
     * @param user The owner of the enchanted item.
     * @param slot The equipment slot the item is in.
     */
    public static void forEachEnchantment (ContextConsumer consumer, class_1309 user, class_1304 slot) {
        CPCEnchantmentHelper.forEachEnchantment(consumer, user, slot, user.method_6118(slot));
    }

    /**
     * Iterates over all enchantments on an item, activating the consumer for each.
     *
     * @param consumer A consumer / lambda to be called.
     * @param user The owner of the enchanted item.
     * @param slot The equipment slot the item is in.
     * @param itemStack The enchanted item.
     */
    public static void forEachEnchantment (ContextConsumer consumer, class_1309 user, class_1304 slot, class_1799 itemStack) {
        if (itemStack.method_7960()) return;

        class_9304 enchantments = itemStack.method_58657();
        if (enchantments.method_57543()) return;

        class_9699 context = new class_9699(itemStack, slot, user);
        for (Object2IntMap.Entry<class_6880<class_1887>> entry : enchantments.method_57539()) {
            if (entry.getKey().comp_349().method_60026(slot)) consumer.accept(entry.getKey(), entry.getIntValue(), context);
        }
    }

    @FunctionalInterface
    public interface Consumer {
        public void accept (class_6880<class_1887> enchantment, int level);
    }

    @FunctionalInterface
    public interface ContextConsumer {
        public void accept (class_6880<class_1887> enchantment, int level, class_9699 context);
    }
}
