document.addEventListener('DOMContentLoaded', function() {
    // Initialize charts
    initPlayerCountChart();
    initPlaytimeChart();
    
    // Update stats
    updateData();
    
    // Set up time period selectors
    document.querySelectorAll('#player-count-selector .time-button').forEach(button => {
        button.addEventListener('click', function() {
            const period = this.getAttribute('data-period');
            
            // Update active button
            document.querySelectorAll('#player-count-selector .time-button').forEach(btn => {
                btn.classList.remove('active');
            });
            this.classList.add('active');
            
            // Update chart with the selected time period
            updatePlayerCountChart(period);
        });
    });
    
    // Auto refresh data every 3 seconds
    setInterval(function() {
        updateData();
    }, 3000); // More frequent updates for better real-time feel
});

// Global chart variables
let playerCountChart;
let playtimeChart;
// Global data storage
let playerCountData = [];
let playerPlaytimeData = [];
let serverStats = {};
let currentPeriod = 'day';

// Main data update function
function updateData() {
    // Subtle animation for updating stats
    document.querySelectorAll('.stats-value').forEach(el => {
        el.classList.add('updating');
    });
    
    // Fetch the data.js file
    fetch('data.js?' + new Date().getTime()) // Add timestamp to bypass cache
        .then(response => {
            if (!response.ok) {
                throw new Error('Network response was not ok');
            }
            return response.text();
        })
        .then(jsContent => {
            // Parse the data more safely
            try {
                // Extract the data using a safer approach with Function constructor
                const dataExtractor = new Function(jsContent + `
                    return {
                        playerCountData: typeof playerCountData !== 'undefined' ? playerCountData : null,
                        playerPlaytimeData: typeof playerPlaytimeData !== 'undefined' ? playerPlaytimeData : null,
                        serverStats: typeof serverStats !== 'undefined' ? serverStats : null
                    };
                `);
                
                const extractedData = dataExtractor();
                
                // Update our global variables if data was successfully extracted
                if (extractedData.playerCountData) {
                    playerCountData = extractedData.playerCountData;
                    updatePlayerCountChart(currentPeriod);
                }
                
                if (extractedData.playerPlaytimeData) {
                    playerPlaytimeData = extractedData.playerPlaytimeData;
                    updatePlaytimeChart();
                }
                
                if (extractedData.serverStats) {
                    serverStats = extractedData.serverStats;
                    updateServerStats();
                }
                
                // Update last refreshed time
                updateLastRefreshed();
                
                // Remove updating animation
                setTimeout(() => {
                    document.querySelectorAll('.stats-value').forEach(el => {
                        el.classList.remove('updating');
                    });
                }, 500);
            } catch (error) {
                console.error('Error processing data.js:', error);
                document.querySelector('footer p').innerHTML = 'Error updating data';
            }
        })
        .catch(error => {
            console.error('Error fetching data.js:', error);
            document.querySelector('footer p').innerHTML = 'Failed to update data';
        });
}

function initPlayerCountChart() {
    const ctx = document.getElementById('player-count-chart').getContext('2d');
    
    playerCountChart = new Chart(ctx, {
        type: 'line',
        data: {
            datasets: [{
                label: 'Player Count',
                backgroundColor: 'rgba(26, 188, 156, 0.2)',
                borderColor: '#1abc9c',
                borderWidth: 3,
                pointBackgroundColor: '#1abc9c',
                pointBorderColor: '#fff',
                pointRadius: 4,
                pointHoverRadius: 6,
                tension: 0.3,
                fill: true
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            scales: {
                x: {
                    type: 'time',
                    time: {
                        unit: 'hour',
                        displayFormats: {
                            hour: 'HH:mm',
                            day: 'MMM d',
                            month: 'MMM yyyy'
                        }
                    },
                    grid: {
                        display: true,
                        color: 'rgba(0, 0, 0, 0.05)'
                    },
                    ticks: {
                        maxRotation: 0,
                        color: '#7f8c8d'
                    },
                    title: {
                        display: true,
                        text: 'Time',
                        color: '#7f8c8d',
                        font: {
                            weight: 'normal'
                        }
                    }
                },
                y: {
                    beginAtZero: true,
                    grid: {
                        display: true,
                        color: 'rgba(0, 0, 0, 0.05)'
                    },
                    ticks: {
                        color: '#7f8c8d',
                        precision: 0
                    },
                    title: {
                        display: true,
                        text: 'Players',
                        color: '#7f8c8d',
                        font: {
                            weight: 'normal'
                        }
                    }
                }
            },
            plugins: {
                legend: {
                    display: false
                },
                tooltip: {
                    backgroundColor: 'rgba(45, 62, 80, 0.9)',
                    titleColor: '#ecf0f1',
                    bodyColor: '#ecf0f1',
                    cornerRadius: 8,
                    boxPadding: 6,
                    callbacks: {
                        title: function(tooltipItems) {
                            const date = new Date(tooltipItems[0].parsed.x);
                            return date.toLocaleString();
                        },
                        label: function(context) {
                            return `Players: ${context.parsed.y}`;
                        }
                    }
                }
            },
            interaction: {
                mode: 'index',
                intersect: false,
            },
            animation: {
                duration: 1000,
                easing: 'easeOutQuart'
            }
        }
    });
}

function updatePlayerCountChart(period) {
    // Store the current period for future refreshes
    currentPeriod = period;
    
    // Filter data based on period
    const now = new Date();
    let filteredData;
    
    switch(period) {
        case 'day':
            filteredData = playerCountData.filter(entry => {
                const entryDate = new Date(entry.time);
                return (now - entryDate) <= 24 * 60 * 60 * 1000;
            });
            playerCountChart.options.scales.x.time.unit = 'hour';
            break;
        case 'week':
            filteredData = playerCountData.filter(entry => {
                const entryDate = new Date(entry.time);
                return (now - entryDate) <= 7 * 24 * 60 * 60 * 1000;
            });
            playerCountChart.options.scales.x.time.unit = 'day';
            break;
        case 'month':
            filteredData = playerCountData.filter(entry => {
                const entryDate = new Date(entry.time);
                return (now - entryDate) <= 30 * 24 * 60 * 60 * 1000;
            });
            playerCountChart.options.scales.x.time.unit = 'day';
            break;
        case 'year':
            filteredData = playerCountData.filter(entry => {
                const entryDate = new Date(entry.time);
                return (now - entryDate) <= 365 * 24 * 60 * 60 * 1000;
            });
            playerCountChart.options.scales.x.time.unit = 'month';
            break;
        default:
            filteredData = playerCountData;
    }
    
    // Sort data chronologically to ensure proper line connection
    filteredData.sort((a, b) => new Date(a.time) - new Date(b.time));
    
    // Handle empty data case - show a flat green line at 0
    if (filteredData.length === 0) {
        // Create default data points
        const startTime = new Date(now);
        const endTime = new Date(now);
        
        switch(period) {
            case 'day':
                startTime.setHours(startTime.getHours() - 24);
                break;
            case 'week':
                startTime.setDate(startTime.getDate() - 7);
                break;
            case 'month':
                startTime.setDate(startTime.getDate() - 30);
                break;
            case 'year':
                startTime.setFullYear(startTime.getFullYear() - 1);
                break;
        }
        
        filteredData = [
            { time: startTime.toISOString(), count: 0 },
            { time: endTime.toISOString(), count: 0 }
        ];
        
        // Change line color to lighter green for empty data
        playerCountChart.data.datasets[0].borderColor = 'rgba(26, 188, 156, 0.5)';
        playerCountChart.data.datasets[0].backgroundColor = 'rgba(26, 188, 156, 0.1)';
    } else {
        // Normal colors for data
        playerCountChart.data.datasets[0].borderColor = '#1abc9c';
        playerCountChart.data.datasets[0].backgroundColor = 'rgba(26, 188, 156, 0.2)';
    }
    
    // Update chart data
    playerCountChart.data.labels = filteredData.map(entry => new Date(entry.time));
    playerCountChart.data.datasets[0].data = filteredData.map(entry => ({
        x: new Date(entry.time),
        y: entry.count
    }));
    
    playerCountChart.update();
}

function initPlaytimeChart() {
    const ctx = document.getElementById('playtime-chart').getContext('2d');
    
    playtimeChart = new Chart(ctx, {
        type: 'bar',
        data: {
            labels: [],
            datasets: [{
                label: 'Playtime (Hours)',
                backgroundColor: '#3498db',
                borderColor: '#2980b9',
                borderWidth: 1,
                borderRadius: 4,
                data: []
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            indexAxis: 'y',
            scales: {
                x: {
                    beginAtZero: true,
                    grid: {
                        display: true,
                        color: 'rgba(0, 0, 0, 0.05)'
                    },
                    ticks: {
                        color: '#7f8c8d'
                    },
                    title: {
                        display: true,
                        text: 'Hours Played',
                        color: '#7f8c8d',
                        font: {
                            weight: 'normal'
                        }
                    }
                },
                y: {
                    grid: {
                        display: false
                    },
                    ticks: {
                        color: '#7f8c8d'
                    }
                }
            },
            plugins: {
                legend: {
                    display: false
                },
                tooltip: {
                    backgroundColor: 'rgba(45, 62, 80, 0.9)',
                    titleColor: '#ecf0f1',
                    bodyColor: '#ecf0f1',
                    cornerRadius: 8,
                    boxPadding: 6,
                    callbacks: {
                        title: function(tooltipItems) {
                            return tooltipItems[0].label;
                        },
                        label: function(context) {
                            const hours = parseFloat(context.parsed.x).toFixed(1);
                            return `${hours} hours played`;
                        }
                    }
                }
            },
            animation: {
                duration: 1000,
                easing: 'easeOutQuart'
            }
        }
    });
    
    updatePlaytimeChart();
}

function updatePlaytimeChart() {
    // If no data or placeholder data, show a message
    if (!playerPlaytimeData || playerPlaytimeData.length === 0 || 
        (playerPlaytimeData.length === 1 && playerPlaytimeData[0].player === "No player data")) {
        playtimeChart.data.labels = ['No player data available'];
        playtimeChart.data.datasets[0].data = [0];
        playtimeChart.data.datasets[0].backgroundColor = '#95a5a6';
    } else {
        // Process and update real data
        playtimeChart.data.labels = playerPlaytimeData.map(entry => entry.player);
        playtimeChart.data.datasets[0].data = playerPlaytimeData.map(entry => (entry.time / 3600).toFixed(2));
        playtimeChart.data.datasets[0].backgroundColor = '#3498db';
    }
    
    playtimeChart.update();
}

function updateServerStats() {
    if (!serverStats) return;
    
    // Update UI with server stats
    document.getElementById('current-player-count').textContent = serverStats.currentPlayerCount;
    
    // Peak players
    document.getElementById('peak-player-count').textContent = serverStats.peakPlayerCount;
    document.getElementById('peak-date').textContent = serverStats.peakPlayerDate || 'No data';
    
    // Format playtime
    const totalHours = Math.floor(serverStats.totalPlaytime / 3600);
    const totalMinutes = Math.floor((serverStats.totalPlaytime % 3600) / 60);
    document.getElementById('total-playtime').textContent = `${totalHours}h ${totalMinutes}m`;
    
    // Format last restart time (server uptime)
    if (serverStats.lastRestartTime) {
        // Make sure we handle both string format and timestamp format
        const restartDate = typeof serverStats.lastRestartTime === 'number' 
            ? new Date(serverStats.lastRestartTime)
            : new Date(serverStats.lastRestartTime);
            
        const timeDiff = Math.floor((Date.now() - restartDate) / 1000);
        const days = Math.floor(timeDiff / (24 * 60 * 60));
        const hours = Math.floor((timeDiff % (24 * 60 * 60)) / (60 * 60));
        const minutes = Math.floor((timeDiff % (60 * 60)) / 60);
        
        let uptime = '';
        if (days > 0) uptime += `${days}d `;
        if (hours > 0 || days > 0) uptime += `${hours}h `;
        uptime += `${minutes}m`;
        
        document.getElementById('last-restart').textContent = uptime;
        console.log("Setting uptime to:", uptime, "from timestamp:", serverStats.lastRestartTime);
    } else {
        document.getElementById('last-restart').textContent = 'Unknown';
    }
}

function updateLastRefreshed() {
    const now = new Date();
    const timeString = now.toLocaleTimeString();
    document.getElementById('last-updated').textContent = `Last updated: ${timeString}`;
}