/*
 * Decompiled with CFR 0.152.
 */
package org.empirewar.orbis.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

public final class QuickHull3D {
    private static final double EPSILON = 1.0E-7;
    private List<Face> faces;
    private List<Point> points;

    public void build(double[] coords) {
        this.build(coords, coords.length / 3);
    }

    public void build(double[] coords, int numPoints) {
        if (numPoints < 4) {
            throw new IllegalArgumentException("Need at least 4 points for 3D convex hull");
        }
        this.points = new ArrayList<Point>();
        for (int i = 0; i < numPoints; ++i) {
            this.points.add(new Point(coords[i * 3], coords[i * 3 + 1], coords[i * 3 + 2], i));
        }
        this.removeDuplicatePoints();
        if (this.points.size() < 4) {
            throw new IllegalArgumentException("Not enough unique points to form a 3D convex hull");
        }
        this.computeConvexHull();
    }

    private void removeDuplicatePoints() {
        ArrayList<Point> uniquePoints = new ArrayList<Point>();
        HashSet<String> seen = new HashSet<String>();
        for (Point p : this.points) {
            String key = String.format("%.6f,%.6f,%.6f", p.x, p.y, p.z);
            if (seen.contains(key)) continue;
            seen.add(key);
            uniquePoints.add(p);
        }
        this.points = uniquePoints;
    }

    private void computeConvexHull() {
        if (!this.findInitialTetrahedron()) {
            throw new IllegalArgumentException("Input points appear to be co-linear or co-planar");
        }
        for (int i = 4; i < this.points.size(); ++i) {
            this.processPoint(i);
        }
    }

    private boolean findInitialTetrahedron() {
        if (this.points.size() < 4) {
            return false;
        }
        Point p0 = this.points.get(0);
        Point p1 = this.points.get(1);
        Point p2 = this.points.get(2);
        Point p3 = this.points.get(3);
        Point v1 = p1.sub(p0);
        Point v2 = p2.sub(p0);
        Point v3 = p3.sub(p0);
        double volume = v1.x * (v2.y * v3.z - v2.z * v3.y) - v1.y * (v2.x * v3.z - v2.z * v3.x) + v1.z * (v2.x * v3.y - v2.y * v3.x);
        if (Math.abs(volume) < 1.0E-7) {
            for (int i = 4; i < this.points.size(); ++i) {
                p3 = this.points.get(i);
                v3 = p3.sub(p0);
                volume = v1.x * (v2.y * v3.z - v2.z * v3.y) - v1.y * (v2.x * v3.z - v2.z * v3.x) + v1.z * (v2.x * v3.y - v2.y * v3.x);
                if (!(Math.abs(volume) > 1.0E-7)) continue;
                Collections.swap(this.points, 3, i);
                break;
            }
            if (Math.abs(volume) < 1.0E-7) {
                return false;
            }
        }
        this.faces = new ArrayList<Face>();
        Point n1 = v1.cross(v2);
        if (n1.dot(p3.sub(p0)) > 0.0) {
            this.faces.add(new Face(0, 1, 2, n1));
        } else {
            this.faces.add(new Face(0, 2, 1, new Point(-n1.x, -n1.y, -n1.z, -1)));
        }
        Point n2 = v2.cross(v3);
        if (n2.dot(p1.sub(p0)) > 0.0) {
            this.faces.add(new Face(0, 2, 3, n2));
        } else {
            this.faces.add(new Face(0, 3, 2, new Point(-n2.x, -n2.y, -n2.z, -1)));
        }
        Point n3 = v3.cross(v1);
        if (n3.dot(p2.sub(p0)) > 0.0) {
            this.faces.add(new Face(0, 3, 1, n3));
        } else {
            this.faces.add(new Face(0, 1, 3, new Point(-n3.x, -n3.y, -n3.z, -1)));
        }
        Point n4 = p3.sub(p1).cross(p2.sub(p1));
        if (n4.dot(p0.sub(p1)) > 0.0) {
            this.faces.add(new Face(1, 3, 2, n4));
        } else {
            this.faces.add(new Face(1, 2, 3, new Point(-n4.x, -n4.y, -n4.z, -1)));
        }
        return true;
    }

    private void processPoint(int pointIndex) {
        Point p = this.points.get(pointIndex);
        ArrayList<Face> visibleFaces = new ArrayList<Face>();
        for (Face face : this.faces) {
            if (!face.visible || !face.isPointOnPositiveSide(p, this.points)) continue;
            face.visible = false;
            visibleFaces.add(face);
        }
        if (visibleFaces.isEmpty()) {
            return;
        }
        HashSet<String> horizonEdges = new HashSet<String>();
        for (Face face : visibleFaces) {
            this.addHorizonEdges(face, horizonEdges, visibleFaces);
        }
        this.faces.removeAll(visibleFaces);
        for (String edge : horizonEdges) {
            String[] indices = edge.split(",");
            int a = Integer.parseInt(indices[0]);
            int b = Integer.parseInt(indices[1]);
            Point pa = this.points.get(a);
            Point pb = this.points.get(b);
            Point normal = pb.sub(pa).cross(p.sub(pa));
            this.faces.add(new Face(a, b, pointIndex, normal));
        }
    }

    private void addHorizonEdges(Face face, Set<String> horizonEdges, List<Face> visibleFaces) {
        int[][] edges;
        for (int[] edge : edges = new int[][]{{face.a, face.b}, {face.b, face.c}, {face.c, face.a}}) {
            String edgeKey = edge[0] + "," + edge[1];
            String reverseKey = edge[1] + "," + edge[0];
            if (horizonEdges.contains(reverseKey)) {
                horizonEdges.remove(reverseKey);
                continue;
            }
            horizonEdges.add(edgeKey);
        }
    }

    public int[][] getFaces() {
        if (this.faces == null) {
            return new int[0][];
        }
        ArrayList<int[]> result = new ArrayList<int[]>();
        for (Face face : this.faces) {
            if (!face.visible) continue;
            result.add(new int[]{face.a, face.b, face.c});
        }
        return (int[][])result.toArray((T[])new int[0][]);
    }

    private record Point(double x, double y, double z, int index) {
        Point sub(Point p) {
            return new Point(this.x - p.x, this.y - p.y, this.z - p.z, -1);
        }

        Point cross(Point p) {
            return new Point(this.y * p.z - this.z * p.y, this.z * p.x - this.x * p.z, this.x * p.y - this.y * p.x, -1);
        }

        double dot(Point p) {
            return this.x * p.x + this.y * p.y + this.z * p.z;
        }

        double distance(Point p) {
            double dx = this.x - p.x;
            double dy = this.y - p.y;
            double dz = this.z - p.z;
            return Math.sqrt(dx * dx + dy * dy + dz * dz);
        }
    }

    private static class Face {
        final int a;
        final int b;
        final int c;
        final Point normal;
        boolean visible = true;

        Face(int a, int b, int c, Point normal) {
            this.a = a;
            this.b = b;
            this.c = c;
            this.normal = normal;
        }

        boolean isPointOnPositiveSide(Point p, List<Point> points) {
            Point pa = points.get(this.a);
            Point pb = points.get(this.b);
            Point pc = points.get(this.c);
            Point v1 = pb.sub(pa);
            Point v2 = pc.sub(pa);
            Point n = v1.cross(v2);
            Point vp = p.sub(pa);
            double dist = vp.dot(n);
            return dist >= -1.0E-7;
        }
    }
}

