package team.creative.creativecore.client.render.text;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

import net.minecraft.client.ComponentCollector;
import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.Font;
import net.minecraft.client.gui.GuiGraphics;
import net.minecraft.locale.Language;
import net.minecraft.network.chat.Component;
import net.minecraft.network.chat.FormattedText;
import net.minecraft.network.chat.Style;
import net.minecraft.util.Mth;
import net.minecraft.util.StringDecomposer;
import team.creative.creativecore.CreativeCore;
import team.creative.creativecore.common.gui.Align;
import team.creative.creativecore.common.gui.VAlign;
import team.creative.creativecore.common.util.mc.ColorUtils;
import team.creative.creativecore.common.util.text.AdvancedComponentHelper;
import team.creative.creativecore.common.util.text.content.AdvancedContent;
import team.creative.creativecore.common.util.text.content.AdvancedContentConsumer;
import team.creative.creativecore.common.util.text.content.AdvancedFormattedText;
import team.creative.creativecore.common.util.type.list.SingletonList;

public class CompiledText {
    
    public static CompiledText createAnySize() {
        return new CompiledText(Integer.MAX_VALUE, Integer.MAX_VALUE);
    }
    
    private static int width(FormattedText text) {
        if (text instanceof AdvancedFormattedText adv)
            return adv.width(AdvancedComponentHelper.SPLITTER.width, Style.EMPTY);
        return Mth.ceil(AdvancedComponentHelper.SPLITTER.stringWidth(text));
    }
    
    private static int lineHeight(FormattedText text) {
        if (text instanceof AdvancedContent adv)
            return adv.height();
        if (text instanceof Component comp && comp.getContents() instanceof AdvancedContent adv)
            return adv.height();
        return AdvancedComponentHelper.SPLITTER.lineHeight;
    }
    
    public static final CompiledText EMPTY = new CompiledText(0, 0) {
        
        {
            original = Collections.EMPTY_LIST;
            lines = Collections.EMPTY_LIST;
        }
        
        @Override
        public void setText(Component component) {}
        
        @Override
        public void setText(List<Component> components) {}
        
        @Override
        protected void compile() {}
        
        @Override
        public int getTotalHeight() {
            return 0;
        }
        
        @Override
        public void render(GuiGraphics graphics) {}
    };
    
    private int maxWidth;
    private int maxWidthScaled;
    private int maxHeight;
    private int maxHeightScaled;
    private int usedWidth;
    private int usedHeight;
    private int lineSpacing = 2;
    private boolean shadow = true;
    private int defaultColor = ColorUtils.WHITE;
    private Align align = Align.LEFT;
    private VAlign valign = VAlign.TOP;
    protected List<CompiledLine> lines;
    protected List<Component> original;
    private double scale = 1;
    
    public CompiledText(int width, int height) {
        this.maxWidth = this.maxWidthScaled = width;
        
        this.maxHeight = this.maxHeightScaled = height;
        setText(Collections.EMPTY_LIST);
    }
    
    public void setMaxHeight(int height) {
        this.maxHeight = height;
        this.maxHeightScaled = (int) (maxHeight * scale);
    }
    
    public void setDimension(int width, int height) {
        this.maxWidth = width;
        this.maxWidthScaled = (int) (maxWidth / scale);
        this.maxHeight = height;
        this.maxHeightScaled = (int) (maxHeight / scale);
        compile();
    }
    
    public int getMaxWidth() {
        return maxWidth;
    }
    
    public int getMaxHeight() {
        return maxHeight;
    }
    
    public int getDefaultColor() {
        return this.defaultColor;
    }
    
    public void setDefaultColor(int color) {
        this.defaultColor = color;
    }
    
    public void setShadow(boolean shadow) {
        this.shadow = shadow;
    }
    
    public void setAlign(Align align) {
        this.align = align;
    }
    
    public void setVAlign(VAlign valign) {
        this.valign = valign;
    }
    
    public void setScale(double scale) {
        this.scale = scale;
        this.maxWidthScaled = (int) (maxWidth / scale);
        this.maxHeightScaled = (int) (maxHeight / scale);
        compile();
    }
    
    public double getScale() {
        return scale;
    }
    
    public void setText(Component component) {
        setText(new SingletonList<>(component));
    }
    
    public void setText(List<Component> components) {
        this.original = components;
        compile();
    }
    
    protected void compile() {
        if (CreativeCore.loader().getOverallSide().isServer())
            return;
        
        List<Component> copy = new ArrayList<>();
        for (Component component : original)
            copy.add(AdvancedComponentHelper.copy(component));
        lines = new ArrayList<>();
        compileNext(null, true, copy);
    }
    
    private CompiledLine compileNext(CompiledLine currentLine, boolean newLine, List<? extends FormattedText> components) {
        for (FormattedText component : components) {
            if (newLine)
                lines.add(currentLine = new CompiledLine());
            currentLine = compileNext(currentLine, component);
        }
        return currentLine;
    }
    
    private CompiledLine compileNext(CompiledLine currentLine, boolean newLine, FormattedText component) {
        if (newLine)
            lines.add(currentLine = new CompiledLine());
        return compileNext(currentLine, component);
    }
    
    private CompiledLine compileNext(CompiledLine currentLine, FormattedText component) {
        List<Component> siblings = null;
        if (component instanceof Component && !((Component) component).getSiblings().isEmpty()) {
            siblings = new ArrayList<>(((Component) component).getSiblings());
            ((Component) component).getSiblings().clear();
        }
        FormattedText next = currentLine.add(component);
        
        if (next != null) {
            lines.add(currentLine = new CompiledLine());
            currentLine = compileNext(currentLine, false, next);
        }
        
        if (siblings != null)
            currentLine = compileNext(currentLine, false, siblings);
        return currentLine;
    }
    
    public int getTotalHeight() {
        int height = -lineSpacing;
        for (CompiledLine line : lines)
            height += line.height + lineSpacing;
        return Mth.ceil(height * scale);
    }
    
    public void render(GuiGraphics graphics) {
        if (lines == null)
            return;
        
        usedWidth = 0;
        usedHeight = -lineSpacing;
        
        int totalHeight = getTotalHeight();
        var stack = graphics.pose();
        
        if (scale != 1) {
            stack.pushMatrix();
            stack.scale((float) scale, (float) scale);
        }
        stack.pushMatrix();
        float y = Math.max(0, switch (valign) {
            case CENTER -> maxHeightScaled / 2 - totalHeight / 2;
            case BOTTOM -> maxHeightScaled - totalHeight;
            default -> 0;
        });
        stack.translate(0, y);
        usedHeight += (int) y;
        
        for (CompiledLine line : lines) {
            switch (align) {
                case CENTER -> {
                    int x = maxWidthScaled / 2 - line.width / 2;
                    stack.translate(x, 0);
                    line.render(graphics);
                    stack.translate(-x, 0);
                    usedWidth = Math.max(usedWidth, maxWidthScaled);
                }
                case RIGHT -> {
                    int x = maxWidthScaled - line.width;
                    stack.translate(x, 0);
                    line.render(graphics);
                    stack.translate(-x, 0);
                    usedWidth = Math.max(usedWidth, maxWidthScaled);
                }
                default -> {
                    line.render(graphics);
                    usedWidth = Math.max(usedWidth, line.width);
                }
            };
            
            int height = line.height + lineSpacing;
            stack.translate(0, height);
            usedHeight += height;
            
            if (usedHeight > maxHeightScaled)
                break;
        }
        
        stack.popMatrix();
        if (scale != 1)
            stack.popMatrix();
        
        usedWidth *= scale;
        usedHeight *= scale;
    }
    
    public class CompiledLine {
        
        private final List<FormattedText> components = new ArrayList<>();
        private int height = 0;
        private int width = 0;
        
        public CompiledLine() {}
        
        public boolean contains(String search) {
            for (FormattedText text : components)
                if (text.getString().contains(search))
                    return true;
            return false;
        }
        
        public void render(GuiGraphics graphics) {
            Font font = Minecraft.getInstance().font;
            var pose = graphics.pose();
            
            int xOffset = 0;
            for (FormattedText text : components) {
                int height = lineHeight(text);
                int width = width(text);
                
                int yOffset = 0;
                if (height < this.height)
                    yOffset = (this.height - height) / 2;
                pose.pushMatrix();
                pose.translate(xOffset, yOffset);
                if (text instanceof AdvancedFormattedText adv)
                    adv.render(graphics, defaultColor);
                else
                    graphics.drawString(font, Language.getInstance().getVisualOrder(text), 0, 0, defaultColor, shadow);
                pose.popMatrix();
                xOffset += width;
            }
            
        }
        
        public void updateDimension(int width, int height) {
            this.width = Math.max(width, this.width);
            this.height = Math.max(height, this.height);
        }
        
        public FormattedText add(FormattedText component) {
            int remainingWidth = maxWidthScaled - width;
            int textWidth = width(component);
            if (remainingWidth >= textWidth) {
                if (component instanceof Component comp && comp.getContents() instanceof AdvancedContent adv)
                    components.add(adv.asText());
                else
                    components.add(component);
                updateDimension(width + textWidth, lineHeight(component));
                return null;
            } else {
                FormattedTextSplit split = splitByWidth(component, remainingWidth, Style.EMPTY, width == 0);
                if (split != null && (split.head != null || width == 0)) {
                    if (split.head != null) {
                        updateDimension(width + width(split.head), lineHeight(split.head));
                        components.add(split.head);
                        return split.tail;
                    }
                    updateDimension(width + width(split.tail), lineHeight(split.tail));
                    components.add(split.tail);
                    return null;
                } else if (width == 0)
                    return null;
                else
                    return component;
            }
        }
        
    }
    
    public FormattedTextSplit splitByWidth(FormattedText text, int width, Style style, boolean force) {
        final WidthLimitedCharSink charSink = new WidthLimitedCharSink(width, Minecraft.getInstance().font.getSplitter());
        ComponentCollector head = new ComponentCollector();
        ComponentCollector tail = new ComponentCollector();
        
        if (text instanceof Component comp) {
            AdvancedComponentHelper.visit(comp, new AdvancedContentConsumer() {
                
                @Override
                public Optional accept(Style style, AdvancedContent content) {
                    if (charSink.accept(style, content) || force)
                        head.append(content.asText());
                    else
                        tail.append(content.asText());
                    return Optional.empty();
                }
                
                @Override
                public Optional accept(Style style, String text) {
                    charSink.resetPosition();
                    if (!StringDecomposer.iterateFormatted(text, style, charSink)) {
                        Linebreaker breaker = charSink.lastBreaker();
                        if (force || breaker != null) {
                            String sHead;
                            String sTail;
                            if (breaker != null) {
                                int pos = charSink.lastBreakerPos();
                                sHead = text.substring(0, pos + (breaker.includeChar && breaker.head ? 1 : 0));
                                sTail = text.substring(pos + (breaker.includeChar && !breaker.head ? 0 : 1));
                            } else {
                                sHead = text.substring(0, charSink.getPosition());
                                sTail = text.substring(charSink.getPosition());
                            }
                            if (!sHead.isEmpty())
                                head.append(FormattedText.of(sHead, style));
                            if (!sTail.isEmpty())
                                tail.append(FormattedText.of(sTail, style));
                        } else
                            tail.append(FormattedText.of(text, style));
                    } else if (!text.isEmpty())
                        head.append(FormattedText.of(text, style));
                    return Optional.empty();
                }
            }, style);
        } else {
            text.visit(new FormattedText.StyledContentConsumer<FormattedText>() {
                
                @Override
                public Optional<FormattedText> accept(Style style, String text) {
                    charSink.resetPosition();
                    if (!StringDecomposer.iterateFormatted(text, style, charSink)) {
                        Linebreaker breaker = charSink.lastBreaker();
                        if (force || breaker != null) {
                            String sHead;
                            String sTail;
                            if (breaker != null) {
                                int pos = charSink.lastBreakerPos();
                                sHead = text.substring(0, pos + (breaker.includeChar && breaker.head ? 1 : 0));
                                sTail = text.substring(pos + (breaker.includeChar && !breaker.head ? 0 : 1));
                            } else {
                                sHead = text.substring(0, charSink.getPosition());
                                sTail = text.substring(charSink.getPosition());
                            }
                            if (!sHead.isEmpty())
                                head.append(FormattedText.of(sHead, style));
                            if (!sTail.isEmpty())
                                tail.append(FormattedText.of(sTail, style));
                        } else
                            tail.append(FormattedText.of(text, style));
                    } else if (!text.isEmpty())
                        head.append(FormattedText.of(text, style));
                    return Optional.empty();
                }
            }, style);
        }
        
        FormattedText headText = head.getResult();
        FormattedText tailText = tail.getResult();
        
        if (headText == null && tailText == null)
            return null;
        return new FormattedTextSplit(headText, tailText);
    }
    
    public record FormattedTextSplit(FormattedText head, FormattedText tail) {
        
    }
    
    public int getTotalWidth() {
        return Mth.ceil(calculateWidth(0, true, original) * scale);
    }
    
    private int calculateWidth(int width, boolean newLine, List<? extends FormattedText> components) {
        for (FormattedText component : components) {
            int result = width(component);
            if (newLine)
                width = Math.max(width, result);
            else
                width += result;
        }
        return width;
    }
    
    public CompiledText sameDimensions() {
        CompiledText copy = new CompiledText(maxWidth, maxHeight);
        copy.align = align;
        copy.valign = valign;
        copy.lineSpacing = lineSpacing;
        copy.scale = scale;
        copy.maxWidthScaled = maxWidthScaled;
        copy.maxHeightScaled = maxHeightScaled;
        return copy;
    }
    
    public CompiledText copy() {
        CompiledText copy = new CompiledText(maxWidth, maxHeight);
        copy.align = align;
        copy.valign = valign;
        copy.defaultColor = defaultColor;
        copy.lineSpacing = lineSpacing;
        copy.shadow = shadow;
        copy.scale = scale;
        copy.maxWidthScaled = maxWidthScaled;
        copy.maxHeightScaled = maxHeightScaled;
        List<Component> components = new ArrayList<>();
        for (Component component : original)
            components.add(component.copy());
        copy.setText(components);
        return copy;
    }
    
    public boolean contains(String search) {
        for (CompiledLine line : lines)
            if (line.contains(search))
                return true;
        return false;
    }
    
    public int getUsedWidth() {
        return usedWidth;
    }
    
    public int getUsedHeight() {
        return usedHeight;
    }
    
    public Iterable<Component> untrimmedContent() {
        return original;
    }
    
}
