package team.creative.creativecore.common.config.gui;

import java.util.ArrayList;
import java.util.List;

import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.world.level.GameType;
import team.creative.creativecore.common.gui.GuiParent;
import team.creative.creativecore.common.gui.control.collection.GuiListBoxBase;
import team.creative.creativecore.common.gui.control.simple.GuiButton;
import team.creative.creativecore.common.gui.control.simple.GuiStateButton;
import team.creative.creativecore.common.gui.control.simple.GuiTextfield;
import team.creative.creativecore.common.gui.event.GuiControlChangedEvent;
import team.creative.creativecore.common.util.player.PlayerSelector;
import team.creative.creativecore.common.util.player.PlayerSelector.PlayerSelectorAnd;
import team.creative.creativecore.common.util.player.PlayerSelector.PlayerSelectorCommandSelector;
import team.creative.creativecore.common.util.player.PlayerSelector.PlayerSelectorGamemode;
import team.creative.creativecore.common.util.player.PlayerSelector.PlayerSelectorLevel;
import team.creative.creativecore.common.util.player.PlayerSelector.PlayerSelectorNot;
import team.creative.creativecore.common.util.player.PlayerSelector.PlayerSelectorOr;
import team.creative.creativecore.common.util.registry.NamedHandlerRegistry;
import team.creative.creativecore.common.util.text.TextMapBuilder;

@Environment(EnvType.CLIENT)
public abstract class GuiPlayerSelectorHandler<T extends PlayerSelector> {
    
    public static final NamedHandlerRegistry<GuiPlayerSelectorHandler> REGISTRY = new NamedHandlerRegistry<>(null);
    
    public static GuiPlayerSelectorHandler get(PlayerSelector selector) {
        return REGISTRY.get(PlayerSelector.REGISTRY.getId(selector));
    }
    
    static {
        REGISTRY.register("and", new GuiPlayerSelectorHandlerMultiple<PlayerSelectorAnd>() {
            
            @Override
            public PlayerSelectorAnd parseSelector(PlayerSelector[] selectors) {
                return new PlayerSelectorAnd(selectors);
            }
        });
        REGISTRY.register("or", new GuiPlayerSelectorHandlerMultiple<PlayerSelectorOr>() {
            
            @Override
            public PlayerSelectorOr parseSelector(PlayerSelector[] selectors) {
                return new PlayerSelectorOr(selectors);
            }
        });
        REGISTRY.register("not", new GuiPlayerSelectorHandler<PlayerSelectorNot>() {
            
            @Override
            public void createControls(GuiParent gui, PlayerSelector selector) {
                gui.add(new GuiPlayerSelectorButton(gui, "not", selector instanceof PlayerSelectorNot select ? select.selector : new PlayerSelectorLevel(0)));
            }
            
            @Override
            public PlayerSelectorNot parseSelector(GuiParent gui) {
                GuiPlayerSelectorButton button = gui.get("not");
                PlayerSelector selector = button.get();
                if (selector != null)
                    return new PlayerSelectorNot(selector);
                return null;
            }
        });
        REGISTRY.register("level", new GuiPlayerSelectorHandler<PlayerSelectorLevel>() {
            
            @Override
            public void createControls(GuiParent gui, PlayerSelector selector) {
                gui.add(new GuiTextfield(gui, "content", selector instanceof PlayerSelectorLevel select ? "" + select.permissionLevel : "0").setNumbersOnly());
            }
            
            @Override
            public PlayerSelectorLevel parseSelector(GuiParent gui) {
                GuiTextfield text = gui.get("content");
                return new PlayerSelectorLevel(text.parseInteger());
            }
            
        });
        REGISTRY.register("mode", new GuiPlayerSelectorHandler<PlayerSelectorGamemode>() {
            
            @Override
            public void createControls(GuiParent gui, PlayerSelector selector) {
                gui.add(
                    new GuiStateButton<GameType>(gui, "mode", selector instanceof PlayerSelectorGamemode select ? select.type : GameType.SURVIVAL, new TextMapBuilder<GameType>()
                            .addComponent(GameType.values(), x -> x.getShortDisplayName())));
            }
            
            @Override
            public PlayerSelectorGamemode parseSelector(GuiParent gui) {
                GuiStateButton<GameType> mode = gui.get("mode");
                return new PlayerSelectorGamemode(mode.selected());
            }
            
        });
        REGISTRY.register("selector", new GuiPlayerSelectorHandler<PlayerSelectorCommandSelector>() {
            
            @Override
            public void createControls(GuiParent gui, PlayerSelector selector) {
                gui.add(new GuiTextfield(gui, "content", selector instanceof PlayerSelectorCommandSelector select ? select.pattern : "@a[]"));
            }
            
            @Override
            public PlayerSelectorCommandSelector parseSelector(GuiParent gui) {
                GuiTextfield text = gui.get("content");
                if (text.getText().isEmpty())
                    return null;
                return new PlayerSelectorCommandSelector(text.getText());
            }
            
        });
    }
    
    public String getName() {
        return REGISTRY.getId(this);
    }
    
    public abstract void createControls(GuiParent parent, PlayerSelector selector);
    
    public abstract T parseSelector(GuiParent parent);
    
    public void onChanged(PlayerSelectorDialog gui, GuiControlChangedEvent event) {}
    
    public static abstract class GuiPlayerSelectorHandlerMultiple<T extends PlayerSelector> extends GuiPlayerSelectorHandler<T> {
        
        public PlayerSelector[] getChildren(PlayerSelector selector) {
            if (selector instanceof PlayerSelectorAnd)
                return ((PlayerSelectorAnd) selector).selectors;
            else if (selector instanceof PlayerSelectorOr)
                return ((PlayerSelectorOr) selector).selectors;
            return new PlayerSelector[] { selector };
        }
        
        @Override
        public void createControls(GuiParent gui, PlayerSelector selector) {
            PlayerSelector[] selectors = getChildren(selector);
            List<GuiPlayerSelectorButton> buttons = new ArrayList<>();
            if (selectors != null)
                for (int i = 0; i < selectors.length; i++)
                    buttons.add(new GuiPlayerSelectorButton(gui, "" + i, selectors[i]));
            GuiListBoxBase<GuiPlayerSelectorButton> list = new GuiListBoxBase<>(gui, "list", true, buttons);
            gui.add(list.setExpandable());
            gui.add(new GuiButton(gui, "add", x -> list.addItem(new GuiPlayerSelectorButton(gui, "new", new PlayerSelectorLevel(0)))).setTranslate("gui.add"));
        }
        
        @Override
        public T parseSelector(GuiParent gui) {
            GuiListBoxBase<GuiPlayerSelectorButton> list = gui.get("list");
            PlayerSelector[] selectors = new PlayerSelector[list.size()];
            for (int i = 0; i < selectors.length; i++)
                selectors[i] = list.getItem(i).get();
            if (selectors.length > 0)
                return parseSelector(selectors);
            return null;
        }
        
        public abstract T parseSelector(PlayerSelector[] selectors);
    }
    
}
