package team.creative.creativecore.common.gui.control.parent;

import com.mojang.blaze3d.systems.RenderSystem;
import com.mojang.blaze3d.vertex.PoseStack;

import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.client.gui.GuiGraphics;
import net.minecraft.client.gui.screens.Screen;
import net.minecraft.sounds.SoundEvents;
import net.neoforged.api.distmarker.Dist;
import net.neoforged.api.distmarker.OnlyIn;
import team.creative.creativecore.common.gui.GuiControl;
import team.creative.creativecore.common.gui.GuiParent;
import team.creative.creativecore.common.gui.flow.GuiFlow;
import team.creative.creativecore.common.gui.style.ControlFormatting;
import team.creative.creativecore.common.gui.style.ControlFormatting.ControlStyleFace;
import team.creative.creativecore.common.gui.style.GuiStyle;
import team.creative.creativecore.common.util.math.geo.Rect;
import team.creative.creativecore.common.util.math.vec.SmoothValue;

public class GuiScrollXY extends GuiParent {
    
    public int maxScrollX = 0;
    public SmoothValue scrolledX = new SmoothValue(200);
    public boolean draggedX;
    
    public int maxScrollY = 0;
    public SmoothValue scrolledY = new SmoothValue(200);
    public boolean draggedY;
    
    public int scrollbarThickness = 3;
    public ControlStyleFace scrollbarFace = ControlStyleFace.CLICKABLE;
    
    public boolean alternativeScrolling = false;
    
    protected int cachedWidth;
    protected int cachedHeight;
    
    public GuiScrollXY() {
        this("");
    }
    
    public GuiScrollXY(String name) {
        this(name, GuiFlow.STACK_X);
    }
    
    public GuiScrollXY(String name, GuiFlow flow) {
        super(name, flow);
    }
    
    @Override
    public double getOffsetX() {
        return -scrolledX.current();
    }
    
    @Override
    public double getOffsetY() {
        return -scrolledY.current();
    }
    
    @Override
    public ControlFormatting getControlFormatting() {
        return ControlFormatting.NESTED;
    }
    
    public void onScrolledX() {
        if (this.scrolledX.aimed() < 0)
            this.scrolledX.set(0);
        if (this.scrolledX.aimed() > maxScrollX)
            this.scrolledX.set(maxScrollX);
    }
    
    public void onScrolledY() {
        if (this.scrolledY.aimed() < 0)
            this.scrolledY.set(0);
        if (this.scrolledY.aimed() > maxScrollY)
            this.scrolledY.set(maxScrollY);
    }
    
    @Override
    public boolean mouseScrolled(double x, double y, double scrolled) {
        if (super.mouseScrolled(x, y, scrolled))
            return true;
        
        scroll(scrolled);
        return true;
    }
    
    public void scroll(double scrolled) {
        if (alternativeScrolling) {
            if (Screen.hasShiftDown()) {
                if (needsScrollbarX()) {
                    this.scrolledX.set(this.scrolledX.aimed() - scrolled * 10);
                    onScrolledX();
                }
                return;
            }
            
            this.scrolledY.set(this.scrolledY.aimed() - scrolled * 10);
            onScrolledY();
            return;
        }
        
        boolean shouldScrollY = needsScrollbarY();
        if (shouldScrollY)
            if (scrolled > 0 && this.scrolledY.aimed() == 0)
                shouldScrollY = false;
            else if (scrolled < 0 && this.scrolledY.aimed() == maxScrollY)
                shouldScrollY = false;
            
        if (shouldScrollY) {
            this.scrolledY.set(this.scrolledY.aimed() - scrolled * 10);
            onScrolledY();
            return;
        }
        
        this.scrolledX.set(this.scrolledX.aimed() - scrolled * 10);
        onScrolledX();
    }
    
    @Override
    public boolean mouseClicked(double x, double y, int button) {
        if (button == 0 && rect.getHeight() - y <= scrollbarThickness && needsScrollbarX()) {
            playSound(SoundEvents.UI_BUTTON_CLICK);
            draggedX = true;
            return true;
        }
        if (button == 0 && rect.getWidth() - x <= scrollbarThickness && needsScrollbarY()) {
            playSound(SoundEvents.UI_BUTTON_CLICK);
            draggedY = true;
            return true;
        }
        return super.mouseClicked(x, y, button);
    }
    
    @Override
    public void mouseMoved(double x, double y) {
        if (draggedX) {
            GuiStyle style = getStyle();
            ControlFormatting formatting = getControlFormatting();
            int completeWidth = rect.getWidth() - style.getBorder(formatting.border) * 2;
            
            int scrollThingWidth = Math.max(10, Math.min(completeWidth, (int) ((float) completeWidth / cachedWidth * completeWidth)));
            if (cachedWidth < completeWidth)
                scrollThingWidth = completeWidth;
            
            double percent = (x) / (completeWidth - scrollThingWidth);
            this.scrolledX.set((int) (percent * maxScrollX));
            onScrolledX();
        }
        if (draggedY) {
            GuiStyle style = getStyle();
            ControlFormatting formatting = getControlFormatting();
            int completeHeight = rect.getHeight() - style.getBorder(formatting.border) * 2;
            
            int scrollThingHeight = Math.max(10, Math.min(completeHeight, (int) ((float) completeHeight / cachedHeight * completeHeight)));
            if (cachedHeight < completeHeight)
                scrollThingHeight = completeHeight;
            
            double percent = (y) / (completeHeight - scrollThingHeight);
            this.scrolledY.set((int) (percent * maxScrollY));
            onScrolledY();
        }
        super.mouseMoved(x, y);
    }
    
    @Override
    public void mouseReleased(double x, double y, int button) {
        super.mouseReleased(x, y, button);
        draggedX = draggedY = false;
    }
    
    public boolean needsScrollbarX() {
        return cachedWidth > rect.getContentWidth();
    }
    
    public boolean needsScrollbarY() {
        return cachedHeight > rect.getContentHeight();
    }
    
    @Override
    @Environment(EnvType.CLIENT)
    @OnlyIn(Dist.CLIENT)
    protected void renderContent(GuiGraphics graphics, ControlFormatting formatting, int borderWidth, Rect controlRect, Rect realRect, double scale, int mouseX, int mouseY) {
        PoseStack pose = graphics.pose();
        pose.pushPose();
        super.renderContent(graphics, formatting, borderWidth, controlRect, realRect, scale, mouseX, mouseY);
        pose.popPose();
        
        float controlInvScale = (float) scaleFactorInv();
        pose.scale(controlInvScale, controlInvScale, controlInvScale);
        
        realRect.scissor();
        GuiStyle style = getStyle();
        
        RenderSystem.disableDepthTest();
        
        scrolledX.tick();
        
        if (needsScrollbarX()) {
            int completeWidth = rect.getWidth() - borderWidth * 2;
            
            int scrollThingWidth = Math.max(10, Math.min(completeWidth, (int) ((float) completeWidth / cachedWidth * completeWidth)));
            if (cachedWidth < completeWidth)
                scrollThingWidth = completeWidth;
            double percent = scrolledX.current() / maxScrollX;
            
            style.get(scrollbarFace, false).render(graphics, (int) (percent * (completeWidth - scrollThingWidth)) + borderWidth, rect
                    .getHeight() - scrollbarThickness - borderWidth, scrollThingWidth, scrollbarThickness);
            
            maxScrollX = Math.max(0, (cachedWidth - completeWidth) + formatting.padding * 2 + 1);
            double newScroll = Math.clamp(scrolledX.aimed(), 0, maxScrollX);
            if (newScroll != scrolledX.aimed())
                scrolledX.set(newScroll);
        } else
            scrolledX.set(0);
        
        scrolledY.tick();
        
        if (needsScrollbarY()) {
            int completeHeight = rect.getHeight() - borderWidth * 2;
            
            int scrollThingHeight = Math.max(10, Math.min(completeHeight, (int) ((float) completeHeight / cachedHeight * completeHeight)));
            if (cachedHeight < completeHeight)
                scrollThingHeight = completeHeight;
            double percent = scrolledY.current() / maxScrollY;
            
            style.get(scrollbarFace, false).render(graphics, rect.getWidth() - scrollbarThickness - borderWidth,
                (int) (percent * (completeHeight - scrollThingHeight)) + borderWidth, scrollbarThickness, scrollThingHeight);
            
            maxScrollY = Math.max(0, (cachedHeight - completeHeight) + formatting.padding * 2 + 1);
            double newScroll = Math.clamp(scrolledY.aimed(), 0, maxScrollY);
            if (newScroll != scrolledY.aimed())
                scrolledY.set(newScroll);
        } else
            scrolledY.set(0);
        
        float controlScale = (float) scaleFactor();
        pose.scale(controlScale, controlScale, controlScale);
        
        RenderSystem.enableDepthTest();
    }
    
    @Override
    protected int minWidth(int availableWidth) {
        return 10;
    }
    
    @Override
    protected int minHeight(int width, int availableHeight) {
        return 10;
    }
    
    @Override
    public void flowX(int width, int preferred) {
        super.flowX(preferred, preferred);
        updateWidth();
    }
    
    protected void updateWidth() {
        int maxX = 0;
        for (GuiControl control : controls)
            maxX = Math.max(control.rect.getRight(), maxX);
        cachedWidth = maxX;
    }
    
    @Override
    public void flowY(int width, int height, int preferred) {
        super.flowY(cachedWidth, preferred, preferred);
        updateHeight();
    }
    
    protected void updateHeight() {
        int maxY = 0;
        for (GuiControl control : controls)
            maxY = Math.max(control.rect.getBottom(), maxY);
        cachedHeight = maxY;
    }
}
