package com.drathonix.loadmychunks.common.system;


import com.drathonix.loadmychunks.common.bridge.ILevelChunkMixin;
import com.drathonix.loadmychunks.common.config.LMCConfig;
import com.drathonix.loadmychunks.common.system.control.ILoadState;
import com.drathonix.loadmychunks.common.system.loaders.IChunkLoader;
import com.drathonix.loadmychunks.common.system.loaders.IOwnable;
import it.unimi.dsi.fastutil.longs.Long2ObjectLinkedOpenHashMap;
import it.unimi.dsi.fastutil.longs.LongOpenHashSet;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.function.Supplier;
import net.minecraft.class_156;
import net.minecraft.class_18;
import net.minecraft.class_1923;
import net.minecraft.class_2338;
import net.minecraft.class_2487;
import net.minecraft.class_3218;
import net.minecraft.class_5268;

/**
 * Maintains the records of all chunk loaders and chunk load states handled by LoadMyChunks.
 */
public class ChunkDataManager {
    private static final Map<class_3218,LevelChunkLoaderManager> levelManagers = new IdentityHashMap<>();

    public static synchronized boolean hasExceededOwnershipCap(UUID uuid) {
        return hasExceededOwnershipCap(uuid,0);
    }

    public static synchronized boolean hasExceededOwnershipCap(UUID uuid, int added){
        if(uuid == null) uuid = class_156.field_25140;
        if(LMCConfig.limitSettings.enabledForEnvironment && uuid == class_156.field_25140){
            return getCountLoadedChunksOf(uuid)+added > LMCConfig.limitSettings.limit;
        }
        else if(LMCConfig.limitSettings.enabledForPlayers){
            return getCountLoadedChunksOf(uuid)+added > LMCConfig.limitSettings.limit;
        }
        return false;
    }

    public synchronized static void markChunkOwnedBy(class_3218 level, long longChunkPos, @Nullable UUID uuid){
        getManager(level).markChunkOwnedBy(longChunkPos,uuid);
        if(hasExceededOwnershipCap(uuid)){
            updateCDMSofUUID(uuid);
        }
    }
    public synchronized static void markChunkNotOwnedBy(class_3218 level, long longChunkPos, @Nullable UUID uuid){
        getManager(level).markChunkNotOwnedBy(longChunkPos,uuid);
        if(!hasExceededOwnershipCap(uuid)){
            updateCDMSofUUID(uuid);
        }
    }

    public static void updateCDMSofUUID(UUID uuid){
        for (class_3218 serverLevel : levelManagers.keySet()) {
            LevelChunkLoaderManager value = levelManagers.get(serverLevel);
            for (long l : value.forcedChunksByUUID.getOrDefault(uuid, new LongOpenHashSet())) {
                ChunkDataModule cdm = value.getOrCreateData(l);
                ILoadState loadState = cdm.getLoadState();
                cdm.update();
                cdm.updateChunkLoadState(serverLevel,loadState);
            }
        }
    }

    public static synchronized LevelChunkLoaderManager getManager(class_3218 level){
        return levelManagers.computeIfAbsent(level, k->new LevelChunkLoaderManager(level));
    }

    //? if >1.16.5 {
    public static synchronized LevelChunkLoaderManager loadManager(class_3218 level, class_2487 tag){
        LevelChunkLoaderManager manager = getManager(level);
        manager.load(tag);
        return manager;
    }
    //?}

    public static @NotNull Map<String,List<IChunkLoader>> getChunkLoadersOf(@Nullable UUID owner) {
        if(owner == null){
            return new HashMap<>();
        }
        Map<String,List<IChunkLoader>> results = new HashMap<>();
        for (LevelChunkLoaderManager value : levelManagers.values()) {
            List<IChunkLoader> loaders = results.computeIfAbsent(value.getLevelName(),k->new ArrayList<>());
            for (ChunkDataModule dataModule : value.getChunkDataModules()) {
                for (IChunkLoader loader : dataModule.getLoaders()) {
                    if(loader instanceof IOwnable && owner.equals(((IOwnable)loader).getOwner())){
                        loaders.add(loader);
                    }
                }
            }
        }
        return results;
    }
    public static int getCountChunkLoadersOf(@NotNull UUID owner) {
        if(owner == null){
            return 0;
        }
        int count = 0;
        for (LevelChunkLoaderManager value : levelManagers.values()) {
            for (ChunkDataModule dataModule : value.getChunkDataModules()) {
                for (IChunkLoader loader : dataModule.getLoaders()) {
                    if(loader instanceof IOwnable && owner.equals(((IOwnable)loader).getOwner())){
                        count++;
                    }
                }
            }
        }
        return count;
    }
    public synchronized static int getCountLoadedChunksOf(@NotNull UUID owner) {
        int count = 0;
        for (LevelChunkLoaderManager value : levelManagers.values()) {
            count += value.getCountLoadedChunksOf(owner);
        }
        return count;
    }

    public static void markShutDown(class_3218 level, class_1923 chunkPos, ILoadState previous) {
        getManager(level).shutDown(chunkPos,previous);
    }

    public static void removeChunkLoader(class_3218 level, class_2338 pos, IChunkLoader loader){
        removeChunkLoader(level,new class_1923(pos),loader);
    }

    public static void removeChunkLoader(class_3218 level, class_1923 pos, IChunkLoader loader){
        getManager(level).removeChunkLoader(loader,pos);
    }

    public static void removeChunkLoader(class_3218 level, long pos, IChunkLoader loader) {
        getManager(level).removeChunkLoader(loader, pos);
    }

    public static void addChunkLoader(class_3218 level, class_2338 pos, IChunkLoader loader){
        addChunkLoader(level,new class_1923(pos),loader);
    }

    public static void addChunkLoader(class_3218 level, class_1923 pos, IChunkLoader loader){
        getManager(level).addChunkLoader(loader,pos);
    }

    public static void addChunkLoader(class_3218 level, long pos, IChunkLoader loader) {
        getManager(level).addChunkLoader(loader, pos);
    }

    public static @NotNull ChunkDataModule getOrCreateChunkData(class_3218 level, class_2338 pos) {
        return getOrCreateChunkData(level,new class_1923(pos));
    }

    public static @NotNull ChunkDataModule getOrCreateChunkData(class_3218 level, class_1923 pos) {
        return getManager(level).getOrCreateData(pos);
    }
    public static @NotNull ChunkDataModule getOrCreateChunkData(class_3218 level, long pos) {
        return getManager(level).getOrCreateData(pos);
    }

    public static <T extends IChunkLoader> T computeChunkLoaderIfAbsent(class_3218 sl, class_2338 blockPos, Class<T> type, boolean doAdd, Predicate<T> predicate, Supplier<T> supplier) {
        return getManager(sl).computeChunkLoaderIfAbsent(blockPos,type,doAdd,predicate,supplier);
    }
    public static <T extends IChunkLoader> T computeChunkLoaderIfAbsent(class_3218 sl, class_1923 chunkPos, Class<T> type, boolean doAdd, Predicate<T> predicate, Supplier<T> supplier) {
        return getManager(sl).computeChunkLoaderIfAbsent(chunkPos,type,doAdd,predicate,supplier);
    }

    public static <T extends IChunkLoader> T computeChunkLoaderIfAbsent(class_3218 sl, class_2338 blockPos, Class<T> type, Predicate<T> predicate, Supplier<T> supplier) {
        return getManager(sl).computeChunkLoaderIfAbsent(blockPos,type,true,predicate,supplier);
    }
    public static <T extends IChunkLoader> T computeChunkLoaderIfAbsent(class_3218 sl, class_1923 chunkPos, Class<T> type, Predicate<T> predicate, Supplier<T> supplier) {
        return getManager(sl).computeChunkLoaderIfAbsent(chunkPos,type,true,predicate,supplier);
    }

    public static void clear() {
        for (LevelChunkLoaderManager value : levelManagers.values()) {
            value.clear();
        }
        levelManagers.clear();
    }

    public static void setDirty(class_3218 level){
        getManager(level).method_80();
    }

    public static boolean isForced(class_3218 level,class_1923 pos) {
        return getOrCreateChunkData(level,pos).getLoadState().shouldLoad();
    }

    public synchronized static void handleConfigReload() {
        for (class_3218 level : levelManagers.keySet()) {
            LevelChunkLoaderManager value = levelManagers.get(level);
            value.configReloaded=true;
        }
    }

    public synchronized static void requestUpdate(class_3218 level, class_1923 chunkPos) {
        getManager(level).requestUpdate(chunkPos);
    }

    public static class LevelChunkLoaderManager extends class_18{
        protected final Long2ObjectLinkedOpenHashMap<List<Consumer<ChunkDataModule>>> WAITING_FOR_INIT = new Long2ObjectLinkedOpenHashMap<>();
        private final Long2ObjectLinkedOpenHashMap<ChunkDataModule> data = new Long2ObjectLinkedOpenHashMap<>();
        private final Set<ChunkDataModule> shutoffLoaders = new HashSet<>();
        private final Map<UUID, LongOpenHashSet> forcedChunksByUUID = new HashMap<>();
        private final class_3218 level;
        protected boolean configReloaded = false;

        public synchronized void markChunkOwnedBy(long longChunkPos, @Nullable UUID uuid){
            if(uuid == null) uuid = class_156.field_25140;
            forcedChunksByUUID.computeIfAbsent(uuid, k -> new LongOpenHashSet()).add(longChunkPos);
        }
        public synchronized void markChunkNotOwnedBy(long longChunkPos, @Nullable UUID uuid){
            if(uuid == null) uuid = class_156.field_25140;
            if(forcedChunksByUUID.containsKey(uuid)){
                LongOpenHashSet set = forcedChunksByUUID.get(uuid);
                set.remove(longChunkPos);
                if(set.isEmpty()){
                    forcedChunksByUUID.remove(uuid);
                }
            }
        }

        public synchronized int getCountLoadedChunksOf(@NotNull UUID owner) {
            return forcedChunksByUUID.getOrDefault(owner,new LongOpenHashSet()).size();
        }

        public LevelChunkLoaderManager(@NotNull class_3218 level){
            //? if <=1.16.5
            /*super("loadmychunks_manager");*/
            this.level=level;
            level.method_8503().method_3742(this::tick);
        }

        public @NotNull ChunkDataModule getOrCreateData(@NotNull class_1923 pos){
            return getOrCreateData(pos.method_8324());
        }

        public void addChunkLoader(IChunkLoader loader, class_1923 pos){
            addChunkLoader(loader,pos.method_8324());
        }

        public synchronized void addChunkLoader(IChunkLoader loader, long pos){
            ChunkDataModule cdm = getOrCreateData(pos);
            if(loader instanceof IOwnable){
                markChunkOwnedBy(pos, ((IOwnable) loader).getOwner());
            }
            cdm.consumeLoadState(previous->{
                if(cdm.addLoader(level,loader)) {
                    cdm.updateChunkLoadState(level,previous);
                }
                method_80();
            });
        }

        public void removeChunkLoader(IChunkLoader loader, class_1923 pos){
            removeChunkLoader(loader,pos.method_8324());
        }

        public synchronized void removeChunkLoader(IChunkLoader loader, long pos){
            ChunkDataModule cdm = getOrCreateData(pos);
            cdm.consumeLoadState(previous-> {
                if(cdm.removeLoader(level,loader)) {
                    cdm.updateChunkLoadState(level,previous);
                }
                method_80();
            });
        }

        public synchronized @NotNull ChunkDataModule getOrCreateData(long pos){
            ChunkDataModule cdm = data.computeIfAbsent(pos, ChunkDataModule::new);
            method_80();
            return cdm;
        }

        public void load(class_2487 tag) {
            for (String key : tag.method_10541()) {
                long index = Long.parseLong(key);
                class_1923 pos = new class_1923(index);
                ChunkDataModule module = getOrCreateData(index);
                module.consumeLoadState(previous->{
                    module.load(tag.method_10562(key),level);
                    module.update();
                    if(module.onCooldown()){
                        shutDown(pos,previous);
                    }
                    else{
                        module.getLoadState().apply(level,pos,previous);
                    }
                });
            }
        }

        //? if <=1.20.5
        @Override
        public synchronized @NotNull class_2487 method_75(@NotNull class_2487 compoundTag) {
            data.forEach((k,v)->{
                if(v.shouldPersist()) {
                    compoundTag.method_10566(String.valueOf(k), v.save());
                }
            });
            return compoundTag;
        }

        private int tickCounter = 0;
        private static final int purgeTimer = 20*100;

        public synchronized void tick(){
            if(configReloaded){
                for (ChunkDataModule cdm : getChunkDataModules()) {
                    cdm.consumeLoadState(previous->{
                        cdm.update();
                        cdm.updateChunkLoadState(level,previous);
                    });
                }
                configReloaded=false;
            }
            if(tickCounter >= purgeTimer){
                data.values().removeIf(module -> !module.shouldPersist() && !level.method_8393(module.getPosition().field_9181, module.getPosition().field_9180));
                tickCounter = 0;
            }

            Iterator<ChunkDataModule> iterator = shutoffLoaders.iterator();
            while (iterator.hasNext()){
                ChunkDataModule cdm = iterator.next();
                if(!cdm.onCooldown()) {
                    iterator.remove();
                    cdm.consumeLoadState(previous->{
                        cdm.update();
                        if(cdm.getLoadState().shouldLoad()){
                            cdm.startGrace();
                        }
                        cdm.getLoadState().apply(level, cdm.getPosition(),previous);
                    });
                }

            }
            method_80();
            tickCounter++;
        }

        public synchronized void shutDown(@NotNull class_1923 chunkPos, @NotNull ILoadState previous) {
            ChunkDataModule module = data.get(chunkPos.method_8324());
            shutoffLoaders.add(module);
            module.getLoadState().apply(level,chunkPos,previous);
            method_80();
        }

        public Collection<ChunkDataModule> getChunkDataModules() {
            return data.values();
        }

        public String getLevelName() {
            return ((class_5268)level.method_8401()).method_150();
        }
        public synchronized  <T extends IChunkLoader> T computeChunkLoaderIfAbsent(class_2338 blockPos, Class<T> type, boolean doAdd, Predicate<T> predicate, Supplier<T> supplier) {
            return computeChunkLoaderIfAbsent(new class_1923(blockPos),type,doAdd,predicate,supplier);
        }
        @SuppressWarnings("all")
        public synchronized  <T extends IChunkLoader> T computeChunkLoaderIfAbsent(class_1923 pos, Class<T> type, boolean doAdd, Predicate<T> predicate, Supplier<T> supplier) {
            ChunkDataModule cdm = getOrCreateData(pos);
            for (IChunkLoader loader : cdm.getLoaders()) {
                if(loader.getClass() == type){
                    if(predicate.test((T) loader)){
                        return (T) loader;
                    }
                }
            }
            T out = supplier.get();
            if(doAdd) {
                addChunkLoader(out, pos);
            }
            return out;
        }

        public void requestUpdate(class_1923 chunkPos) {
            ChunkDataModule cdm = getOrCreateData(chunkPos);
            cdm.consumeLoadState(previous -> {
                cdm.update(()->{
                    cdm.updateChunkLoadState(level,previous);
                });
            });
        }

        public synchronized void clear() {
            data.clear();
        }

        //? if >1.20.5 {
        /*@Override
        public CompoundTag save(CompoundTag compoundTag, HolderLookup.Provider provider) {
            return save(compoundTag);
        }
        *///?}
    }
}
