package com.drathonix.loadmychunks.common.system.loaders;

import com.drathonix.loadmychunks.common.config.LMCConfig;
import com.drathonix.loadmychunks.common.registry.LoaderTypeKeys;
import com.drathonix.loadmychunks.common.registry.custom.LoadStateRegistry;
import com.drathonix.loadmychunks.common.system.control.ILoadState;
import com.drathonix.loadmychunks.common.system.control.LoadStateEnum;
import com.drathonix.loadmychunks.common.system.loaders.extension.ExtensionChunkLoaders;
import com.drathonix.loadmychunks.common.system.loaders.extension.IExtensionChunkLoader;
import com.drathonix.loadmychunks.common.system.loaders.extension.PlacedExtensionChunkLoader;
import net.minecraft.core.BlockPos;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.world.level.ChunkPos;
import net.minecraft.world.level.chunk.ChunkAccess;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Objects;
import java.util.UUID;

public class PlacedChunkLoader implements IChunkLoader,IOwnable {
    @Nullable protected ExtensionChunkLoaders extensions = null;
    protected int extensionRange = 0;
    @Nullable protected UUID owner;
    protected BlockPos position;
    protected ILoadState defaultState = LMCConfig.placedChunkLoaderDefaultLevel.get();
    protected ILoadState loadState = defaultState;
    protected long activityEnd = -1;

    public PlacedChunkLoader(){
    }

    public PlacedChunkLoader(BlockPos pos){
        this.position = pos;
    }
    public PlacedChunkLoader(BlockPos pos, long activityEnd){
        this.position = pos;
        this.activityEnd=activityEnd;
    }
    public PlacedChunkLoader(BlockPos pos, @Nullable UUID owner){
        this.position = pos;
        this.owner = owner;
    }

    @Override
    public @NotNull CompoundTag save(CompoundTag tag) {
        if(hasOwner()) {
            tag.putUUID("owner", owner);
        }
        if(hasExtensions()){
            tag.putInt("extensions",extensionRange);
        }
        tag.putLong("duration", activityEnd);
        loadState.putCompound("state",tag);
        defaultState.putCompound("default",tag);
        tag.putLong("pos",position.asLong());
        return tag;
    }

    @Override
    public void load(@NotNull CompoundTag tag, ServerLevel level) throws DoNotAddException {
        position = BlockPos.of(tag.getLong("pos"));
        if(tag.contains("owner")){
            owner = tag.getUUID("owner");
        }
        if(tag.contains("duration")){
            activityEnd = tag.getLong("duration");
        }
        defaultState = LoadStateRegistry.fromCompound("default",tag,LoadStateRegistry.TICKING);
        loadState = LoadStateRegistry.fromCompound("state",tag,defaultState);
        if(tag.contains("extensions")){
            extend(level,tag.getInt("extensions"));
        }
    }

    @Override
    public boolean postLoad(ChunkAccess chunk) throws DoNotAddException {
        //TODO: fix an infinite loop caused by this line.
        //if(!(chunk.getBlockEntity(position) instanceof IHasChunkloader)){
        //throw new DoNotAddException();
        //}
        return false;
    }

    @Override
    public void setExtensionRange(int extensionRange) {
        this.extensionRange = extensionRange;
    }

    @Override
    public void setExtensionsMap(ExtensionChunkLoaders extensions) {
        this.extensions=extensions;
    }

    public int getExtensionRange(){
        return extensionRange;
    }

    public PlacedExtensionChunkLoader createExtension(ChunkPos position) {
        return new PlacedExtensionChunkLoader(position,this);
    }

    @Override
    @SuppressWarnings("unchecked")
    public <T extends IExtensionChunkLoader<?>> Class<T> getExtensionClass() {
        return (Class<T>) PlacedExtensionChunkLoader.class;
    }

    @Override
    public boolean supportsExtensions() {
        return true;
    }

    @Override
    public boolean supportsEntityTicking() {
        return true;
    }

    @Override
    public @Nullable ExtensionChunkLoaders getExtensionChunkLoaders() {
        return extensions;
    }

    @Override
    public ExtensionChunkLoaders.Factory<?> getExtensionFactory() {
        return this::createExtension;
    }

    @Override
    public @Nullable UUID getOwner() {
        return owner;
    }

    protected boolean outOfTime(){
        return shouldConsumeItems() && activityEnd == -1;
    }

    @Override
    public ILoadState getActiveState() {
        if(hasExceededChunkLimit() || outOfTime()){
            return LoadStateEnum.DISABLED;
        }
        return loadState;
    }

    @Override
    public ILoadState getDefaultState() {
        return defaultState;
    }

    @Override
    public void setActiveState(ILoadState state) {
        this.loadState =state;
    }

    @Override
    public void setOwner(@NotNull UUID owner) {
        this.owner=owner;
    }

    public @NotNull BlockPos getPosition() {
        return position;
    }

    @Override
    public ResourceLocation getTypeId() {
        return LoaderTypeKeys.PLACED_LOADER;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        PlacedChunkLoader that = (PlacedChunkLoader) o;
        return Objects.equals(position, that.position);
    }

    @Override
    public int hashCode() {
        return Objects.hash(position);
    }

    @Override
    public @NotNull ChunkPos getChunkPos() {
        return new ChunkPos(getPosition());
    }

    @Override
    public long getActivityEnd() {
        return activityEnd;
    }

    @Override
    public void setActivityEnd(long activityEnd) {
        this.activityEnd = activityEnd;
    }

    @Override
    public @NotNull BlockPos getItemSource() {
        return position.above();
    }

    @Override
    public ILoadState setDefaultState(ILoadState defaultState) {
        ILoadState prev = this.defaultState;
        this.defaultState = defaultState;
        // Will be this unless disabled by CCT
        if(this.loadState == LoadStateRegistry.TICKING){
            this.loadState=defaultState;
        }
        return prev;
    }
}
