package com.drathonix.loadmychunks.common.util;

import com.mojang.brigadier.Command;
import com.mojang.brigadier.arguments.ArgumentType;
import com.mojang.brigadier.arguments.StringArgumentType;
import com.mojang.brigadier.builder.ArgumentBuilder;
import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import com.mojang.brigadier.builder.RequiredArgumentBuilder;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.Predicate;
import net.minecraft.class_1923;
import net.minecraft.class_2168;
import net.minecraft.class_2170;
import net.minecraft.class_2262;
import net.minecraft.class_2338;
import net.minecraft.class_243;
import net.minecraft.class_3218;

/**
 * Multi-versioning Abstraction Utility class for the Brigadier API.
 * Subcommand structure is maintained by a list builder.
 */
public class Brigadier {
    /**
     * Creates a literal node.
     * @param name the literal name
     * @param inner child nodes.
     * @return a completed command.
     */
    public static @NotNull LiteralArgumentBuilder<class_2168> literal(@NotNull String name, @NotNull InternalBuilder<class_2168> inner){
        LiteralArgumentBuilder<class_2168> out = class_2170.method_9247(name);
        for (ArgumentBuilder<class_2168, ?> builder : inner.make()) {
            out.then(builder);
        }
        return out;
    }

    /**
     * Creates an argument node.
     * @param name the arg name
     * @param type the arg type
     * @param inner child nodes.
     * @return a completed argument.
     * @param <T> the argument builder type
     */
    public static <T extends ArgumentBuilder<class_2168,T>> @NotNull RequiredArgumentBuilder<class_2168,?> argument(@NotNull String name, @NotNull ArgumentType<?> type, @NotNull InternalBuilder<class_2168> inner){
        RequiredArgumentBuilder<class_2168,?> out = class_2170.method_9244(name,type);
        for (ArgumentBuilder<class_2168, ?> builder : inner.make()) {
            out.then(builder);
        }
        return out;
    }

    /**
     * Sets the required permission level to 2.
     * @param argumentBuilder the node to apply to.
     * @return the modified node.
     * @param <T> the argument builder type
     */
    public static <T extends ArgumentBuilder<class_2168,T>> @NotNull T admin(@NotNull ArgumentBuilder<class_2168,T> argumentBuilder){
        return argumentBuilder.requires(ctx->ctx.method_9259(2));
    }

    /**
     * Adds a requirement condition for the command to be executable.
     * @param argumentBuilder the node to apply to.
     * @param predicate an arbitrary test condition.
     * @return the modified node.
     * @param <T> the argument builder type
     */
    public static <T extends ArgumentBuilder<class_2168,T>> @NotNull ArgumentBuilder<class_2168,?> requires(@NotNull ArgumentBuilder<class_2168,T> argumentBuilder, @NotNull Predicate<class_2168> predicate){
        return argumentBuilder.requires(predicate);
    }

    /**
     * Creates a BlockPos argument node.
     * @param name the arg name.
     * @param inner child nodes.
     * @return the modified node
     * @param <T> the argument builder type
     */
    public static <T extends ArgumentBuilder<class_2168,T>> @NotNull RequiredArgumentBuilder<class_2168,?> blockPos(@NotNull String name, @NotNull InternalBuilder<class_2168> inner){
        return argument(name, class_2262.method_9698(),inner);
    }

    /**
     * Creates a String argument node.
     * @param name the arg name.
     * @param inner child nodes.
     * @return the modified node
     * @param <T> the argument builder type
     */
    public static <T extends ArgumentBuilder<class_2168,T>> @NotNull RequiredArgumentBuilder<class_2168,?> string(@NotNull String name, @NotNull InternalBuilder<class_2168> inner){
        return argument(name, StringArgumentType.string(),inner);
    }

    /**
     * Creates a Greedy String argument node.
     * @param name the arg name.
     * @param inner child nodes.
     * @return the modified node
     * @param <T> the argument builder type
     */
    public static <T extends ArgumentBuilder<class_2168,T>> @NotNull RequiredArgumentBuilder<class_2168,?> stringRemaining(@NotNull String name, @NotNull InternalBuilder<class_2168> inner){
        return argument(name, StringArgumentType.greedyString(),inner);
    }

    /**
     * Creates a bool argument node.
     * @param name the arg name.
     * @param inner child nodes.
     * @return the modified node
     * @param <T> the argument builder type
     */
    public static <T extends ArgumentBuilder<class_2168,T>> @NotNull RequiredArgumentBuilder<class_2168,?> bool(@NotNull String name, @NotNull InternalBuilder<class_2168> inner){
        return argument(name, BoolArgument.boolArgument(),inner);
    }

    /**
     *
     * @param argumentBuilder the source node.
     * @param executor an arbitrary executable.
     * @return the modified node.
     * @param <S> the command source type.
     * @param <T> the argument builder type.
     */
    public static <S,T extends ArgumentBuilder<S,T>> @NotNull T executes(@NotNull ArgumentBuilder<S,T> argumentBuilder, @NotNull Command<S> executor){
        return argumentBuilder.executes(executor);
    }

    /**
     * Gets the ServerLevel from a CommandContext of type CommandSourceStack.
     * @param ctx the context.
     * @return the ServerLevel of execution.
     */
    public static @NotNull class_3218 getLevel(@NotNull CommandContext<class_2168> ctx) {
        return ctx.getSource().method_9225();
    }

    /**
     * Gets a block pos from an argument.
     * @param ctx the command context.
     * @param arg the argument name.
     * @return a blockPos
     * @throws CommandSyntaxException when the argument input is invalid.
     */
    public static @NotNull class_2338 getBlockPos(@NotNull CommandContext<class_2168> ctx, @NotNull String arg) throws CommandSyntaxException {
        //? if <1.18.2 {
        return class_2262.method_9697(ctx,arg);
        //?} else if <1.20.1 {
        /*return BlockPosArgument.getSpawnablePos(ctx,arg);
        *///?} else {
        /*return BlockPosArgument.getBlockPos(ctx, arg);
        *///?}
    }

    /**
     * Uses the context's blockpos if the provided pos is null.
     * @param ctx the command context.
     * @param bp a blockpos or null.
     * @return a blockPos
     */
    public static @NotNull class_2338 defaultedPos(@NotNull CommandContext<class_2168> ctx, @Nullable class_2338 bp) {
        if(bp != null){
            return bp;
        }
        else{
            class_243 vec = ctx.getSource().method_9222();
            return new class_2338((int)vec.field_1352,(int)vec.field_1351,(int)vec.field_1350);
        }
    }

    /**
     * Creates a blockpos for the center of a chunk at a specific y.
     * @param pos the chunkpos to center on.
     * @param y the y level.
     * @return a BlockPos at the center of the chunk at the y.
     */
    public static @NotNull class_2338 centralized(@NotNull class_1923 pos, int y) {
        return new class_2338(pos.field_9181*16+8,y,pos.field_9180*16+8);
    }

    /**
     * Allows adding children to a command node using a pre-made list.
     * @param <S> the command source type.
     */
    @FunctionalInterface
    public interface InternalBuilder<S> extends Consumer<@NotNull List<ArgumentBuilder<S,?>>> {
        /**
         * Creates an argument builder list using the lambda expression.
         * @return an arbitrary list of ArgumentBuilders
         */
        default @NotNull List<ArgumentBuilder<S,?>> make(){
            List<ArgumentBuilder<S,?>> out = new ArrayList<>();
            accept(out);
            return out;
        }
    }
}
