package thelm.jaopca.api.fluids;

import java.util.EnumMap;
import java.util.IdentityHashMap;
import java.util.Map;

import org.apache.commons.lang3.tuple.Pair;

import it.unimi.dsi.fastutil.objects.Object2ByteLinkedOpenHashMap;
import it.unimi.dsi.fastutil.shorts.Short2BooleanMap;
import it.unimi.dsi.fastutil.shorts.Short2BooleanOpenHashMap;
import it.unimi.dsi.fastutil.shorts.Short2ObjectMap;
import it.unimi.dsi.fastutil.shorts.Short2ObjectOpenHashMap;
import net.minecraft.block.Block;
import net.minecraft.block.BlockState;
import net.minecraft.block.Blocks;
import net.minecraft.block.DoorBlock;
import net.minecraft.block.ILiquidContainer;
import net.minecraft.block.material.Material;
import net.minecraft.fluid.Fluid;
import net.minecraft.fluid.FluidState;
import net.minecraft.fluid.Fluids;
import net.minecraft.state.IntegerProperty;
import net.minecraft.state.StateContainer;
import net.minecraft.tags.BlockTags;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.Direction;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.shapes.VoxelShape;
import net.minecraft.util.math.shapes.VoxelShapes;
import net.minecraft.util.math.vector.Vector3d;
import net.minecraft.world.IBlockReader;
import net.minecraft.world.IWorld;
import net.minecraft.world.IWorldReader;
import net.minecraft.world.World;

public abstract class PlaceableFluid extends Fluid {

	public static final float EIGHT_NINTHS = 8/9F;
	private static final ThreadLocal<Object2ByteLinkedOpenHashMap<Block.RenderSideCacheKey>> CACHE = ThreadLocal.withInitial(()->{
		Object2ByteLinkedOpenHashMap<Block.RenderSideCacheKey> object2bytelinkedopenhashmap = new Object2ByteLinkedOpenHashMap<Block.RenderSideCacheKey>(200) {
			@Override
			protected void rehash(int newN) {}
		};
		object2bytelinkedopenhashmap.defaultReturnValue((byte)127);
		return object2bytelinkedopenhashmap;
	});

	protected final StateContainer<Fluid, FluidState> stateContainer;
	private final Map<FluidState, VoxelShape> shapeMap = new IdentityHashMap<>();

	protected final int maxLevel;
	protected final IntegerProperty levelProperty;

	public PlaceableFluid(int maxLevel) {
		this.maxLevel = maxLevel;
		levelProperty = IntegerProperty.func_177719_a("level", 1, maxLevel+1);

		StateContainer.Builder<Fluid, FluidState> builder = new StateContainer.Builder<>(this);
		func_207184_a(builder);
		stateContainer = builder.func_235882_a_(Fluid::func_207188_f, FluidState::new);
		func_207183_f(stateContainer.func_177621_b().func_206870_a(levelProperty, maxLevel));
	}

	public IntegerProperty getLevelProperty() {
		return levelProperty;
	}

	@Override
	protected void func_207184_a(StateContainer.Builder<Fluid, FluidState> builder) {
		if(levelProperty != null) {
			builder.func_206894_a(levelProperty);
		}
	}

	@Override
	public StateContainer<Fluid, FluidState> func_207182_e() {
		return stateContainer;
	}

	protected abstract boolean canSourcesMultiply();

	@Override
	protected boolean func_215665_a(FluidState fluidState, IBlockReader world, BlockPos pos, Fluid fluid, Direction face) {
		return face == Direction.DOWN && !func_207187_a(fluid);
	}

	protected abstract int getLevelDecreasePerBlock(IWorldReader world);

	@Override
	protected BlockState func_204527_a(FluidState fluidState) {
		PlaceableFluidBlock block = getFluidBlock();
		IntegerProperty blockLevelProperty = block.getLevelProperty();
		int fluidLevel = fluidState.func_177229_b(levelProperty);
		int blockLevel = fluidLevel > maxLevel ? maxLevel : maxLevel-fluidLevel;
		return block.func_176223_P().func_206870_a(blockLevelProperty, blockLevel);
	}

	protected abstract PlaceableFluidBlock getFluidBlock();

	/**
	 * getFlow
	 */
	@Override
	protected Vector3d func_215663_a(IBlockReader world, BlockPos pos, FluidState state) {
		double x = 0;
		double y = 0;
		BlockPos.Mutable mutablePos = new BlockPos.Mutable();
		for(Direction offset : Direction.Plane.HORIZONTAL) {
			mutablePos.func_239622_a_(pos, offset);
			FluidState offsetState = world.func_204610_c(mutablePos);
			if(isSameOrEmpty(offsetState)) {
				float offsetHeight = offsetState.func_223408_f();
				float heightDiff = 0;
				if(offsetHeight == 0) {
					if(!world.func_180495_p(mutablePos).func_185904_a().func_76230_c()) {
						BlockPos posDown = mutablePos.func_177977_b();
						FluidState belowState = world.func_204610_c(posDown);
						if(isSameOrEmpty(belowState)) {
							offsetHeight = belowState.func_223408_f();
							if(offsetHeight > 0) {
								heightDiff = state.func_223408_f()-(offsetHeight-EIGHT_NINTHS);
							}
						}
					}
				}
				else if(offsetHeight > 0) {
					heightDiff = state.func_223408_f() - offsetHeight;
				}
				if(heightDiff != 0) {
					x += offset.func_82601_c()*heightDiff;
					y += offset.func_82599_e()*heightDiff;
				}
			}
		}
		Vector3d flow = new Vector3d(x, 0, y);
		if(state.func_177229_b(levelProperty).intValue() == 0) {
			for(Direction offset : Direction.Plane.HORIZONTAL) {
				mutablePos.func_239622_a_(pos, offset);
				if(causesDownwardCurrent(world, mutablePos, offset) || causesDownwardCurrent(world, mutablePos.func_177984_a(), offset)) {
					flow = flow.func_72432_b().func_72441_c(0, -6, 0);
					break;
				}
			}
		}
		return flow.func_72432_b();
	}

	private boolean isSameOrEmpty(FluidState otherState) {
		return otherState.func_206888_e() || otherState.func_206886_c().func_207187_a(this);
	}

	protected boolean causesDownwardCurrent(IBlockReader world, BlockPos pos, Direction face) {
		BlockState blockState = world.func_180495_p(pos);
		FluidState fluidState = world.func_204610_c(pos);
		return !fluidState.func_206886_c().func_207187_a(this) && (face == Direction.UP ||
				(blockState.func_185904_a() != Material.field_151588_w && blockState.func_224755_d(world, pos, face)));
	}

	protected void flowAround(IWorld world, BlockPos pos, FluidState fluidState) {
		if(!fluidState.func_206888_e()) {
			BlockState blockState = world.func_180495_p(pos);
			BlockPos downPos = pos.func_177977_b();
			BlockState downBlockState = world.func_180495_p(downPos);
			FluidState newFluidState = calculateCorrectState(world, downPos, downBlockState);
			if(canFlow(world, pos, blockState, Direction.DOWN, downPos, downBlockState, world.func_204610_c(downPos), newFluidState.func_206886_c())) {
				flowInto(world, downPos, downBlockState, Direction.DOWN, newFluidState);
				if(getAdjacentSourceCount(world, pos) >= 3) {
					flowAdjacent(world, pos, fluidState, blockState);
				}
			}
			else if(fluidState.func_206889_d() || !canFlowDown(world, newFluidState.func_206886_c(), pos, blockState, downPos, downBlockState)) {
				flowAdjacent(world, pos, fluidState, blockState);
			}
		}
	}

	protected void flowAdjacent(IWorld world, BlockPos pos, FluidState fluidState, BlockState blockState) {
		int i = fluidState.func_206882_g() - getLevelDecreasePerBlock(world);
		if(i > 0) {
			Map<Direction, FluidState> map = calculateAdjacentStates(world, pos, blockState);
			for(Map.Entry<Direction, FluidState> entry : map.entrySet()) {
				Direction direction = entry.getKey();
				FluidState offsetFluidState = entry.getValue();
				BlockPos offsetPos = pos.func_177972_a(direction);
				BlockState offsetBlockState = world.func_180495_p(offsetPos);
				if(canFlow(world, pos, blockState, direction, offsetPos, offsetBlockState, world.func_204610_c(offsetPos), offsetFluidState.func_206886_c())) {
					flowInto(world, offsetPos, offsetBlockState, direction, offsetFluidState);
				}
			}
		}
	}

	protected FluidState calculateCorrectState(IWorldReader world, BlockPos pos, BlockState blockState) {
		int i = 0;
		int j = 0;
		for(Direction direction : Direction.Plane.HORIZONTAL) {
			BlockPos offsetPos = pos.func_177972_a(direction);
			BlockState offsetBlockState = world.func_180495_p(offsetPos);
			FluidState offsetFluidState = offsetBlockState.func_204520_s();
			if(offsetFluidState.func_206886_c().func_207187_a(this) && doShapesFillSquare(direction, world, pos, blockState, offsetPos, offsetBlockState)) {
				if(offsetFluidState.func_206889_d()) {
					++j;
				}
				i = Math.max(i, offsetFluidState.func_206882_g());
			}
		}
		if(canSourcesMultiply() && j >= 2) {
			BlockState blockstate1 = world.func_180495_p(pos.func_177977_b());
			FluidState FluidState1 = blockstate1.func_204520_s();
			if(blockstate1.func_185904_a().func_76220_a() || isSameSource(FluidState1)) {
				return func_207188_f().func_206870_a(levelProperty, maxLevel);
			}
		}
		BlockPos upPos = pos.func_177984_a();
		BlockState upBlockState = world.func_180495_p(upPos);
		FluidState upFluidState = upBlockState.func_204520_s();
		if(!upFluidState.func_206888_e() && upFluidState.func_206886_c().func_207187_a(this) && doShapesFillSquare(Direction.UP, world, pos, blockState, upPos, upBlockState)) {
			return func_207188_f().func_206870_a(levelProperty, maxLevel+1);
		}
		else {
			int k = i - getLevelDecreasePerBlock(world);
			if(k <= 0) {
				return Fluids.field_204541_a.func_207188_f();
			}
			else {
				return func_207188_f().func_206870_a(levelProperty, k);
			}
		}
	}

	protected boolean doShapesFillSquare(Direction direction, IBlockReader world, BlockPos fromPos, BlockState fromBlockState, BlockPos toPos, BlockState toBlockState) {
		Object2ByteLinkedOpenHashMap<Block.RenderSideCacheKey> cache;
		if(!fromBlockState.func_177230_c().func_208619_r() && !toBlockState.func_177230_c().func_208619_r()) {
			cache = CACHE.get();
		}
		else {
			cache = null;
		}
		Block.RenderSideCacheKey cacheKey;
		if(cache != null) {
			cacheKey = new Block.RenderSideCacheKey(fromBlockState, toBlockState, direction);
			byte b0 = cache.getAndMoveToFirst(cacheKey);
			if(b0 != 127) {
				return b0 != 0;
			}
		}
		else {
			cacheKey = null;
		}
		VoxelShape fromShape = fromBlockState.func_196952_d(world, fromPos);
		VoxelShape toShape = toBlockState.func_196952_d(world, toPos);
		boolean flag = !VoxelShapes.func_204642_b(fromShape, toShape, direction);
		if(cache != null) {
			if(cache.size() == 200) {
				cache.removeLastByte();
			}
			cache.putAndMoveToFirst(cacheKey, (byte)(flag ? 1 : 0));
		}
		return flag;
	}

	protected void flowInto(IWorld world, BlockPos pos, BlockState blockState, Direction direction, FluidState fluidState) {
		if(blockState.func_177230_c() instanceof ILiquidContainer) {
			((ILiquidContainer)blockState.func_177230_c()).func_204509_a(world, pos, blockState, fluidState);
		}
		else {
			if(!blockState.isAir(world, pos)) {
				beforeReplacingBlock(world, pos, blockState);
			}
			world.func_180501_a(pos, fluidState.func_206883_i(), 3);
		}
	}

	protected void beforeReplacingBlock(IWorld world, BlockPos pos, BlockState blockState) {
		TileEntity tile = blockState.hasTileEntity() ? world.func_175625_s(pos) : null;
		Block.func_220059_a(blockState, world, pos, tile);
	}

	protected static short getPosKey(BlockPos pos, BlockPos otherPos) {
		int dx = otherPos.func_177958_n() - pos.func_177958_n();
		int dz = otherPos.func_177952_p() - pos.func_177952_p();
		return (short)((dx + 128 & 255) << 8 | dz + 128 & 255);
	}

	protected Map<Direction, FluidState> calculateAdjacentStates(IWorldReader world, BlockPos pos, BlockState blockState) {
		int i = 1000;
		Map<Direction, FluidState> map = new EnumMap<>(Direction.class);
		Short2ObjectMap<Pair<BlockState, FluidState>> stateMap = new Short2ObjectOpenHashMap<>();
		Short2BooleanMap canFlowDownMap = new Short2BooleanOpenHashMap();
		for(Direction direction : Direction.Plane.HORIZONTAL) {
			BlockPos offsetPos = pos.func_177972_a(direction);
			short key = getPosKey(pos, offsetPos);
			Pair<BlockState, FluidState> offsetState = stateMap.computeIfAbsent(key, k->{
				BlockState offsetBlockState = world.func_180495_p(offsetPos);
				return Pair.of(offsetBlockState, offsetBlockState.func_204520_s());
			});
			BlockState offsetBlockState = offsetState.getLeft();
			FluidState offsetFluidState = offsetState.getRight();
			FluidState newOffsetFluidState = calculateCorrectState(world, offsetPos, offsetBlockState);
			if(canFlowSource(world, newOffsetFluidState.func_206886_c(), pos, blockState, direction, offsetPos, offsetBlockState, offsetFluidState)) {
				boolean flag = canFlowDownMap.computeIfAbsent(key, k->{
					BlockPos offsetDownPos = offsetPos.func_177977_b();
					BlockState offsetDownState = world.func_180495_p(offsetDownPos);
					return canFlowDown(world, this, offsetPos, offsetBlockState, offsetDownPos, offsetDownState);
				});
				int j = 0;
				if(!flag) {
					j = getFlowDistance(world, offsetPos, 1, direction.func_176734_d(), offsetBlockState, pos, stateMap, canFlowDownMap);
				}
				if(j < i) {
					map.clear();
				}
				if(j <= i) {
					map.put(direction, newOffsetFluidState);
					i = j;
				}
			}
		}
		return map;
	}

	protected int getFlowDistance(IWorldReader world, BlockPos pos, int distance, Direction fromDirection, BlockState blockState, BlockPos startPos, Short2ObjectMap<Pair<BlockState, FluidState>> stateMap, Short2BooleanMap canFlowDownMap) {
		int i = 1000;
		for(Direction direction : Direction.Plane.HORIZONTAL) {
			if(direction != fromDirection) {
				BlockPos offsetPos = pos.func_177972_a(direction);
				short key = getPosKey(startPos, offsetPos);
				Pair<BlockState, FluidState> pair = stateMap.computeIfAbsent(key, k->{
					BlockState offsetBlockState = world.func_180495_p(offsetPos);
					return Pair.of(offsetBlockState, offsetBlockState.func_204520_s());
				});
				BlockState offsetBlockState = pair.getLeft();
				FluidState offsetFluidstate = pair.getRight();
				if(canFlowSource(world, this, pos, blockState, direction, offsetPos, offsetBlockState, offsetFluidstate)) {
					boolean flag = canFlowDownMap.computeIfAbsent(key, k->{
						BlockPos offsetDownPos = offsetPos.func_177977_b();
						BlockState offsetDownState = world.func_180495_p(offsetDownPos);
						return canFlowDown(world, this, offsetPos, offsetBlockState, offsetDownPos, offsetDownState);
					});
					if(flag) {
						return distance;
					}
					if(distance < getSlopeFindDistance(world)) {
						int j = getFlowDistance(world, offsetPos, distance+1, direction.func_176734_d(), offsetBlockState, startPos, stateMap, canFlowDownMap);
						if(j < i) {
							i = j;
						}
					}
				}
			}
		}
		return i;
	}

	protected boolean isSameSource(FluidState fluidState) {
		return fluidState.func_206886_c().func_207187_a(this) && fluidState.func_206889_d();
	}

	protected int getSlopeFindDistance(IWorldReader world) {
		return ceilDiv(ceilDiv(maxLevel, getLevelDecreasePerBlock(world)), 2);
	}

	protected int getAdjacentSourceCount(IWorldReader world, BlockPos pos) {
		int count = 0;
		for(Direction offset : Direction.Plane.HORIZONTAL) {
			BlockPos offsetPos = pos.func_177972_a(offset);
			FluidState offsetState = world.func_204610_c(offsetPos);
			if(isSameSource(offsetState)) {
				++count;
			}
		}
		return count;
	}

	protected boolean canFlowIntoBlock(IBlockReader world, BlockPos pos, BlockState blockState, Fluid fluid) {
		Block block = blockState.func_177230_c();
		if(block instanceof ILiquidContainer) {
			return ((ILiquidContainer)block).func_204510_a(world, pos, blockState, fluid);
		}
		if(block instanceof DoorBlock || block.func_203417_a(BlockTags.field_219753_V) || block == Blocks.field_150468_ap || block == Blocks.field_196608_cF ||
				block == Blocks.field_203203_C) {
			return false;
		}
		Material blockMaterial = blockState.func_185904_a();
		return blockMaterial != Material.field_151567_E && blockMaterial != Material.field_189963_J && blockMaterial != Material.field_203243_f
				&& blockMaterial != Material.field_204868_h && !blockMaterial.func_76230_c();
	}

	protected boolean canFlow(IBlockReader world, BlockPos fromPos, BlockState fromBlockState, Direction direction, BlockPos toPos, BlockState toBlockState, FluidState toFluidState, Fluid fluid) {
		return toFluidState.func_215677_a(world, toPos, fluid, direction) && this.doShapesFillSquare(direction, world, fromPos, fromBlockState, toPos, toBlockState) && canFlowIntoBlock(world, toPos, toBlockState, fluid);
	}

	protected boolean canFlowSource(IBlockReader world, Fluid fluid, BlockPos fromPos, BlockState fromBlockState, Direction direction, BlockPos toPos, BlockState toBlockState, FluidState toFluidState) {
		return !isSameSource(toFluidState) && doShapesFillSquare(direction, world, fromPos, fromBlockState, toPos, toBlockState) && canFlowIntoBlock(world, toPos, toBlockState, fluid);
	}

	protected boolean canFlowDown(IBlockReader world, Fluid fluid, BlockPos fromPos, BlockState fromBlockState, BlockPos downPos, BlockState downState) {
		return doShapesFillSquare(Direction.DOWN, world, fromPos, fromBlockState, downPos, downState)
				&& (downState.func_204520_s().func_206886_c().func_207187_a(this) || canFlowIntoBlock(world, downPos, downState, fluid));
	}

	protected int getDelay(World world, BlockPos pos, FluidState fluidState, FluidState newFluidState) {
		return func_205569_a(world);
	}

	@Override
	public void func_207191_a(World world, BlockPos pos, FluidState fluidState) {
		if(!fluidState.func_206889_d()) {
			FluidState newFluidState = calculateCorrectState(world, pos, world.func_180495_p(pos));
			int delay = getDelay(world, pos, fluidState, newFluidState);
			if(newFluidState.func_206888_e()) {
				fluidState = newFluidState;
				world.func_180501_a(pos, Blocks.field_150350_a.func_176223_P(), 3);
			}
			else if(!newFluidState.equals(fluidState)) {
				fluidState = newFluidState;
				BlockState blockState = fluidState.func_206883_i();
				world.func_180501_a(pos, blockState, 2);
				world.func_205219_F_().func_205360_a(pos, fluidState.func_206886_c(), delay);
				world.func_195593_d(pos, blockState.func_177230_c());
			}
		}
		flowAround(world, pos, fluidState);
	}

	protected int getBlockLevelFromState(FluidState fluidState) {
		int level = fluidState.func_177229_b(levelProperty);
		if(level > maxLevel) {
			return maxLevel;
		}
		return maxLevel - Math.min(fluidState.func_206882_g(), maxLevel);
	}

	protected static boolean isFluidAboveSame(FluidState fluidState, IBlockReader world, BlockPos pos) {
		return fluidState.func_206886_c().func_207187_a(world.func_204610_c(pos.func_177984_a()).func_206886_c());
	}

	@Override
	public float func_215662_a(FluidState fluidState, IBlockReader world, BlockPos pos) {
		return isFluidAboveSame(fluidState, world, pos) ? 1 : fluidState.func_223408_f();
	}

	@Override
	public float func_223407_a(FluidState fluidState) {
		return 0.9F*fluidState.func_206882_g()/maxLevel;
	}

	@Override
	public boolean func_207193_c(FluidState fluidState) {
		return fluidState.func_177229_b(levelProperty).intValue() == maxLevel;
	}

	@Override
	public int func_207192_d(FluidState fluidState) {
		return Math.min(maxLevel, fluidState.func_177229_b(levelProperty));
	}

	@Override
	public VoxelShape func_215664_b(FluidState fluidState, IBlockReader world, BlockPos pos) {
		return shapeMap.computeIfAbsent(fluidState, s->VoxelShapes.func_197873_a(0, 0, 0, 1, s.func_215679_a(world, pos), 1));
	}

	public static int ceilDiv(int x, int y) {
		int r = x/y;
		if((x^y) >= 0 && (r*y != x)) {
			r++;
		}
		return r;
	}
}
