package thelm.jaopca.blocks;

import java.util.function.BooleanSupplier;
import java.util.function.DoubleSupplier;
import java.util.function.IntSupplier;
import java.util.function.Supplier;

import net.minecraft.block.Block;
import net.minecraft.block.BlockState;
import net.minecraft.block.SoundType;
import net.minecraft.entity.Entity;
import net.minecraft.util.Direction;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.shapes.ISelectionContext;
import net.minecraft.util.math.shapes.VoxelShape;
import net.minecraft.util.math.shapes.VoxelShapes;
import net.minecraft.util.text.IFormattableTextComponent;
import net.minecraft.world.IBlockReader;
import net.minecraft.world.IWorldReader;
import net.minecraftforge.common.ToolType;
import thelm.jaopca.api.blocks.IBlockFormSettings;
import thelm.jaopca.api.blocks.IMaterialFormBlock;
import thelm.jaopca.api.forms.IForm;
import thelm.jaopca.api.functions.MemoizingSuppliers;
import thelm.jaopca.api.materials.IMaterial;
import thelm.jaopca.utils.ApiImpl;

public class JAOPCABlock extends Block implements IMaterialFormBlock {

	private final IForm form;
	private final IMaterial material;
	protected final IBlockFormSettings settings;

	protected boolean blocksMovement;
	protected Supplier<SoundType> soundType;
	protected IntSupplier lightOpacity;
	protected IntSupplier lightValue;
	protected DoubleSupplier explosionResistance;
	protected DoubleSupplier slipperiness;
	protected VoxelShape shape;
	protected VoxelShape raytraceShape;
	protected Supplier<ToolType> harvestTool;
	protected IntSupplier harvestLevel;
	protected IntSupplier flammability;
	protected IntSupplier fireSpreadSpeed;
	protected BooleanSupplier isFireSource;

	public JAOPCABlock(IForm form, IMaterial material, IBlockFormSettings settings) {
		super(getProperties(form, material, settings));
		this.form = form;
		this.material = material;
		this.settings = settings;

		blocksMovement = settings.getBlocksMovement();
		soundType = MemoizingSuppliers.of(settings.getSoundTypeFunction(), ()->material);
		lightOpacity = MemoizingSuppliers.of(settings.getLightOpacityFunction(), ()->material);
		lightValue = MemoizingSuppliers.of(settings.getLightValueFunction(), ()->material);
		explosionResistance = MemoizingSuppliers.of(settings.getExplosionResistanceFunction(), ()->material);
		slipperiness = MemoizingSuppliers.of(settings.getSlipperinessFunction(), ()->material);
		shape = settings.getShape();
		raytraceShape = settings.getRaytraceShape();
		harvestTool = MemoizingSuppliers.of(settings.getHarvestToolFunction(), ()->material);
		harvestLevel = MemoizingSuppliers.of(settings.getHarvestLevelFunction(), ()->material);
		flammability = MemoizingSuppliers.of(settings.getFlammabilityFunction(), ()->material);
		fireSpreadSpeed = MemoizingSuppliers.of(settings.getFireSpreadSpeedFunction(), ()->material);
		isFireSource = MemoizingSuppliers.of(settings.getIsFireSourceFunction(), ()->material);
	}

	public static Block.Properties getProperties(IForm form, IMaterial material, IBlockFormSettings settings) {
		Block.Properties prop = Block.Properties.func_200949_a(
				settings.getMaterialFunction().apply(material),
				settings.getMaterialColorFunction().apply(material));
		prop.func_200943_b((float)settings.getBlockHardnessFunction().applyAsDouble(material));
		prop.func_235838_a_(state->settings.getLightValueFunction().applyAsInt(material));
		if(settings.getRequiresToolFunction().test(material)) {
			prop.func_235861_h_();
		}
		prop.func_226896_b_();
		return prop;
	}

	@Override
	public IForm getForm() {
		return form;
	}

	@Override
	public IMaterial getMaterial() {
		return material;
	}

	@Override
	public SoundType func_220072_p(BlockState blockState) {
		return soundType.get();
	}

	@Override
	public int func_200011_d(BlockState state, IBlockReader world, BlockPos pos) {
		return lightOpacity.getAsInt();
	}

	@Override
	public int getLightValue(BlockState state, IBlockReader world, BlockPos pos) {
		return lightValue.getAsInt();
	}

	@Override
	public float func_149638_a() {
		return (float)explosionResistance.getAsDouble();
	}

	@Override
	public float getSlipperiness(BlockState blockState, IWorldReader world, BlockPos pos, Entity entity) {
		return (float)slipperiness.getAsDouble();
	}

	@Override
	public VoxelShape func_220053_a(BlockState blockState, IBlockReader world, BlockPos pos, ISelectionContext context) {
		return shape;
	}

	@Override
	public VoxelShape func_220071_b(BlockState blockState, IBlockReader world, BlockPos pos, ISelectionContext context) {
		return blocksMovement ? blockState.func_196954_c(world, pos) : VoxelShapes.func_197880_a();
	}

	@Override
	public VoxelShape func_199600_g(BlockState blockState, IBlockReader world, BlockPos pos) {
		return raytraceShape;
	}

	@Override
	public ToolType getHarvestTool(BlockState blockState) {
		return harvestTool.get();
	}

	@Override
	public int getHarvestLevel(BlockState blockState) {
		return harvestLevel.getAsInt();
	}

	@Override
	public int getFlammability(BlockState blockState, IBlockReader world, BlockPos pos, Direction face) {
		return flammability.getAsInt();
	}

	@Override
	public int getFireSpreadSpeed(BlockState blockState, IBlockReader world, BlockPos pos, Direction face) {
		return fireSpreadSpeed.getAsInt();
	}

	@Override
	public boolean isFireSource(BlockState blockState, IWorldReader world, BlockPos pos, Direction side) {
		return isFireSource.getAsBoolean();
	}

	@Override
	public IFormattableTextComponent func_235333_g_() {
		return ApiImpl.INSTANCE.currentLocalizer().localizeMaterialForm("block.jaopca."+form.getName(), material, func_149739_a());
	}
}
