package thelm.jaopca.data;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.objectweb.asm.Type;

import com.google.common.base.Predicates;
import com.google.common.collect.TreeMultimap;

import net.minecraft.resources.IResourcePack;
import net.minecraft.resources.ResourcePackType;
import net.minecraft.resources.SimpleReloadableResourceManager;
import net.minecraft.resources.VanillaPack;
import net.minecraft.util.ResourceLocation;
import net.minecraftforge.fml.ModList;
import net.minecraftforge.fml.packs.ModFileResourcePack;
import net.minecraftforge.forgespi.language.ModFileScanData.AnnotationData;
import net.minecraftforge.registries.ForgeRegistry;
import net.minecraftforge.registries.RegistryManager;
import thelm.jaopca.api.resources.IPackSupplier;
import thelm.jaopca.api.resources.JAOPCAPackSupplier;
import thelm.jaopca.utils.MiscHelper;

public class DataCollector {

	private DataCollector() {}

	private static final Logger LOGGER = LogManager.getLogger();
	private static final int TAGS_PATH_LENGTH = "tags/".length();
	private static final int RECIPES_PATH_LENGTH = "recipes/".length();
	private static final int LOOT_TABLES_PATH_LENGTH = "loot_tables/".length();
	private static final int ADVANCEMENTS_PATH_LENGTH = "advancements/".length();
	private static final int JSON_EXTENSION_LENGTH = ".json".length();
	private static final Type JAOPCA_PACK_SUPPLIER = Type.getType(JAOPCAPackSupplier.class);
	private static final TreeMultimap<String, ResourceLocation> DEFINED_TAGS = TreeMultimap.create();
	private static final TreeSet<ResourceLocation> DEFINED_RECIPES = new TreeSet<>();
	private static final TreeSet<ResourceLocation> DEFINED_LOOT_TABLES = new TreeSet<>();
	private static final TreeSet<ResourceLocation> DEFINED_ADVANCEMENTS = new TreeSet<>();

	public static void collectData() {
		DEFINED_TAGS.clear();
		DEFINED_RECIPES.clear();
		DEFINED_ADVANCEMENTS.clear();

		List<IResourcePack> resourcePacks = new ArrayList<>();
		resourcePacks.add(new VanillaPack("minecraft"));
		ModList.get().getModFiles().stream().
		map(mf->new ModFileResourcePack(mf.getFile())).
		forEach(resourcePacks::add);
		List<AnnotationData> annotationData = ModList.get().getAllScanData().stream().
				flatMap(data->data.getAnnotations().stream()).
				filter(data->JAOPCA_PACK_SUPPLIER.equals(data.getAnnotationType())).
				collect(Collectors.toList());
		Predicate<String> modVersionNotLoaded = MiscHelper.INSTANCE.modVersionNotLoaded(LOGGER);
		Predicate<String> classNotExists = MiscHelper.INSTANCE::classNotExists;
		for(AnnotationData aData : annotationData) {
			List<String> modDeps = (List<String>)aData.getAnnotationData().get("modDependencies");
			List<String> classDeps = (List<String>)aData.getAnnotationData().get("classDependencies");
			String className = aData.getClassType().getClassName();
			if(modDeps != null && modDeps.stream().filter(Predicates.notNull()).anyMatch(modVersionNotLoaded)) {
				LOGGER.info("Pack supplier {} has missing mod dependencies, skipping", className);
				continue;
			}
			if(classDeps != null && classDeps.stream().filter(Predicates.notNull()).anyMatch(classNotExists)) {
				LOGGER.info("Pack supplier {} has missing class dependencies, skipping", className);
				continue;
			}
			try {
				Class<?> supplierClass = Class.forName(className);
				Class<? extends IPackSupplier> supplierInstanceClass = supplierClass.asSubclass(IPackSupplier.class);
				IPackSupplier supplier;
				try {
					Method method = supplierClass.getMethod("getInstance");
					supplier = (IPackSupplier)method.invoke(null);
				}
				catch(NoSuchMethodException | InvocationTargetException e) {
					supplier = supplierInstanceClass.newInstance();
				}
				supplier.addPacks(resourcePacks::add);
				LOGGER.debug("Loaded pack supplier {}", className);
			}
			catch(ClassNotFoundException | InstantiationException | IllegalAccessException e) {
				LOGGER.fatal("Unable to load pack supplier {}", className, e);
			}
		}

		SimpleReloadableResourceManager resourceManager = new SimpleReloadableResourceManager(ResourcePackType.SERVER_DATA);
		resourcePacks.forEach(resourceManager::func_199021_a);
		for(ResourceLocation location : resourceManager.func_199003_a("tags", name->name.endsWith(".json"))) {
			String namespace = location.func_110624_b();
			String path = location.func_110623_a();
			path = path.substring(TAGS_PATH_LENGTH, path.length()-JSON_EXTENSION_LENGTH);
			String[] split = path.split("/", 2);
			if(split.length == 2) {
				String type = split[0];
				path = split[1];
				DEFINED_TAGS.put(type, new ResourceLocation(namespace, path));
			}
			else {
				LOGGER.error("Tag {} in namespace {} has no type", path, namespace);
			}
		}
		LOGGER.info("Found {} unique defined tags", DEFINED_TAGS.size());
		for(ResourceLocation location : resourceManager.func_199003_a("recipes", name->name.endsWith(".json"))) {
			String namespace = location.func_110624_b();
			String path = location.func_110623_a();
			if(!path.equals("recipes/_constants.json") && !path.equals("recipes/_factories.json")) {
				path = path.substring(RECIPES_PATH_LENGTH, path.length()-JSON_EXTENSION_LENGTH);
				DEFINED_RECIPES.add(new ResourceLocation(namespace, path));
			}
		}
		LOGGER.info("Found {} unique defined recipes", DEFINED_RECIPES.size());
		for(ResourceLocation location : resourceManager.func_199003_a("loot_tables", name->name.endsWith(".json"))) {
			String namespace = location.func_110624_b();
			String path = location.func_110623_a();
			path = path.substring(LOOT_TABLES_PATH_LENGTH, path.length()-JSON_EXTENSION_LENGTH);
			DEFINED_LOOT_TABLES.add(new ResourceLocation(namespace, path));
		}
		LOGGER.info("Found {} unique defined loot tables", DEFINED_LOOT_TABLES.size());
		for(ResourceLocation location : resourceManager.func_199003_a("advancements", name->name.endsWith(".json"))) {
			String namespace = location.func_110624_b();
			String path = location.func_110623_a();
			path = path.substring(ADVANCEMENTS_PATH_LENGTH, path.length()-JSON_EXTENSION_LENGTH);
			DEFINED_ADVANCEMENTS.add(new ResourceLocation(namespace, path));
		}
		LOGGER.info("Found {} unique defined advancements", DEFINED_ADVANCEMENTS.size());
		resourceManager.close();
	}

	public static Set<ResourceLocation> getDefinedTags(ResourceLocation registry) {
		ForgeRegistry<?> forgeRegistry = RegistryManager.ACTIVE.getRegistry(registry);
		if(forgeRegistry != null) {
			String tagFolder = RegistryManager.ACTIVE.getRegistry(registry).getTagFolder();
			if(tagFolder != null) {
				return getDefinedTags(tagFolder);
			}
		}
		return Collections.emptySet();
	}

	public static Set<ResourceLocation> getDefinedTags(String type) {
		return DEFINED_TAGS.get(type);
	}

	public static Set<ResourceLocation> getDefinedRecipes() {
		return DEFINED_RECIPES;
	}

	public static Set<ResourceLocation> getDefinedLootTables() {
		return DEFINED_LOOT_TABLES;
	}

	public static Set<ResourceLocation> getDefinedAdvancements() {
		return DEFINED_ADVANCEMENTS;
	}
}
