package io.github.xrickastley.sevenelements.compat;

import java.util.HashMap;
import java.util.Map;

import org.jetbrains.annotations.ApiStatus;

import io.github.xrickastley.sevenelements.element.Element;
import io.github.xrickastley.sevenelements.element.ElementalApplication.Type;
import io.github.xrickastley.sevenelements.element.ElementalApplication;
import io.github.xrickastley.sevenelements.element.ElementalApplications;
import io.github.xrickastley.sevenelements.element.ElementalDamageSource;
import io.github.xrickastley.sevenelements.element.InternalCooldownContext;
import io.github.xrickastley.sevenelements.element.InternalCooldownTag;
import io.github.xrickastley.sevenelements.element.InternalCooldownType;
import io.github.xrickastley.sevenelements.element.PartialElementalDamageSource;
import io.github.xrickastley.sevenelements.util.ClassInstanceUtil;

import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_1282;
import net.minecraft.class_1309;
import net.minecraft.class_2960;

public class SpellPowerCompat {
	private static final Map<class_2960, Entry> SPELL_INFUSIONS = new HashMap<>();

	/**
	 * Registers a simple elemental infusion for a Spell School.
	 * @param spellSchoolId The Spell School's identifier.
	 * @param element The element this Spell School applies.
	 */
	public static void registerInfusion(class_2960 spellSchoolId, Element element) {
		SpellPowerCompat.registerInfusion(
			spellSchoolId,
			ElementalApplications.builder()
				.setElement(element)
				.setType(Type.GAUGE_UNIT)
				.setGaugeUnits(1.0)
		);
	}

	/**
	 * Registers a complex elemental infusion for a Spell School.
	 * @param spellSchoolId The Spell School's identifier.
	 * @param infusionBuilder The Elemental Application builder to use in creating the Elemental
	 * Application this Spell School applies.
	 */
	public static void registerInfusion(class_2960 spellSchoolId, ElementalApplication.Builder infusionBuilder) {
		SpellPowerCompat.registerInfusion(
			spellSchoolId,
			infusionBuilder,
			InternalCooldownContext.builder()
				.setTag(InternalCooldownTag.of("spell_power:spell_school"))
				.setType(InternalCooldownType.DEFAULT)
		);
	}

	/**
	 * Registers a complex elemental infusion for a Spell School.
	 * @param spellSchoolId The Spell School's identifier.
	 * @param infusionBuilder The Elemental Application builder to use in creating the Elemental
	 * Application this Spell School applies.
	 * @param icdBuilder The Internal Cooldown Context builder to use in creating the Internal
	 * Cooldown context of this Spell School's Elemental Application.
	 */
	public static void registerInfusion(class_2960 spellSchoolId, ElementalApplication.Builder infusionBuilder, InternalCooldownContext.Builder icdBuilder) {
		if (SpellPowerCompat.SPELL_INFUSIONS.containsKey(spellSchoolId))
			throw new IllegalStateException("The provided Spell Power id: " + spellSchoolId + " has already been registered!");

		SpellPowerCompat.SPELL_INFUSIONS.put(spellSchoolId, new Entry(infusionBuilder, icdBuilder));
	}

	@ApiStatus.Internal
	public static class_1282 create(class_1309 target, class_1282 source, class_2960 spellSchoolId) {
		if (!FabricLoader.getInstance().isModLoaded("spell_power")) return source;

		return SpellPowerCompat.SPELL_INFUSIONS.containsKey(spellSchoolId)
			? SpellPowerCompat.SPELL_INFUSIONS.get(spellSchoolId).create(target, source)
			: source;
	}

	@ApiStatus.Internal
	public static class_1282 create(class_1282 source, class_2960 spellSchoolId) {
		if (!FabricLoader.getInstance().isModLoaded("spell_power")) return source;

		return SpellPowerCompat.SPELL_INFUSIONS.containsKey(spellSchoolId)
			? SpellPowerCompat.SPELL_INFUSIONS.get(spellSchoolId).create(source)
			: source;
	}

	@ApiStatus.Internal
	private static final record Entry(ElementalApplication.Builder infusionBuilder, InternalCooldownContext.Builder icdBuilder) {
		public ElementalDamageSource create(class_1309 target, class_1282 source) {
			return new ElementalDamageSource(
				source,
				infusionBuilder.build(target),
				icdBuilder.build(ClassInstanceUtil.castOrNull(source.method_5529(), class_1309.class))
			);
		}

		public PartialElementalDamageSource create(class_1282 source) {
			return new PartialElementalDamageSource(
				source,
				infusionBuilder,
				icdBuilder.build(ClassInstanceUtil.castOrNull(source.method_5529(), class_1309.class))
			);
		}
	}

	// Uses Identifiers to reduces the chance for "cannot load class xxxx" errors when Spell Engine/Spell Power is missing.
	static {
		SpellPowerCompat.registerInfusion(
			class_2960.method_43902("spell_power", "fire"),
			Element.PYRO
		);

		SpellPowerCompat.registerInfusion(
			class_2960.method_43902("spell_power", "water"),
			Element.HYDRO
		);

		SpellPowerCompat.registerInfusion(
			class_2960.method_43902("spell_power", "air"),
			Element.ANEMO
		);

		SpellPowerCompat.registerInfusion(
			class_2960.method_43902("spell_power", "lightning"),
			Element.ELECTRO
		);

		SpellPowerCompat.registerInfusion(
			class_2960.method_43902("spell_power", "frost"),
			Element.CRYO
		);

		SpellPowerCompat.registerInfusion(
			class_2960.method_43902("spell_power", "earth"),
			Element.GEO
		);
	}
}
