package io.github.xrickastley.sevenelements.element;

import java.util.UUID;
import net.minecraft.class_1309;
import net.minecraft.class_2487;
import net.minecraft.class_2520;
import net.minecraft.class_2561;
import io.github.xrickastley.sevenelements.element.ElementalApplication.Type;
import io.github.xrickastley.sevenelements.exception.ElementalApplicationOperationException.Operation;
import io.github.xrickastley.sevenelements.exception.ElementalApplicationOperationException;

public class ElementalApplications {
	/**
	 * Creates an Elemental Application using Elemental Gauge Units.
	 * @param entity The entity to create an Elemental Application for.
	 * @param element The Element of this Elemental Application.
	 * @param gaugeUnits The amount of Elemental Gauge Units this Elemental Application has.
	 */
	public static ElementalApplication gaugeUnits(class_1309 entity, Element element, double gaugeUnits) {
		return ElementalApplications.gaugeUnits(entity, element, gaugeUnits, true);
	}

	/**
	 * Creates an Elemental Application using Elemental Gauge Units.
	 * @param entity The entity to create an Elemental Application for.
	 * @param element The Element of this Elemental Application.
	 * @param gaugeUnits The amount of Elemental Gauge Units this Elemental Application has.
	 * @param aura Whether this Elemental Application is an Aura Element. This means that the <a href="https://genshin-impact.fandom.com/wiki/Elemental_Gauge_Theory#Aura_Tax">Aura Tax</a> applies to the current gauge units of this Element.
	 */
	public static ElementalApplication gaugeUnits(class_1309 entity, Element element, double gaugeUnits, boolean aura) {
		return new GaugeUnitElementalApplication(entity, element, UUID.randomUUID(), gaugeUnits, aura);
	}

	/**
	 * Creates an Elemental Application with a specified duration.
	 * @param entity The entity to create an Elemental Application for.
	 * @param element The Element of this Elemental Application.
	 * @param gaugeUnits The amount of Gauge Units this Elemental Application has.
	 * @param duration The duration of the Elemental Application, in ticks.
	 */
	public static ElementalApplication duration(class_1309 entity, Element element, double gaugeUnits, double duration) {
		return new DurationElementalApplication(entity, element, UUID.randomUUID(), gaugeUnits, duration);
	}

	/**
	 * Creates an Elemental Application from an NBT.
	 * @param entity The entity to create an Elemental Application for.
	 * @param nbt The NBT to create the Elemental Application from.
	 */
	public static ElementalApplication fromNbt(class_1309 entity, class_2520 nbt) {
		return ElementalApplications.fromNbt(entity, nbt, entity.method_37908().method_8510());
	}

	/**
	 * Creates an Elemental Application from an NBT.
	 * @param entity The entity to create an Elemental Application for.
	 * @param nbt The NBT to create the Elemental Application from.
	 * @param syncedAt The world time this Elemental Application was last synced at.
	 */
	public static ElementalApplication fromNbt(class_1309 entity, class_2520 nbt, long syncedAt) {
		if (!(nbt instanceof final class_2487 compound)) throw new ElementalApplicationOperationException(Operation.INVALID_NBT_DATA, null, null);

		final Type type = Type.valueOf(compound.method_10558("Type"));

		return type == Type.GAUGE_UNIT
			? GaugeUnitElementalApplication.fromNbt(entity, compound, syncedAt)
			: DurationElementalApplication.fromNbt(entity, compound, syncedAt);
	}

	/**
	 * Creates a new Elemental Application Builder. <br> <br>
	 *
	 * A builder is used for creating an Elemental Application without a {@code LivingEntity} to
	 * "own" the Elemental Application as of the moment. <br> <br>
	 *
	 * Once an entity exists, an instance of {@code ElementalApplication} may be created through
	 * {@link ElementalApplication.Builder#build(class_1309) Builder#build()}.
	 */
	public static ElementalApplication.Builder builder() {
		return new ElementalApplication.Builder();
	}

	/**
	 * Gets the timer text of a provided {@link ElementalApplication}, or the standard text if
	 * unapplicable.
	 */
	public static class_2561 getTimerText(ElementalApplication application) {
		return application instanceof final DurationElementalApplication durApp
			? durApp.getTimerText()
			: application.getText();
	}
}
