package io.github.xrickastley.sevenelements.util;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;
import net.minecraft.class_243;
import net.minecraft.class_287;
import net.minecraft.class_310;
import net.minecraft.class_9799;
import org.apache.commons.lang3.function.TriConsumer;
import org.joml.Matrix4f;

import io.github.xrickastley.sevenelements.renderer.SevenElementsRenderLayer;
import io.github.xrickastley.sevenelements.renderer.SevenElementsRenderPipelines;
import io.github.xrickastley.sevenelements.renderer.SevenElementsRenderer;

/**
 * Utility class that renders circles around a certain point.
 */
public class CircleRenderer {
	private final List<Renderable> dataArray = new ArrayList<>();
	private double scaleFactor = 1 / class_310.method_1551().method_22683().method_4495();
	private double x;
	private double y;
	private double z;

	public CircleRenderer(double x, double y, double z) {
		this.x = x;
		this.y = y;
		this.z = z;
	}

	public double getX() {
		return x;
	}

	public double getY() {
		return y;
	}

	public double getZ() {
		return z;
	}

	/**
	 * Adds a circle with the set point as the center.
	 * @param radius The radius of this circle. This is measured in pixels using the Minecraft window's resolution.
	 * @param percentFilled A percentage in decimal form indicating how much of the circle is filled in. Circles are rendered starting from Quadrant 2 (90°) back to Quadrant 2 (90° + 360°).
	 * @param color The color of this circle.
	 * @return This {@code CircleRenderer}
	 */
	public CircleRenderer add(double radius, double percentFilled, Color color) {
		this.dataArray.add(
			new Circle(radius, percentFilled, color.asARGB())
		);

		return this;
	}

	/**
	 * Adds a circle with the set point as the center.
	 * @param radius The radius of this circle. This is measured in pixels using the Minecraft window's resolution.
	 * @param percentFilled A percentage in decimal form indicating how much of the circle is filled in. Circles are rendered starting from Quadrant 2 (90°) back to Quadrant 2 (90° + 360°).
	 * @param color The color of this circle.
	 * @return This {@code CircleRenderer}
	 */
	public CircleRenderer add(double radius, double percentFilled, int color) {
		this.dataArray.add(
			new Circle(radius, percentFilled, color)
		);

		return this;
	}

	/**
	 * Adds a circle outline with the set point as the center.
	 * @param radius The radius of the inner circle. This is measured in pixels using the Minecraft window's resolution.
	 * @param length The length of the outline. The resulting circle's radius, including the inner circle will be {@code radius + length}.
	 * @param percentFilled A percentage in decimal form indicating how much of the circle is filled in. Circles are rendered starting from Quadrant 2 (90°) back to Quadrant 2 (90° + 360°).
	 * @param color The color of this circle.
	 * @return This {@code CircleRenderer}
	 */
	public CircleRenderer addOutline(double radius, double length, double percentFilled, Color color) {
		this.dataArray.add(
			new CircleOutline(radius, length, percentFilled, color.asARGB())
		);

		return this;
	}

	/**
	 * Adds a circle outline with the set point as the center.
	 * @param radius The radius of the inner circle. This is measured in pixels using the Minecraft window's resolution.
	 * @param length The length of the outline. The resulting circle's radius, including the inner circle will be {@code radius + length}.
	 * @param percentFilled A percentage in decimal form indicating how much of the circle is filled in. Circles are rendered starting from Quadrant 2 (90°) back to Quadrant 2 (90° + 360°).
	 * @param color The color of this circle.
	 * @return This {@code CircleRenderer}
	 */
	public CircleRenderer addOutline(double radius, double length, double percentFilled, int color) {
		this.dataArray.add(
			new CircleOutline(radius, length, percentFilled, color)
		);

		return this;
	}

	/**
	 * Draws all the data added to this {@code CircleRenderer}, then clears all the previous data. This {@code CircleRenderer} can still be used even after {@code draw()} has been invoked, but the data provided before {@code draw()} will be cleared.
	 * @param tessellator The {@code Tessellator} instance. Obtained from {@code Tessellator.getInstance()}.
	 * @param posMatrix The position matrix. Normally obtained from {@code <MatrixStack>.peek().getPositionMatrix()}.
	 */
	public void draw(Matrix4f posMatrix) {
		final class_243 origin = new class_243(x, y, 0)
			.method_1021(scaleFactor);

		dataArray.forEach(CircleRenderer.args(Renderable::render, origin, posMatrix));
		dataArray.clear();
	}

	private class Circle implements Renderable {
		private static final class_9799 allocator = new class_9799(SevenElementsRenderLayer.getTriangleFan().method_22722());
		private static class_287 buffer;
		protected double radius;
		protected double percentFilled;
		protected int color;

		protected Circle(double radius, double percentFilled, int color) {
			this.radius = radius;
			this.percentFilled = percentFilled;
			this.color = color;
		}

		@Override
		public void render(class_243 origin, Matrix4f posMatrix) {
			buffer = SevenElementsRenderer.createBuffer(buffer, allocator, SevenElementsRenderPipelines.TRIANGLE_FAN);

			buffer
				.method_22918(posMatrix, (float) origin.method_10216(), (float) origin.method_10214(), (float) origin.method_10215())
				.method_39415(this.color);

			final double subdivisions = Math.ceil(360.0 * this.percentFilled);

			for (int i = 0; i <= subdivisions; i++) {
				final float x = (float) (origin.method_10216() + (Math.cos((i * (Math.PI / 180)) + (Math.PI / 2)) * (this.radius * scaleFactor)));
				final float y = (float) (origin.method_10214() - (Math.sin((i * (Math.PI / 180)) + (Math.PI / 2)) * (this.radius * scaleFactor)));

				buffer
					.method_22918(posMatrix, x, y, (float) z)
					.method_39415(this.color);
			}

			SevenElementsRenderLayer.getTriangleFan().method_60895(buffer.method_60800());
		}
	}

	private class CircleOutline extends Circle {
		private static final class_9799 allocator = new class_9799(SevenElementsRenderLayer.getTriangleStrip().method_22722());
		private static class_287 buffer;
		protected double length;

		protected CircleOutline(double radius, double length, double percentFilled, int color) {
			super(radius, percentFilled, color);

			this.length = length;
		}

		@Override
		public void render(class_243 origin, Matrix4f posMatrix) {
			buffer = SevenElementsRenderer.createBuffer(buffer, allocator, SevenElementsRenderPipelines.TRIANGLE_STRIP);

			final float innerRadius = (float) (this.radius * scaleFactor);
			final float totalRadius = (float) ((this.radius + this.length) * scaleFactor);

			final float x = (float) (origin.method_10216() + (Math.cos((0 * (Math.PI / 180)) + (Math.PI / 2)) * innerRadius));
			final float y = (float) (origin.method_10214() - (Math.sin((0 * (Math.PI / 180)) + (Math.PI / 2)) * innerRadius));

			buffer
				.method_22918(posMatrix, x, y, (float) origin.method_10215())
				.method_39415(this.color);

			final double subdivisions = Math.ceil(360.0 * this.percentFilled);

			for (int i = 0; i <= subdivisions; i++) {
				final float outerX = (float) (origin.method_10216() + (Math.cos((i * (Math.PI / 180)) + (Math.PI / 2)) * totalRadius));
				final float outerY = (float) (origin.method_10214() - (Math.sin((i * (Math.PI / 180)) + (Math.PI / 2)) * totalRadius));

				buffer
					.method_22918(posMatrix, outerX, outerY, (float) origin.method_10215())
					.method_39415(this.color);

				// Add vertices for the inner circle
				final float innerX = (float) (origin.method_10216() + (Math.cos((i * (Math.PI / 180)) + (Math.PI / 2)) * innerRadius));
				final float innerY = (float) (origin.method_10214() - (Math.sin((i * (Math.PI / 180)) + (Math.PI / 2)) * innerRadius));

				buffer
					.method_22918(posMatrix, innerX, innerY, (float) origin.method_10215())
					.method_39415(this.color);
			}

			SevenElementsRenderLayer.getTriangleFan().method_60895(buffer.method_60800());
		}
	}

	private interface Renderable {
		public void render(class_243 origin, Matrix4f posMatrix);
	}

	private static <T, I, J> Consumer<T> args(TriConsumer<T, I, J> consumer, I i, J j) {
		return t -> consumer.accept(t, i, j);
	}
}
