package io.github.xrickastley.sevenelements.element;

import com.mojang.serialization.Codec;

import java.text.DecimalFormat;
import java.util.UUID;
import net.minecraft.class_1309;
import net.minecraft.class_1937;
import net.minecraft.class_2487;
import net.minecraft.class_2520;
import net.minecraft.class_2561;
import net.minecraft.class_4844;
import org.jetbrains.annotations.Nullable;

import io.github.xrickastley.sevenelements.component.ElementComponent;
import io.github.xrickastley.sevenelements.events.ElementEvents;
import io.github.xrickastley.sevenelements.exception.ElementalApplicationOperationException.Operation;
import io.github.xrickastley.sevenelements.exception.ElementalApplicationOperationException;
import io.github.xrickastley.sevenelements.util.ClassInstanceUtil;
import io.github.xrickastley.sevenelements.util.JavaScriptUtil;
import io.github.xrickastley.sevenelements.util.NbtHelper;
import io.github.xrickastley.sevenelements.util.TextHelper;

public final class DurationElementalApplication extends ElementalApplication {
	private double duration;

	DurationElementalApplication(class_1309 entity, Element element, UUID uuid, double gaugeUnits, double duration) {
		super(Type.DURATION, entity, element, uuid, gaugeUnits, true);

		this.duration = duration;
	}

	static ElementalApplication fromNbt(class_1309 entity, class_2487 nbt, long syncedAt) {
		final Element element = NbtHelper.get(nbt, "Element", Element.CODEC);
		final UUID uuid = NbtHelper.get(nbt, "UUID", class_4844.field_40825);
		final double gaugeUnits = NbtHelper.get(nbt, "GaugeUnits", Codec.doubleRange(0, Double.MAX_VALUE));
		final double duration = NbtHelper.get(nbt, "Duration", Codec.doubleRange(0, Double.MAX_VALUE));

		final var application = new DurationElementalApplication(entity, element, uuid, gaugeUnits, duration);

		application.currentGauge = NbtHelper.get(nbt, "CurrentGauge", Codec.doubleRange(0, Double.MAX_VALUE));
		application.appliedAt = NbtHelper.get(nbt, "AppliedAt", Codec.LONG);

		return application;
	}

	public double getDuration() {
		return this.duration;
	}

	@Override
	protected double getDefaultDecayRate() {
		return 0;
	}

	@Override
	public int getRemainingTicks() {
		return (int) (appliedAt + duration - entity.method_37908().method_8510());
	}

	@Override
	public class_2561 getText(@Nullable DecimalFormat gaugeFormat, @Nullable DecimalFormat durationFormat) {
		gaugeFormat = JavaScriptUtil.nullishCoalesing(gaugeFormat, GAUGE_UNIT_FORMAT);
		durationFormat = JavaScriptUtil.nullishCoalesing(durationFormat, DURATION_FORMAT);

		return TextHelper.color(
			class_2561.method_43469("formats.seven-elements.elemental_application.duration", gaugeFormat.format(this.currentGauge), this.element.getString(), durationFormat.format(this.duration / 20.0)),
			this.element.getDamageColor()
		);
	}

	public class_2561 getTimerText() {
		return this.getTimerText(GAUGE_UNIT_FORMAT, DURATION_FORMAT);
	}

	public class_2561 getTimerText(@Nullable String gaugeFormat) {
		return this.getTimerText(
			ClassInstanceUtil.mapOrNull(gaugeFormat, DecimalFormat::new),
			DURATION_FORMAT
		);
	}

	public class_2561 getTimerText(@Nullable DecimalFormat gaugeFormat) {
		return this.getTimerText(gaugeFormat, DURATION_FORMAT);
	}

	public class_2561 getTimerText(@Nullable String gaugeFormat, @Nullable String durationFormat) {
		return this.getTimerText(
			ClassInstanceUtil.<String, DecimalFormat>mapOrNull(gaugeFormat, DecimalFormat::new),
			ClassInstanceUtil.<String, DecimalFormat>mapOrNull(durationFormat, DecimalFormat::new)
		);
	}

	public class_2561 getTimerText(@Nullable DecimalFormat gaugeFormat, @Nullable DecimalFormat durationFormat) {
		gaugeFormat = JavaScriptUtil.nullishCoalesing(gaugeFormat, GAUGE_UNIT_FORMAT);
		durationFormat = JavaScriptUtil.nullishCoalesing(durationFormat, DURATION_FORMAT);

		return TextHelper.color(
			class_2561.method_43469("formats.seven-elements.elemental_application.duration.timer", gaugeFormat.format(this.currentGauge), this.element.getString(), durationFormat.format(this.getRemainingTicks() / 20.0)),
			this.element.getDamageColor()
		);
	}

	/**
	 * {@inheritDoc} <br> <br>
	 *
	 * This implementation guarantees this to be {@code true} when {@code currentGauge} reaches
	 * {@code 0} or when the current world time, given by {@link class_1309#method_37908()}
	 * {@link class_1937#method_8510() .getTime()} exceeds {@code duration + appliedAt}.
	 */
	@Override
	public boolean isEmpty() {
		return this.currentGauge <= 0 || entity.method_37908().method_8510() >= (this.appliedAt + this.duration);
	}

	@Override
	public void reapply(ElementalApplication application) {
		if (application.element != this.element) throw new ElementalApplicationOperationException(Operation.REAPPLICATION_INVALID_ELEMENT, this, application);

		if (application.type != this.type || !(application instanceof final DurationElementalApplication durationApp)) throw new ElementalApplicationOperationException(Operation.REAPPLICATION_INVALID_TYPES, this, application);

		this.gaugeUnits = Math.max(this.gaugeUnits, application.gaugeUnits);
		this.currentGauge = this.gaugeUnits;
		this.duration = durationApp.duration;
		this.appliedAt = durationApp.appliedAt;

		ElementEvents.REAPPLIED.invoker().onElementReapplied(this.element, this);

		ElementComponent.sync(this.entity);
	}

	@Override
	public ElementalApplication asAura() {
		throw new UnsupportedOperationException("This method is unsupported on Elemental Applications with a DURATION type!");
	}

	@Override
	public ElementalApplication asNonAura() {
		throw new UnsupportedOperationException("This method is unsupported on Elemental Applications with a DURATION type!");
	}

	@Override
	public class_2487 asNbt()	{
		final class_2487 nbt = super.asNbt();

		nbt.method_10549("Duration", this.duration);

		return nbt;
	}

	@Override
	public void updateFromNbt(class_2520 nbt, long syncedAt) {
		super.updateFromNbt(nbt, syncedAt);

		final ElementalApplication application = ElementalApplications.fromNbt(entity, nbt, syncedAt);

		this.duration = ((DurationElementalApplication) application).duration;
		this.appliedAt = application.appliedAt;
	}

	@Override
	public String toString() {
		return String.format(
			"%s@%s[type=DURATION, element=%s, gaugeUnits=%s, duration=%.2f]",
			this.getClass().getSimpleName(),
			Integer.toHexString(this.hashCode()),
			this.getElement().toString(),
			this.getGaugeUnits(),
			this.getDuration()
		);
	}
}
