package io.github.xrickastley.sevenelements.renderer.genshin;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import org.jetbrains.annotations.Nullable;
import org.joml.Matrix4f;

import io.github.xrickastley.sevenelements.SevenElements;
import io.github.xrickastley.sevenelements.component.ElementComponent;
import io.github.xrickastley.sevenelements.element.Element;
import io.github.xrickastley.sevenelements.networking.PayloadHandler;
import io.github.xrickastley.sevenelements.networking.ShowElectroChargeS2CPayload;
import io.github.xrickastley.sevenelements.renderer.SevenElementsRenderLayer;
import io.github.xrickastley.sevenelements.renderer.SevenElementsRenderPipelines;
import io.github.xrickastley.sevenelements.renderer.SevenElementsRenderer;
import io.github.xrickastley.sevenelements.util.BoxUtil;
import io.github.xrickastley.sevenelements.util.ClientConfig;
import io.github.xrickastley.sevenelements.util.Color;
import io.github.xrickastley.sevenelements.util.Colors;
import io.github.xrickastley.sevenelements.util.Ease;
import io.github.xrickastley.sevenelements.util.Functions;
import io.github.xrickastley.sevenelements.util.JavaScriptUtil;
import io.github.xrickastley.sevenelements.util.polyfill.rendering.WorldRenderContext;

import net.fabricmc.fabric.api.client.networking.v1.ClientPlayNetworking.Context;
import net.minecraft.class_1297;
import net.minecraft.class_1309;
import net.minecraft.class_1921;
import net.minecraft.class_1937;
import net.minecraft.class_238;
import net.minecraft.class_243;
import net.minecraft.class_287;
import net.minecraft.class_310;
import net.minecraft.class_3532;
import net.minecraft.class_4184;
import net.minecraft.class_4587;
import net.minecraft.class_5819;
import net.minecraft.class_638;
import net.minecraft.class_746;
import net.minecraft.class_7833;
import net.minecraft.class_8710;
import net.minecraft.class_9799;

public final class SpecialEffectsRenderer implements PayloadHandler<ShowElectroChargeS2CPayload> {
	private static final int MAX_TICKS = 10;
	private static final double POISSON_DENSITY = 1.5;
	private static final class_5819 RANDOM = class_5819.method_43047();
	private static final int CHARGE_ITERATIONS = 4;
	private static final class_9799 allocator = SevenElementsRenderer.createAllocator(class_1921.field_32772);
	private final List<Entry> entries = new ArrayList<>();
	private final Multimap<class_1309, ChargeLinePositions> chargePositions = HashMultimap.create();

	/**
	 * Returns whether effects should be rendered for the provided entity.
	 * @param entity The entity planned to render effects for.
	 */
	public static boolean shouldRender(class_1297 entity) {
		final class_310 client = class_310.method_1551();

		return entity.method_5805()
			&& (entity != client.field_1724 || client.field_1773.method_19418().method_19333());
	}

	@Override
	public class_8710.class_9154<ShowElectroChargeS2CPayload> getPayloadId() {
		return ShowElectroChargeS2CPayload.ID;
	}

	@Override
	public void receive(ShowElectroChargeS2CPayload payload, Context context) {
		final class_746 player = context.player();
		final class_1937 world = player.method_73183();
		final class_1297 mainEntity = world.method_8469(payload.mainEntity());

		if (mainEntity == null) {
			SevenElements.sublogger().warn("Received packet for unknown main Electro-Charged entity, ignoring!");

			return;
		}

		entries.add(
			new ElectroChargedEffect(
				mainEntity,
				payload
					.otherEntities()
					.stream()
					.map(world::method_8469)
					.filter(e -> e != null)
					.toList()
			)
		);
	}

	public void render(WorldRenderContext context) {
		entries.forEach(Functions.withArgument(Entry::render, context, this));

		this.renderEffects(context);
	}

	public void tick(class_638 world) {
		this.entries.removeIf(Entry::shouldRemove);
		this.entries.forEach(Entry::tick);

		if (world.method_8510() % 10 == 0) this.chargePositions.clear();

		this.chargePositions
			.values()
			.forEach(ChargeLinePositions::clearPositions);
	}

	private Collection<ChargeLinePositions> getChargePositions(class_1309 entity) {
		final Collection<ChargeLinePositions> mapValue = this.chargePositions.get(entity);

		if (!mapValue.isEmpty()) return mapValue;

		final List<ChargeLinePositions> computedValue = new ArrayList<>();
		final class_238 box = BoxUtil.multiplyBox(entity.method_5829(), 0.75);

		for (int i = 0; i < SpecialEffectsRenderer.CHARGE_ITERATIONS + 2; i++) {
			final class_243 initialPos = BoxUtil.randomPos(box);
			final class_243 finalPos = BoxUtil.randomPos(box);

			computedValue.add(new ChargeLinePositions(initialPos, finalPos, entity));
		}

		this.chargePositions.putAll(entity, computedValue);

		return computedValue;
	}

	private void renderQuickenAura(WorldRenderContext context, class_1309 entity) {
		if (!ClientConfig.getEffectRenderType().allowsSpecialEffects()) return;

		this.getChargePositions(entity)
			.forEach(clp -> {
				final Color color = clp.computeColorIfAbsent(() -> Math.random() < 0.5 ? Colors.ELECTRO : Colors.DENDRO);

				this.renderChargeLine(context, clp, entity, color, Colors.PHYSICAL);
			});
	}

	private void renderElectroAura(WorldRenderContext context, class_1309 entity) {
		if (!ClientConfig.getEffectRenderType().allowsNormalEffects()) return;

		this.getChargePositions(entity)
			.forEach(clp -> this.renderChargeLine(context, clp, entity, Colors.ELECTRO, Colors.PHYSICAL));
	}

	private void renderEffects(WorldRenderContext context) {
		for (final class_1297 entity : context.world().method_18112()) {
			if (!(entity instanceof final class_1309 livingEntity) || !shouldRender(livingEntity)) continue;

			final ElementComponent component = ElementComponent.KEY.get(livingEntity);

			if (component.hasElementalApplication(Element.QUICKEN)) this.renderQuickenAura(context, livingEntity);
			else if (component.hasElementalApplication(Element.ELECTRO)) this.renderElectroAura(context, livingEntity);
		}
	}

	private void renderChargeLine(WorldRenderContext context, ChargeLinePositions clp, class_1297 entity, Color outerColor, Color innerColor) {
		final class_243 initialPos = clp.getInitialPos(entity);

		this.renderChargeLine(context, initialPos, clp.generatePositions(this, entity), outerColor, innerColor);
	}

	@SuppressWarnings("unused")
	private void renderChargeLine(WorldRenderContext context, class_243 initialPos, class_243 finalPos, Color outerColor, Color innerColor) {
		final List<class_243> positions = this.generatePositions(class_243.field_1353, initialPos.method_1020(finalPos));

		class_243 randomVec = class_243.field_1353;

		for (int i = 0; i < positions.size(); i++) {
			randomVec = new class_243(RANDOM.method_43058() - 0.5, RANDOM.method_43058() - 0.5, RANDOM.method_43058() - 0.5);

			positions.set(i, positions.get(i).method_1019(randomVec));
		}

		positions.add(0, class_243.field_1353);
		positions.add(finalPos.method_1020(initialPos));

		this.renderChargeLine(context, initialPos, positions, outerColor, innerColor);
	}

	private void renderChargeLine(WorldRenderContext context, class_243 origin, List<class_243> positions, Color outerColor, Color innerColor) {
	    final class_4184 camera = context.camera();
	    final class_243 camPos = camera.method_19326();

	    final class_4587 matrices = new class_4587();
	    matrices.method_22903();
		matrices.method_22907(class_7833.field_40714.rotationDegrees(camera.method_19329()));
		matrices.method_22907(class_7833.field_40716.rotationDegrees(camera.method_19330() + 180.0F));
	    matrices.method_22904(origin.field_1352 - camPos.field_1352, origin.field_1351 - camPos.field_1351, origin.field_1350 - camPos.field_1350);

	    final Matrix4f posMat = matrices.method_23760().method_23761();
	    final class_4587.class_4665 entry = matrices.method_23760();

		final class_287 outerLineBuffer = SevenElementsRenderer.createBuffer(allocator, SevenElementsRenderPipelines.CHARGE_LINE);

		for (int i = 1; i < positions.size(); i++) {
			final class_243 start = positions.get(i - 1);
			final class_243 end = positions.get(i);
			class_243 normal = end.method_1029();

		    outerLineBuffer.method_22918(posMat, (float) start.field_1352, (float) start.field_1351, (float) start.field_1350)
		       .method_39415(outerColor.asARGB())
		       .method_60831(entry, (float) normal.field_1352, (float) normal.field_1351, (float) normal.field_1350);

		    outerLineBuffer.method_22918(posMat, (float) end.field_1352, (float) end.field_1351, (float) end.field_1350)
		       .method_39415(outerColor.asARGB())
		       .method_60831(entry, (float) normal.field_1352, (float) normal.field_1351, (float) normal.field_1350);
		}

		SevenElementsRenderLayer.getOuterChargeLine().method_60895(outerLineBuffer.method_60800());

		final class_287 innerLineBuffer = SevenElementsRenderer.createBuffer(allocator, SevenElementsRenderPipelines.CHARGE_LINE);

		for (int i = 1; i < positions.size(); i++) {
			final class_243 start = positions.get(i - 1);
			final class_243 end = positions.get(i);
			class_243 normal = end.method_1029();

		    innerLineBuffer.method_22918(posMat, (float) start.field_1352, (float) start.field_1351, (float) start.field_1350)
		       .method_39415(innerColor.asARGB())
		       .method_60831(entry, (float) normal.field_1352, (float) normal.field_1351, (float) normal.field_1350);

		    innerLineBuffer.method_22918(posMat, (float) end.field_1352, (float) end.field_1351, (float) end.field_1350)
		       .method_39415(innerColor.asARGB())
		       .method_60831(entry, (float) normal.field_1352, (float) normal.field_1351, (float) normal.field_1350);
		}

		SevenElementsRenderLayer.getInnerChargeLine().method_60895(innerLineBuffer.method_60800());

	    matrices.method_22909();
	}

	private List<class_243> generatePositions(final class_243 initialPos, final class_243 finalPos) {
		final class_243 norm = initialPos.method_1020(finalPos);
		final double length = norm.method_1033();

		final int n = Math.max(1, this.poisson(SpecialEffectsRenderer.POISSON_DENSITY * length));
		final List<Double> doubles = new ArrayList<>();

		for (int i = 0; i < n; i++) doubles.add(RANDOM.method_43058());

		return doubles
			.stream()
			.sorted()
			.map(t -> initialPos.method_1019(norm.method_1021(t)).method_1019(new class_243(RANDOM.method_43058() - 0.5, RANDOM.method_43058() - 0.5, RANDOM.method_43058() - 0.5)))
			.collect(Collectors.toList());
	}

	private int poisson(double lambda) {
		final double L = Math.exp(-lambda);

		int k = 0;
		double p = 1.0;
		do {
			k++;
			p *= SpecialEffectsRenderer.RANDOM.method_43058();
		} while (p > L);

		return k - 1;
	}

	private static abstract class Entry {
		abstract boolean shouldRemove();
		abstract void render(final WorldRenderContext context, final SpecialEffectsRenderer renderer);
		void tick() {};
	}

	private static class ElectroChargedEffect extends Entry {
		private final long time;
		private final class_1297 mainEntity;
		private final List<class_1297> otherEntities;
		private final Map<class_1297, StoredElectroChargedPositions> positionMap = new HashMap<>();

		private ElectroChargedEffect(class_1297 mainEntity, List<class_1297> otherEntities) {
			this.time = class_310.method_1551().field_1687.method_8510();
			this.mainEntity = mainEntity;
			this.otherEntities = otherEntities;
		}

		boolean shouldRemove() {
			return !mainEntity.method_5805() || otherEntities.isEmpty() || class_310.method_1551().field_1687.method_8510() > this.time + MAX_TICKS;
		}

		void render(final WorldRenderContext context, final SpecialEffectsRenderer renderer) {
			final double gradientStep = class_3532.method_15363(class_3532.method_37960(class_310.method_1551().field_1687.method_8510() - this.time + context.tickCounter().method_60637(false), 0, 10), 0, 1);
			final Color outerColor = Color.gradientStep(Colors.ELECTRO, Colors.HYDRO, gradientStep, Ease.IN_QUART);
			final Color innerColor = Colors.PHYSICAL;

			for (final class_1297 other : this.otherEntities) {
				if (other == this.mainEntity) continue;

				final StoredElectroChargedPositions entry = positionMap.computeIfAbsent(other, o -> new StoredElectroChargedPositions(this.mainEntity, other));

				renderer.renderChargeLine(context, entityPos(this.mainEntity), entry.generatePositions(renderer), outerColor, innerColor);
			}
		}

		@Override
		void tick() {
			super.tick();

			this.positionMap.values().forEach(StoredElectroChargedPositions::tick);
		}

		private class_243 entityPos(class_1297 entity) {
			return entity.method_73189().method_1031(0, entity.method_17682() * 0.5, 0);
		}
	}

	private static class StoredElectroChargedPositions {
		private final class_1297 mainEntity;
		private final class_1297 targetEntity;
		private class_243 prevMainEntityPos;
		private class_243 prevTargetEntityPos;
		private @Nullable List<class_243> positions = null;

		private StoredElectroChargedPositions(class_1297 mainEntity, class_1297 targetEntity) {
			this.mainEntity = mainEntity;
			this.targetEntity = targetEntity;
			this.prevMainEntityPos = this.entityPos(mainEntity);
			this.prevTargetEntityPos = this.entityPos(targetEntity);
		}

		private void tick() {
			this.positions = null;
		}

		private boolean shouldPositionsPersist() {
			return this.entityPos(mainEntity).equals(prevMainEntityPos)
				&& this.entityPos(targetEntity).equals(prevTargetEntityPos);
		}

		private List<class_243> generatePositions(SpecialEffectsRenderer renderer) {
			if (this.positions != null && shouldPositionsPersist()) return this.positions;

			// Required unequal due to shouldPositionsPersist(), refresh
			final class_243 initialPos = this.prevMainEntityPos = this.entityPos(this.mainEntity);
			final class_243 finalPos = this.prevTargetEntityPos = this.entityPos(this.targetEntity);

			final List<class_243> positions = renderer.generatePositions(class_243.field_1353, initialPos.method_1020(finalPos));

			class_243 randomVec = class_243.field_1353;

			for (int i = 0; i < positions.size(); i++) {
				randomVec = new class_243(RANDOM.method_43058() - 0.5, RANDOM.method_43058() - 0.5, RANDOM.method_43058() - 0.5);

				positions.set(i, positions.get(i).method_1019(randomVec));
			}

			positions.add(0, class_243.field_1353);
			positions.add(finalPos.method_1020(initialPos));

			this.positions = positions;

			return positions;
		}

		private class_243 entityPos(class_1297 entity) {
			return entity.method_73189().method_1031(0, entity.method_17682() * 0.5, 0);
		}
	}

	private static class ChargeLinePositions {
		private final class_243 initialPos;
		private final class_243 finalPos;
		private @Nullable List<class_243> positions = null;
		private @Nullable Color color = null;

		private ChargeLinePositions(class_243 initialPos, class_243 finalPos, class_1297 relativeTo) {
			final class_243 entityPos = relativeTo.method_73189();

			this.initialPos = initialPos.method_1020(entityPos);
			this.finalPos = finalPos.method_1020(entityPos);
		}

		private class_243 getInitialPos(class_1297 relativeTo) {
			return this.initialPos.method_1019(relativeTo.method_73189());
		}

		private List<class_243> generatePositions(SpecialEffectsRenderer renderer, class_1297 relativeTo) {
			if (this.positions != null) return this.positions;

			final class_243 entityPos = relativeTo.method_73189();
			final List<class_243> positions = renderer.generatePositions(class_243.field_1353, initialPos.method_1020(finalPos));

			class_243 randomVec = class_243.field_1353;

			for (int i = 0; i < positions.size(); i++) {
				randomVec = new class_243(RANDOM.method_43058() - 0.5, RANDOM.method_43058() - 0.5, RANDOM.method_43058() - 0.5);

				positions.set(i, positions.get(i).method_1019(randomVec));
			}

			positions.add(0, class_243.field_1353);
			positions.add(finalPos.method_1020(initialPos));

			this.positions = positions;

			return positions
				.stream()
				.map(Functions.<class_243, class_243, class_243>withArgument(class_243::method_1019, entityPos))
				.toList();
		}

		private void clearPositions() {
			this.positions = null;
		}

		private @Nullable Color getColor() {
			return this.color;
		}

		private Color computeColorIfAbsent(Supplier<Color> ifAbsent) {
			return this.color = JavaScriptUtil.nullishCoalesing(this.color, ifAbsent.get());
		}
	}

	static {

	}
}
