package io.github.xrickastley.sevenelements.registry.dynamic;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import com.google.gson.JsonElement;
import com.google.gson.JsonParser;
import com.mojang.serialization.Codec;
import com.mojang.serialization.DataResult;
import com.mojang.serialization.Decoder;
import com.mojang.serialization.JsonOps;

import java.io.IOException;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.function.BiFunction;
import java.util.function.Predicate;
import net.minecraft.class_2378;
import net.minecraft.class_2385;
import net.minecraft.class_2509;
import net.minecraft.class_2520;
import net.minecraft.class_2960;
import net.minecraft.class_3298;
import net.minecraft.class_3300;
import net.minecraft.class_5321;
import net.minecraft.class_5912;
import net.minecraft.class_6903;
import net.minecraft.class_6903.class_7863;
import net.minecraft.class_7654;
import net.minecraft.class_7655;
import net.minecraft.class_7782.class_9176;
import net.minecraft.class_7924;
import net.minecraft.class_9248;
import org.jetbrains.annotations.Nullable;

import io.github.xrickastley.sevenelements.SevenElements;
import io.github.xrickastley.sevenelements.element.InternalCooldownType;
import io.github.xrickastley.sevenelements.mixin.RegistryLoaderAccessor;
import io.github.xrickastley.sevenelements.registry.SevenElementsRegistryKeys;
import io.github.xrickastley.sevenelements.registry.dynamic.DynamicRegistryLoadEvents.RegistryContextImpl;
import io.github.xrickastley.sevenelements.registry.dynamic.DynamicRegistryLoadEvents.RegistryEntryContextImpl;
import io.github.xrickastley.sevenelements.util.ClassInstanceUtil;

public final class SevenElementsRegistryLoader {
	private static final List<Entry<?, ?>> DYNAMIC_REGISTRIES = new ArrayList<>();
	private static final Multimap<class_5321<?>, class_2960> UNMODIFIABLE_ENTRIES = HashMultimap.create();

	static void add(Entry<?, ?> entry) {
		SevenElementsRegistryLoader.DYNAMIC_REGISTRIES.add(entry);
	}

	static void addUnmodifiableEntries(class_5321<? extends class_2378<?>> key, class_2960... ids) {
		SevenElementsRegistryLoader.addUnmodifiableEntries(key, List.of(ids));
	}

	static void addUnmodifiableEntries(class_5321<? extends class_2378<?>> key, List<class_2960> ids) {
		if (!SevenElementsRegistryLoader.isDynamicRegistry(key))
			throw new IllegalArgumentException("You may only pass a dynamic registry registered to the SevenElementsRegistryLoader!");

		SevenElementsRegistryLoader.UNMODIFIABLE_ENTRIES.putAll(key, ids);
	}

	public static <E> void loadFromResource(class_3300 resourceManager, class_7863 infoGetter, class_2385<E> registry, Decoder<E> elementDecoder, Map<class_5321<?>, Exception> errors) {
		final @Nullable Entry<? extends E, ?> dynRegEntry = SevenElementsRegistryLoader.getDynamicRegistry(registry);

		if (dynRegEntry == null)
			throw new IllegalArgumentException("You may only pass a dynamic registry registered to the SevenElementsRegistryLoader!");

		DynamicRegistryLoadEvents.BEFORE_LOAD.invoker().onBeforeLoad(new RegistryContextImpl<>(registry.method_30517(), registry));

		dynRegEntry.requireUnmodifiableEntries(registry);

		final String path = dynRegEntry.getPath();
		final class_7654 resourceFinder = class_7654.method_45114(path);
		final class_6903<JsonElement> registryOps = class_6903.method_40414(JsonOps.INSTANCE, infoGetter);

		for (java.util.Map.Entry<class_2960, class_3298> entry : resourceFinder.method_45113(resourceManager).entrySet()) {
			final class_2960 identifier = entry.getKey();
			final class_2960 resourceId = resourceFinder.method_45115(identifier);

			if (dynRegEntry.isUnmodifiable(resourceId)) {
				SevenElements
					.sublogger()
					.warn("The data pack (\"{}\") with file at path ({}/{}) attempted to overwrite the preloaded entry {}, ignoring!", entry.getValue().method_14480(), identifier.method_12836(), identifier.method_12832(), resourceId);

				continue;
			}

			final class_5321<E> registryKey = class_5321.method_29179(registry.method_30517(), resourceId);
			final class_3298 resource = entry.getValue();
			final class_9248 registryEntryInfo = RegistryLoaderAccessor.getResourceEntryInfoGetter().apply(resource.method_56936());

			try {
				parseAndAdd(registry, ClassInstanceUtil.cast(dynRegEntry), registryOps, registryKey, resourceId, resource, registryEntryInfo);
			} catch (Exception var15) {
				errors.put(registryKey, new IllegalStateException(String.format(Locale.ROOT, "Failed to parse %s from pack %s", identifier, resource.method_14480()), var15));
			}
		}

		DynamicRegistryLoadEvents.AFTER_LOAD.invoker().onAfterLoad(new RegistryContextImpl<>(registry.method_30517(), registry));
	}

	public static <E> void loadFromNetwork(
		Map<class_5321<? extends class_2378<?>>, List<class_9176>> data,
		class_5912 factory,
		class_7863 infoGetter,
		class_2385<E> registry,
		Decoder<E> decoder,
		Map<class_5321<?>, Exception> loadingErrors
	) {
		final @Nullable Entry<? extends E, ?> dynRegEntry = SevenElementsRegistryLoader.getDynamicRegistry(registry);

		if (dynRegEntry == null)
			throw new IllegalArgumentException("You may only pass a dynamic registry registered to the SevenElementsRegistryLoader!");

		DynamicRegistryLoadEvents.BEFORE_LOAD.invoker().onBeforeLoad(new RegistryContextImpl<>(registry.method_30517(), registry));

		dynRegEntry.requireUnmodifiableEntries(registry);

		List<class_9176> list = data.get(registry.method_30517());
		if (list != null) {
			class_6903<class_2520> registryOps = class_6903.method_40414(class_2509.field_11560, infoGetter);
			class_6903<JsonElement> registryOps2 = class_6903.method_40414(JsonOps.INSTANCE, infoGetter);
			String string = class_7924.method_60915(registry.method_30517());
			class_7654 resourceFinder = class_7654.method_45114(string);

			for (class_9176 serializedRegistryEntry : list) {
				if (dynRegEntry.isUnmodifiable(serializedRegistryEntry.comp_2256())) continue;

				class_5321<E> registryKey = class_5321.method_29179(registry.method_30517(), serializedRegistryEntry.comp_2256());
				Optional<class_2520> optional = serializedRegistryEntry.comp_2257();
				if (optional.isPresent()) {
					try {
						DataResult<E> dataResult = decoder.parse(registryOps, optional.get());
						E object = dataResult.getOrThrow();
						registry.method_10272(registryKey, object, RegistryLoaderAccessor.getExperimentalEntryInfo());

						DynamicRegistryLoadEvents.ENTRY_LOAD.invoker().onEntryLoad(new RegistryEntryContextImpl<>(object, registry.method_30517(), registry));
					} catch (Exception var17) {
						loadingErrors.put(registryKey, new IllegalStateException(String.format(Locale.ROOT, "Failed to parse value %s from server", optional.get()), var17));
					}
				} else {
					class_2960 identifier = resourceFinder.method_45112(serializedRegistryEntry.comp_2256());

					try {
						class_3298 resource = factory.getResourceOrThrow(identifier);
						final class_2960 resourceId = resourceFinder.method_45115(identifier);

						if (dynRegEntry.isUnmodifiable(resourceId)) {
							SevenElements
								.sublogger()
								.warn("The data pack (\"{}\") with file at path ({}/{}) attempted to overwrite the preloaded entry {}, ignoring!", resource.method_14480(), identifier.method_12836(), identifier.method_12832(), resourceId);

							continue;
						}

						parseAndAdd(registry, ClassInstanceUtil.cast(dynRegEntry), registryOps2, registryKey, resourceFinder.method_45115(identifier), resource, RegistryLoaderAccessor.getExperimentalEntryInfo());
					} catch (Exception var18) {
						loadingErrors.put(registryKey, new IllegalStateException("Failed to parse local data", var18));
					}
				}
			}

			DynamicRegistryLoadEvents.AFTER_LOAD.invoker().onAfterLoad(new RegistryContextImpl<>(registry.method_30517(), registry));
		}
	}

	public static <E> void parseAndAdd(class_2385<E> registry, Entry<E, ?> entry, class_6903<JsonElement> ops, class_5321<E> key, class_2960 identifier, class_3298 resource, class_9248 entryInfo) throws IOException {
		Reader reader = resource.method_43039();

		try {
			JsonElement jsonElement = JsonParser.parseReader(reader);
			E object = entry.parse(ops, jsonElement, identifier);
			registry.method_10272(key, object, entryInfo);

			DynamicRegistryLoadEvents.ENTRY_LOAD.invoker().onEntryLoad(new RegistryEntryContextImpl<>(object, registry.method_30517(), registry));
		} catch (Throwable var11) {
			if (reader != null) {
				try {
					reader.close();
				} catch (Throwable var10) {
					var11.addSuppressed(var10);
				}
			}
			throw var11;
		}

		if (reader != null)
			reader.close();
	}

	public static boolean isDynamicRegistry(class_2378<?> registry) {
		return SevenElementsRegistryLoader.isDynamicRegistry(registry.method_30517());
	}

	public static boolean isDynamicRegistry(class_5321<? extends class_2378<?>> registryKey) {
		return SevenElementsRegistryLoader.DYNAMIC_REGISTRIES
			.stream()
			.anyMatch(entry -> entry.key == registryKey);
	}

	private static <T, C> @Nullable Entry<T, C> getDynamicRegistry(class_2378<T> registry) {
		return SevenElementsRegistryLoader.getDynamicRegistry(registry.method_30517());
	}

	private static <T, C> @Nullable Entry<T, C> getDynamicRegistry(class_5321<? extends class_2378<T>> registryKey) {
		for (final Entry<?, ?> entry : SevenElementsRegistryLoader.DYNAMIC_REGISTRIES) {
			if (entry.key != registryKey) continue;

			return ClassInstanceUtil.cast(entry);
		}

		return null;
	}

	/**
	 * A dynamic registry entry. <br> <br>
	 *
	 * Here, {@code C} must equal {@code T}.
	 */
	public static class Entry<T, C> {
		private final Class<T> entryClass;
		private final class_5321<? extends class_2378<T>> key;
		private final Codec<C> elementCodec;
		private final boolean requiredNonEmpty;
		private boolean useNamespace = false;

		public Entry(Class<T> entryClass, class_5321<? extends class_2378<T>> registryKey, Codec<C> codec) {
			this(entryClass, registryKey, codec, false);
		}

		public Entry(Class<T> entryClass, class_5321<? extends class_2378<T>> key, Codec<C> elementCodec, boolean requiredNonEmpty) {
			this.entryClass = entryClass;
			this.key = key;
			this.elementCodec = elementCodec;
			this.requiredNonEmpty = requiredNonEmpty;
		}

		/**
		 * Whether the namespace should be used in the data pack entry path. <br> <br>
		 *
		 * This avoids conflict with other mods that may use the same folder path.
		 */
		public void shouldUseNamespace(boolean useNamespace) {
			this.useNamespace = useNamespace;
		}

		/**
		 * The expected path of data pack entries. <br> <br>
		 *
		 * When using an {@code SevenElementsRegistryLoader.Entry}, <b>always</b> prefer this
		 * method over {@link class_7924#method_60915(class_5321)}.
		 *
		 * @return The expected path of data pack entries.
		 */
		public String getPath() {
			final String path = class_7924.method_60915(key);

			return this.useNamespace
				? key.method_29177().method_12836() + "/" + path
				: path;
		}

		public class_7655.class_7657<T> asRegistryLoaderEntry() {
			return new class_7655.class_7657<>(key, ClassInstanceUtil.cast(elementCodec), requiredNonEmpty);
		}

		public T parse(class_6903<JsonElement> ops, JsonElement jsonElement, class_2960 identifier) {
			DataResult<C> dataResult = this.elementCodec.parse(ops, jsonElement);

			return entryClass.cast(dataResult.getOrThrow());
		}

		public T parse(class_6903<class_2520> ops, class_2520 nbt, class_2960 identifier) {
			DataResult<C> dataResult = this.elementCodec.parse(ops, nbt);

			return entryClass.cast(dataResult.getOrThrow());
		}

		public boolean isUnmodifiable(class_2960 id) {
			final @Nullable Collection<class_2960> entries = SevenElementsRegistryLoader.UNMODIFIABLE_ENTRIES.get(key);

			return entries != null && entries.contains(id);
		}

		public void requireUnmodifiableEntries(class_2385<?> registry) {
			final @Nullable Collection<class_2960> entries = SevenElementsRegistryLoader.UNMODIFIABLE_ENTRIES.get(key);

			if (entries == null) return;

			final List<class_2960> unregistered = entries
				.stream()
				.filter(Predicate.not(registry::method_10250))
				.toList();

			if (unregistered.isEmpty()) return;

			throw new IllegalStateException("Some unmodifiable holders were not registered: " + unregistered);
		}
	}

	/**
	 * Variant of Entry that creates a "builder" object, then passes an Identifier to create the
	 * target object. <br> <br>
	 *
	 * Here, {@code T} is the "builder" for the serialized data and {@code T} is the result object of the builder.
	 */
	public static class IdentifiedEntry<T, R> extends Entry<T, R> {
		private final BiFunction<R, class_2960, T> resultFn;

		public IdentifiedEntry(Class<T> resultClass, class_5321<? extends class_2378<T>> registryKey, Codec<R> resultCodec, BiFunction<R, class_2960, T> resultFn) {
			this(resultClass, registryKey, resultCodec, resultFn, false);
		}

		public IdentifiedEntry(Class<T> resultClass, class_5321<? extends class_2378<T>> registryKey, Codec<R> resultCodec, BiFunction<R, class_2960, T> resultFn, boolean requiredNonEmpty) {
			// T is a generic anyway, just ensure transformation before setting.
			super(resultClass, ClassInstanceUtil.cast(registryKey), resultCodec, requiredNonEmpty);

			this.resultFn = resultFn;
		}

		@Override
		public T parse(class_6903<JsonElement> ops, JsonElement jsonElement, class_2960 identifier) {
			DataResult<R> dataResult = super.elementCodec.parse(ops, jsonElement);

			return this.resultFn.apply(dataResult.getOrThrow(), identifier);
		}

		@Override
		public T parse(class_6903<class_2520> ops, class_2520 nbt, class_2960 identifier) {
			DataResult<R> dataResult = super.elementCodec.parse(ops, nbt);

			return this.resultFn.apply(dataResult.getOrThrow(), identifier);
		}
	}

	static {
		new SevenElementsRegistryLoader.IdentifiedEntry<>(
			InternalCooldownType.class,
			SevenElementsRegistryKeys.INTERNAL_COOLDOWN_TYPE,
			InternalCooldownType.Builder.CODEC,
			InternalCooldownType.Builder::getInstance
		);
	}
}
