package de.larsensmods.mctrains;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonIOException;
import com.google.gson.JsonSyntaxException;
import com.google.gson.reflect.TypeToken;
import de.larsensmods.mctrains.config.MCTConfig;
import de.larsensmods.mctrains.interfaces.IChainable;
import net.fabricmc.api.ModInitializer;
import net.fabricmc.fabric.api.event.player.UseEntityCallback;
import net.minecraft.class_124;
import net.minecraft.class_1269;
import net.minecraft.class_1297;
import net.minecraft.class_1657;
import net.minecraft.class_1688;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_2487;
import net.minecraft.class_2588;
import net.minecraft.class_3218;
import net.minecraft.class_3417;
import net.minecraft.class_3419;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

public class MinecartTrains implements ModInitializer {

    public static final String MOD_ID = "minecart-trains";
    public static final Logger LOGGER = LoggerFactory.getLogger(MOD_ID);

    public static MCTConfig CONFIG;

    // NBT 键与持久化回退文件
    private static final String NBT_PARENT_KEY = "minecart_trains:parent_id";
    private static final File FALLBACKS_FILE = new File("config", "minecart-trains-fallbacks.json");
    private static final Gson FALLBACKS_GSON = new GsonBuilder().create();

    // 内存回退缓存（player UUID -> parent UUID）
    private static final ConcurrentMap<UUID, UUID> PLAYER_PARENT_FALLBACK = new ConcurrentHashMap<>();

    @Override
    public void onInitialize() {
        // 先加载持久化回退（如果存在）
        loadPersistentFallbacks();

        // 再加载/创建配置
        loadOrCreateConfig();

        // UseEntityCallback：潜行 + 链条行为（服务器端执行主要逻辑）
        UseEntityCallback.EVENT.register((player, world, hand, entity, hitResult) -> {
            if (!(entity instanceof class_1688 cart)) return class_1269.field_5811;
            class_1799 stack = player.method_5998(hand);

            if (!player.method_5715() || !stack.method_31574(class_1802.field_23983) || CONFIG == null || !CONFIG.enableCartChaining()) {
                return class_1269.field_5811;
            }

            if (!(world instanceof class_3218 server)) {
                return class_1269.field_5812;
            }

            UUID stored = getStackParent(stack, player);

            if (stored != null && !cart.method_5667().equals(stored)) {
                class_1297 maybeParent = server.method_14190(stored);
                if (maybeParent instanceof class_1688 && maybeParent instanceof IChainable parentChain) {
                    Set<IChainable> train = new HashSet<>();
                    train.add(parentChain);

                    // 遍历父链（全部以 IChainable 操作）
                    IChainable cursor = parentChain.getChainedParent();
                    while (cursor != null && !train.contains(cursor)) {
                        train.add(cursor);
                        cursor = cursor.getChainedParent();
                    }

                    // 确保 cart 实现 IChainable
                    if (!(cart instanceof IChainable cartChain)) {
                        removeStackParent(stack, player);
                    } else {
                        // 检查环或父已有子车
                        if (train.contains(cartChain) || parentChain.getChainedChild() != null) {
                            player.method_7353(new class_2588(MOD_ID + ".invalid_chaining").method_27692(class_124.field_1061), true);
                        } else {
                            // 先断开已有父链（如果有）
                            if (cartChain.getChainedParent() != null) {
                                IChainable.unsetChainedParentChild(cartChain, cartChain.getChainedParent());
                            }
                            // 建立新链（均为 IChainable）
                            IChainable.setChainedParentChild(parentChain, cartChain);
                        }
                    }
                } else {
                    removeStackParent(stack, player);
                }


                world.method_8465(null, cart.method_23317(), cart.method_23318(), cart.method_23321(), class_3417.field_24063, class_3419.field_15254, 1f, 1.1f);

                if (!player.method_7337()) {
                    stack.method_7934(1);
                }

                removeStackParent(stack, player);
            } else {
                setStackParent(stack, cart.method_5667(), player);
                world.method_8465(null, cart.method_23317(), cart.method_23318(), cart.method_23321(), class_3417.field_24062, class_3419.field_15254, 1f, 1.1f);
            }

            return class_1269.field_5812;
        });
    }

    // --------------------------
    // 配置加载（保留原来的行为）
    // --------------------------
    private void loadOrCreateConfig() {
        Gson gson = new GsonBuilder().setPrettyPrinting().create();
        File configFile = new File("config", "minecart-trains.json");
        boolean createNewConfig = false;
        if (configFile.exists()) {
            try (FileReader reader = new FileReader(configFile)) {
                CONFIG = gson.fromJson(reader, MCTConfig.class);
            } catch (IOException e) {
                throw new RuntimeException(e);
            } catch (JsonSyntaxException | JsonIOException e) {
                LOGGER.warn("Error reading config file", e);
                if (!configFile.renameTo(new File("config", "minecart-trains.json.bak"))) {
                    throw new RuntimeException("Couldn't access broken config file, aborting...", e);
                }
                createNewConfig = true;
            }
        } else {
            createNewConfig = true;
        }
        if (createNewConfig) {
            CONFIG = new MCTConfig();
            configFile.getParentFile().mkdirs();
            try (FileWriter writer = new FileWriter(configFile)) {
                gson.toJson(CONFIG, writer);
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }

    // --------------------------
    // ItemStack NBT runtime adapter（运行期反射，兼容不同映射）
    // --------------------------
    private static final class ItemStackNbtRuntime {
        private static Method methodGetOrCreate = null;
        private static Method methodGet = null;
        private static Method methodHas = null;
        private static Method methodSet = null;
        private static Method methodRemove = null;
        private static boolean initialized = false;

        private static void ensureInit() {
            if (initialized) return;
            initialized = true;
            Class<?> cls = class_1799.class;
            Method[] methods = cls.getMethods();

            for (Method m : methods) {
                try {
                    String name = m.getName().toLowerCase();
                    if (m.getParameterCount() == 0 && m.getReturnType() == class_2487.class) {
                        if (name.contains("getorcreate")) methodGetOrCreate = m;
                        else if (methodGet == null && (name.contains("get") && (name.contains("tag") || name.contains("nbt") || name.equals("get")))) methodGet = m;
                    }
                    if (m.getParameterCount() == 0 && (m.getReturnType() == boolean.class || m.getReturnType() == Boolean.class)) {
                        if (name.contains("has") || name.contains("hastag") || name.contains("hasnbt")) methodHas = m;
                    }
                    if (m.getParameterCount() == 1 && m.getParameterTypes()[0] == class_2487.class) {
                        if (name.contains("set") || name.contains("settag") || name.contains("setnbt")) methodSet = m;
                    }
                    if (m.getParameterCount() == 0 && m.getReturnType() == void.class) {
                        if (name.contains("remove") && (name.contains("nbt") || name.contains("tag") || name.equals("remove"))) methodRemove = m;
                    }
                } catch (Throwable ignored) {}
            }

            // Declared fallback
            try {
                if (methodSet == null) {
                    Method d = class_1799.class.getDeclaredMethod("setTag", class_2487.class);
                    d.setAccessible(true);
                    methodSet = d;
                }
                if (methodGetOrCreate == null) {
                    Method d = class_1799.class.getDeclaredMethod("getOrCreateTag");
                    d.setAccessible(true);
                    methodGetOrCreate = d;
                }
                if (methodGet == null) {
                    Method d = class_1799.class.getDeclaredMethod("getTag");
                    d.setAccessible(true);
                    methodGet = d;
                }
                if (methodHas == null) {
                    Method d = class_1799.class.getDeclaredMethod("hasTag");
                    d.setAccessible(true);
                    methodHas = d;
                }
                if (methodRemove == null) {
                    Method d = class_1799.class.getDeclaredMethod("removeTag");
                    d.setAccessible(true);
                    methodRemove = d;
                }
            } catch (Throwable ignored) {}
        }

        public static class_2487 getOrCreateTag(class_1799 stack) {
            ensureInit();
            try {
                if (methodGetOrCreate != null) {
                    Object r = methodGetOrCreate.invoke(stack);
                    if (r instanceof class_2487) return (class_2487) r;
                }
                if (methodGet != null) {
                    Object r = methodGet.invoke(stack);
                    if (r instanceof class_2487) return (class_2487) r;
                }
                class_2487 n = new class_2487();
                if (methodSet != null) {
                    methodSet.invoke(stack, n);
                    return n;
                }
            } catch (Throwable t) {
                t.printStackTrace();
            }
            return null;
        }

        public static class_2487 getTag(class_1799 stack) {
            ensureInit();
            try {
                if (methodGet != null) {
                    Object r = methodGet.invoke(stack);
                    if (r instanceof class_2487) return (class_2487) r;
                }
                if (methodGetOrCreate != null) {
                    Object r = methodGetOrCreate.invoke(stack);
                    if (r instanceof class_2487) return (class_2487) r;
                }
            } catch (Throwable ignored) {}
            return null;
        }

        @SuppressWarnings("unused")
        public static boolean hasTag(class_1799 stack) {
            ensureInit();
            try {
                if (methodHas != null) {
                    Object r = methodHas.invoke(stack);
                    if (r instanceof Boolean) return (Boolean) r;
                }
                return getTag(stack) != null;
            } catch (Throwable ignored) {}
            return false;
        }

        public static void setTag(class_1799 stack, class_2487 tag) {
            ensureInit();
            try {
                if (methodSet != null) {
                    methodSet.invoke(stack, tag);
                    return;
                }
                try {
                    Method fallback = class_1799.class.getDeclaredMethod("setTag", class_2487.class);
                    fallback.setAccessible(true);
                    fallback.invoke(stack, tag);
                } catch (Throwable ignored) {}
            } catch (Throwable t) {
                t.printStackTrace();
            }
        }

        public static void removeTag(class_1799 stack) {
            ensureInit();
            try {
                if (methodRemove != null) {
                    methodRemove.invoke(stack);
                    return;
                }
                if (methodSet != null) {
                    methodSet.invoke(stack, new Object[]{ null });
                    return;
                }
                try {
                    Method fallback = class_1799.class.getDeclaredMethod("setTag", class_2487.class);
                    fallback.setAccessible(true);
                    fallback.invoke(stack, new Object[]{ null });
                } catch (Throwable ignored) {}
            } catch (Throwable t) {
                t.printStackTrace();
            }
        }
    }

    // --------------------------
    // NBT + fallback helpers（使用磁盘持久化回退）
    // --------------------------
    private static void setStackParent(class_1799 stack, UUID uuid, class_1657 player) {
        if (stack != null) {
            class_2487 n = ItemStackNbtRuntime.getOrCreateTag(stack);
            if (n != null) {
                n.method_25927(NBT_PARENT_KEY, uuid);
                ItemStackNbtRuntime.setTag(stack, n);
                // 清理持久化回退与内存缓存（以 ItemStack 为准）
                clearPlayerFallback(player);
                System.out.println("mctrains: setStackParent wrote uuid=" + uuid + " nbt=" + n.toString());
                return;
            }
        }
        // fallback: 写入内存缓存并持久化到磁盘
        if (player != null) {
            savePlayerFallback(player, uuid);
        } else {
            System.out.println("mctrains: setStackParent failed: no stack nbt and no player fallback");
        }
    }

    private static UUID getStackParent(class_1799 stack, class_1657 player) {
        if (stack != null) {
            class_2487 n = ItemStackNbtRuntime.getTag(stack);
            if (n != null && n.method_10545(NBT_PARENT_KEY)) {
                try {
                    return n.method_25926(NBT_PARENT_KEY);
                } catch (Throwable ignored) {}
            }
        }
        // 内存缓存优先
        if (player != null) {
            UUID mem = PLAYER_PARENT_FALLBACK.get(player.method_5667());
            if (mem != null) {
                System.out.println("mctrains: getStackParent returned from memory fallback " + mem + " for player " + player.method_5667());
                return mem;
            }
            // 从磁盘持久化读取（已在内存缓存中）
            UUID loaded = loadPlayerFallback(player);
            if (loaded != null) {
                System.out.println("mctrains: getStackParent returned from persistent fallback " + loaded + " for player " + player.method_5667());
                return loaded;
            }
        }
        return null;
    }

    private static void removeStackParent(class_1799 stack, class_1657 player) {
        boolean removed = false;
        if (stack != null) {
            class_2487 n = ItemStackNbtRuntime.getTag(stack);
            if (n != null && n.method_10545(NBT_PARENT_KEY)) {
                n.method_10551(NBT_PARENT_KEY);
                if (n.method_33133()) {
                    ItemStackNbtRuntime.removeTag(stack);
                } else {
                    ItemStackNbtRuntime.setTag(stack, n);
                }
                removed = true;
                System.out.println("mctrains: removeStackParent removed from item");
            }
        }
        if (player != null) {
            UUID prev = PLAYER_PARENT_FALLBACK.remove(player.method_5667());
            if (prev != null) {
                removed = true;
                clearPlayerFallback(player);
                System.out.println("mctrains: removeStackParent removed fallback for player " + player.method_5667());
            }
        }
        if (!removed) {
            System.out.println("mctrains: removeStackParent nothing removed");
        }
    }

    // --------------------------
    // 磁盘持久化回退方法（JSON）
    // --------------------------
    private static void loadPersistentFallbacks() {
        try {
            if (FALLBACKS_FILE.exists()) {
                try (FileReader r = new FileReader(FALLBACKS_FILE)) {
                    Type t = new TypeToken<Map<String, String>>(){}.getType();
                    Map<String, String> m = FALLBACKS_GSON.fromJson(r, t);
                    if (m != null) {
                        PLAYER_PARENT_FALLBACK.clear();
                        for (Map.Entry<String, String> e : m.entrySet()) {
                            try {
                                UUID playerId = UUID.fromString(e.getKey());
                                UUID parentId = UUID.fromString(e.getValue());
                                PLAYER_PARENT_FALLBACK.put(playerId, parentId);
                            } catch (Throwable ignored) {}
                        }
                    }
                }
            }
        } catch (Throwable ex) {
            LOGGER.warn("mctrains: failed to load fallbacks file", ex);
        }
    }

    private static synchronized void persistFallbacks() {
        try {
            FALLBACKS_FILE.getParentFile().mkdirs();
            try (FileWriter w = new FileWriter(FALLBACKS_FILE)) {
                Map<String,String> out = new java.util.HashMap<>();
                for (Map.Entry<UUID,UUID> e : PLAYER_PARENT_FALLBACK.entrySet()) {
                    out.put(e.getKey().toString(), e.getValue().toString());
                }
                FALLBACKS_GSON.toJson(out, w);
            }
        } catch (Throwable ex) {
            LOGGER.warn("mctrains: failed to persist fallbacks file", ex);
        }
    }

    private static void savePlayerFallback(class_1657 player, UUID parent) {
        PLAYER_PARENT_FALLBACK.put(player.method_5667(), parent);
        persistFallbacks();
        LOGGER.debug("mctrains: saved fallback to memory+file for player {} -> {}", player.method_5667(), parent);
    }

    private static UUID loadPlayerFallback(class_1657 player) {
        return PLAYER_PARENT_FALLBACK.get(player.method_5667());
    }

    private static void clearPlayerFallback(class_1657 player) {
        PLAYER_PARENT_FALLBACK.remove(player.method_5667());
        persistFallbacks();
        LOGGER.debug("mctrains: cleared fallback for player {}", player.method_5667());
    }
}